"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.filter = exports.isEmpty = exports.ConfigurationMapper = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const toml_1 = require("@iarna/toml");
const param_case_1 = require("param-case");
const pascal_case_1 = require("pascal-case");
const snake_case_1 = require("snake-case");
class ConfigurationMapper {
    constructor(props) {
        this.props = props;
    }
    static withDefaults(props) {
        const { globalConfiguration, runnersConfiguration } = props;
        return new ConfigurationMapper({
            globalConfiguration: {
                concurrent: 10,
                checkInterval: 0,
                logFormat: "runner",
                logLevel: "info",
                ...globalConfiguration,
            },
            runnersConfiguration: runnersConfiguration.map((item) => {
                return {
                    url: "https://gitlab.com",
                    limit: 10,
                    outputLimit: 52428800,
                    executor: "docker+machine",
                    environment: ["DOCKER_DRIVER=overlay2", "DOCKER_TLS_CERTDIR=/certs"],
                    ...item,
                    docker: {
                        tlsVerify: false,
                        image: "docker:19.03.5",
                        privileged: true,
                        capAdd: ["CAP_SYS_ADMIN"],
                        waitForServicesTimeout: 300,
                        disableCache: false,
                        volumes: ["/certs/client", "/cache"],
                        shmSize: 0,
                        ...item.docker,
                    },
                    machine: {
                        idleCount: 0,
                        idleTime: 300,
                        maxBuilds: 20,
                        machineDriver: "amazonec2",
                        machineName: "gitlab-runner-%s",
                        ...item.machine,
                        machineOptions: {
                            requestSpotInstance: true,
                            spotPrice: 0.03,
                            metadataToken: "required",
                            metadataTokenResponseHopLimit: 2,
                            ...item.machine?.machineOptions,
                        },
                        autoscaling: item.machine?.autoscaling
                            ? item.machine?.autoscaling
                            : [
                                {
                                    periods: ["* * 7-22 * * mon-fri *"],
                                    idleCount: 1,
                                    idleTime: 1800,
                                    timezone: "Etc/UTC",
                                },
                            ],
                    },
                    cache: item.cache?.s3 && Object.keys(item.cache?.s3).length
                        ? {
                            type: "s3",
                            shared: true,
                            ...item.cache,
                        }
                        : undefined,
                };
            }),
        });
    }
    static fromProps(props) {
        return new ConfigurationMapper(props);
    }
    toToml() {
        return toml_1.stringify(this._toJsonMap());
    }
    /**
     * @internal
     */
    _toJsonMap() {
        const { globalConfiguration, runnersConfiguration } = this.props;
        const result = toJsonMap(globalConfiguration, snake_case_1.snakeCase);
        result.runners = [];
        for (const config of runnersConfiguration) {
            const runner = toJsonMap(config, snake_case_1.snakeCase);
            // Fix naming convention inconsistencies
            runner["tls-ca-file"] = runner.tls_ca_file;
            delete runner.tls_ca_file;
            runner["tls-cert-file"] = runner.tls_cert_file;
            delete runner.tls_ca_file;
            runner["tls-key-file"] = runner.tls_key_file;
            delete runner.tls_ca_file;
            if (config.docker) {
                runner.docker = toJsonMap(config.docker, snake_case_1.snakeCase);
            }
            runner.machine = toJsonMap(config.machine, pascal_case_1.pascalCase);
            if (config.machine?.machineOptions) {
                runner.machine.MachineOptions = this._mapMachineOptions(config.machine?.machineOptions);
            }
            if (config.machine?.autoscaling?.length) {
                runner.machine.autoscaling = config.machine?.autoscaling.map((autoscaling) => toJsonMap(autoscaling, pascal_case_1.pascalCase));
            }
            delete runner.machine.Autoscaling;
            if (config?.cache?.s3 && Object.keys(config.cache.s3).length) {
                runner.cache = toJsonMap(config.cache, pascal_case_1.pascalCase);
                runner.cache.s3 = toJsonMap(config.cache.s3, pascal_case_1.pascalCase);
                delete runner.cache.s3;
            }
            else {
                delete runner.cache;
            }
            result.runners.push(runner);
        }
        return filter(result, (item) => !isEmpty(item));
    }
    _mapMachineOptions(machineOptions) {
        return toProperties(machineOptions, (key) => `amazonec2-${param_case_1.paramCase(key)}`);
    }
}
exports.ConfigurationMapper = ConfigurationMapper;
_a = JSII_RTTI_SYMBOL_1;
ConfigurationMapper[_a] = { fqn: "@pepperize/cdk-autoscaling-gitlab-runner.ConfigurationMapper", version: "0.2.132" };
/**
 * Transforms configuration objects to JsonMap. Pass an inflector function to transform object keys.
 *
 * @param configuration
 * @param inflector A function to transform the object key
 */
function toJsonMap(configuration, inflector) {
    const result = {};
    for (const key in configuration) {
        const value = configuration[key];
        if (value === undefined) {
            continue;
        }
        result[inflector(key)] = value;
    }
    return result;
}
/**
 * Transforms configuration objects to a property array. Pass an inflector function to transform object keys.
 *
 * @param configuration
 * @param inflector A function to transform the object key
 * @example
 * // returns ["foo=bar"]
 * toProperties({foo: "bar", (s) => s});
 */
function toProperties(configuration, inflector) {
    const result = [];
    for (const key in configuration) {
        const value = configuration[key];
        if (value === undefined) {
            continue;
        }
        result.push(`${inflector(key)}=${value}`);
    }
    return result;
}
function isEmpty(subject) {
    if (Array.isArray(subject)) {
        return !subject.length;
    }
    if (typeof subject === "object" && !(subject instanceof Date)) {
        return !Object.keys(subject).length;
    }
    if (subject === undefined) {
        return true;
    }
    if (subject === null) {
        return true;
    }
    return false;
}
exports.isEmpty = isEmpty;
function filter(subject, predicate) {
    if (Array.isArray(subject)) {
        const result = [];
        subject.forEach((element) => {
            const filtered = filter(element, predicate);
            if (predicate.call(subject, filtered)) {
                result.push(filtered);
            }
        });
        return result;
    }
    if (typeof subject === "object" && !(subject instanceof Date)) {
        const result = {};
        for (const key in subject) {
            const value = subject[key];
            const filtered = filter(value, predicate);
            if (predicate.call(subject, filtered)) {
                result[key] = filtered;
            }
        }
        return result;
    }
    return subject;
}
exports.filter = filter;
//# sourceMappingURL=data:application/json;base64,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