# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/02-cleaning.ipynb (unless otherwise specified).

__all__ = ['identify_latest_set_num', 'reindex_df_dt_idx', 'load_training_dataset', 'combine_training_datasets',
           'identify_df_dt_entries', 'construct_df_temp_features', 'split_X_y_data', 'split_X_y_data_with_index',
           'generate_kfold_preds', 'evaluate_models', 'interpolate_missing_panel_temps',
           'construct_df_irradiance_features', 'interpolate_missing_site_irradiance', 'construct_df_power_features',
           'pv_anomalies_to_nan', 'interpolate_missing_site_power', 'interpolate_missing_weather_solar',
           'interpolate_missing_temps']

# Cell
import numpy as np
import pandas as pd

import seaborn as sns
import matplotlib.pyplot as plt

from sklearn.model_selection import KFold
from sklearn.metrics import mean_absolute_error, mean_squared_error
from sklearn.linear_model import LinearRegression
from sklearn.ensemble import RandomForestRegressor, GradientBoostingRegressor

import os
import glob
from ipypb import track

from batopt import utils, retrieval

# Cell
def identify_latest_set_num(data_dir):
    set_num = max([
        int(f.split('_set')[1].replace('.csv', ''))
        for f in os.listdir(data_dir)
        if 'set' in f
    ])

    return set_num

# Cell
def reindex_df_dt_idx(df, freq='30T'):
    full_dt_idx = pd.date_range(df.index.min(), df.index.max(), freq=freq)
    df = df.reindex(full_dt_idx)

    return df

def load_training_dataset(raw_data_dir: str, dataset_name: str='demand', set_num=None, parse_dt_idx: bool=True, dt_idx_freq: str='30T') -> pd.DataFrame:
    if set_num is None:
        set_num = identify_latest_set_num(raw_data_dir)

    allowed_datasets = ['demand', 'pv', 'weather']
    assert dataset_name in allowed_datasets, f"`dataset_name` must be one of: {', '.join(allowed_datasets)} - not {dataset_name}"

    df = pd.read_csv(glob.glob(f'{raw_data_dir}/{dataset_name}*set{set_num}.csv')[0].replace('\\', '/'))

    if parse_dt_idx == True:
        assert 'datetime' in df.columns, 'if `parse_dt_idx` is True then `datetime` must be a column in the dataset'

        df['datetime'] = pd.to_datetime(df['datetime'], utc=True)
        df = df.set_index('datetime').pipe(reindex_df_dt_idx, freq=dt_idx_freq).sort_index(axis=1)
        df.index.name = 'datetime'

    return df

# Cell
def combine_training_datasets(raw_data_dir, set_num=None):
    # Loading provided training datasets
    single_datasets = dict()
    dataset_names = ['demand', 'pv', 'weather']

    for dataset_name in dataset_names:
        single_datasets[dataset_name] = load_training_dataset(raw_data_dir, dataset_name, set_num=set_num)

    # Constructing date range
    min_dt = min([df.index.min() for df in single_datasets.values()])
    max_dt = max([df.index.max() for df in single_datasets.values()]) + pd.Timedelta(minutes=30)

    dt_rng = pd.date_range(min_dt, max_dt, freq='30T')

    # Constructing combined dataframe
    df_combined = pd.DataFrame(index=dt_rng, columns=dataset_names)

    for dataset_name in dataset_names:
        df_single_dataset = single_datasets[dataset_name]
        cols_to_be_overwritten = set(df_combined.columns) - (set(df_combined.columns) - set(df_single_dataset.columns))
        assert len(cols_to_be_overwritten) == 0, f"The following columns exist in multiple datasets meaning data would be overwritten: {', '.join(cols_to_be_overwritten)}"

        df_combined[df_single_dataset.columns] = df_single_dataset

    df_combined = df_combined.sort_index()

    # Adding holiday dates
    s_holidays = retrieval.load_holidays_s(raw_data_dir)

    s_cropped_holidays = s_holidays[max(df_combined.index.min(), s_holidays.index.min()):
                                    min(df_combined.index.max(), s_holidays.index.max())]

    df_combined.loc[s_cropped_holidays.index, 'holidays'] = s_cropped_holidays

    return df_combined

# Cell
def identify_df_dt_entries(df_demand, df_pv, df_weather):
    min_dt = min(df_demand.index.min(), df_pv.index.min(), df_weather.index.min())
    max_dt = max(df_demand.index.max(), df_pv.index.max(), df_weather.index.max())

    dt_rng = pd.date_range(min_dt, max_dt, freq='30T')
    df_nulls = pd.DataFrame(index=dt_rng)

    df_nulls['demand'] = df_demand.reindex(dt_rng).isnull().mean(axis=1).astype(int)
    df_nulls['pv'] = df_pv.reindex(dt_rng).isnull().mean(axis=1).astype(int)
    df_nulls['weather'] = df_weather.reindex(dt_rng).ffill(limit=1).isnull().mean(axis=1).astype(int)

    df_entries = 1 - df_nulls

    return df_entries

# Cell
def construct_df_temp_features(df_weather, df_pv):
    df_weather = df_weather.reindex(pd.date_range(df_weather.index.min(), df_weather.index.max(), freq='30T')).ffill(limit=1)
    temp_loc_cols = df_weather.columns[df_weather.columns.str.contains('temp')]

    df_temp_features = (df_weather
                        .copy()
                        [temp_loc_cols]
                        .assign(site_temp=df_pv['panel_temp_C'])
                       )

    df_temp_features[[col+'_rolling' for col in temp_loc_cols]] = df_temp_features.rolling(3).mean()[temp_loc_cols]

    df_temp_features = df_temp_features.sort_index(axis=1)

    return df_temp_features

# Cell
def split_X_y_data(df, target_col='site_temp'):
    df = df.dropna()
    X_cols = df.drop(target_col, axis=1).columns

    X = df[X_cols].values
    y = df[target_col].values

    return X, y

def split_X_y_data_with_index(df, target_col='site_temp'):
    df = df.dropna()
    X_cols = df.drop(target_col, axis=1).columns

    X = df[X_cols].values
    y = df[target_col].values
    index = df.index

    return X, y, index

# Cell
def generate_kfold_preds(
    X,
    y,
    model=LinearRegression(),
    kfold_kwargs={'n_splits': 5, 'shuffle': True},
    index=None
):

    kfold = KFold(**kfold_kwargs)
    df_pred = pd.DataFrame(columns=['pred', 'true'], index=np.arange(X.shape[0]))

    for train_idxs, test_idxs in kfold.split(X):
        X_train, y_train = X[train_idxs], y[train_idxs]
        X_test, y_test = X[test_idxs], y[test_idxs]

        model.fit(X_train, y_train)

        df_pred.loc[test_idxs, 'true'] = y_test
        df_pred.loc[test_idxs, 'pred'] = model.predict(X_test)

    df_pred = df_pred.sort_index()

    if index is not None:
        assert len(index) == df_pred.shape[0], 'The passed index must be the same length as X and y'
        df_pred.index = index

    return df_pred

# Cell
def evaluate_models(X, y, models, post_pred_proc_func=None, index=None):
    model_scores = dict()

    for model_name, model in track(models.items()):
        df_pred = generate_kfold_preds(X, y, model, index=index)

        if post_pred_proc_func is not None:
            df_pred['pred'] = post_pred_proc_func(df_pred['pred'])

        model_scores[model_name] = {
            'mae': mean_absolute_error(df_pred['true'], df_pred['pred']),
            'rmse': np.sqrt(mean_squared_error(df_pred['true'], df_pred['pred']))
        }

    df_model_scores = pd.DataFrame(model_scores)

    df_model_scores.index.name = 'metric'
    df_model_scores.columns.name = 'model'

    return df_model_scores

# Cell
def interpolate_missing_panel_temps(df_pv, df_weather, model=RandomForestRegressor()):
    missing_panel_temp_dts = df_pv.index[df_pv['panel_temp_C'].isnull()]

    if len(missing_panel_temp_dts) == 0: # i.e. no missing values
        return df_pv

    df_temp_features = construct_df_temp_features(df_weather, df_pv)
    missing_dt_X = df_temp_features.loc[missing_panel_temp_dts].drop('site_temp', axis=1).values
    X, y = split_X_y_data(df_temp_features, 'site_temp')

    model.fit(X, y)
    df_pv.loc[missing_panel_temp_dts, 'panel_temp_C'] = model.predict(missing_dt_X)

    assert df_pv['panel_temp_C'].isnull().sum() == 0, 'There are still null values for the PV panel temperature'

    return df_pv

# Cell
def construct_df_irradiance_features(df_weather, df_pv):
    df_weather = df_weather.reindex(pd.date_range(df_weather.index.min(), df_weather.index.max(), freq='30T')).ffill(limit=1)
    temp_loc_cols = df_weather.columns[df_weather.columns.str.contains('solar')]

    df_irradiance_features = (df_weather
                              .copy()
                              [temp_loc_cols]
                              .assign(site_solar=df_pv['irradiance_Wm-2'])
                              .pipe(lambda df: df.assign(hour=df.index.hour + df.index.minute/60))
                             )

    df_irradiance_features = df_irradiance_features.sort_index(axis=1)

    return df_irradiance_features

# Cell
def interpolate_missing_site_irradiance(df_pv, df_weather, model=RandomForestRegressor()):
    missing_site_irradiance_dts = df_pv.index[df_pv['irradiance_Wm-2'].isnull()]

    if len(missing_site_irradiance_dts) == 0: # i.e. no missing values
        return df_pv

    df_irradiance_features = construct_df_irradiance_features(df_weather, df_pv)
    missing_dt_X = df_irradiance_features.loc[missing_site_irradiance_dts].drop('site_solar', axis=1).values
    X, y = split_X_y_data(df_irradiance_features, 'site_solar')

    model.fit(X, y)
    df_pv.loc[missing_site_irradiance_dts, 'irradiance_Wm-2'] = model.predict(missing_dt_X)

    assert df_pv['irradiance_Wm-2'].isnull().sum() == 0, 'There are still null values for the solar site irradiance'

    return df_pv

# Cell
def construct_df_power_features(df_pv):
    df_power_features = (df_pv
                         .pipe(lambda df: df.assign(hour=df.index.hour + df.index.minute/60))
                         .sort_index(axis=1)
                        )

    return df_power_features

# Cell
def pv_anomalies_to_nan(df_pv, model=GradientBoostingRegressor(), tolerance=0.1):
    """
    Run this function to identify places where predicted values for pv_power_mw are much larger
    than true values and where the true value is 0 (we expect these are anomalies) and make these values nan.

    """
    df_power_features = construct_df_power_features(df_pv)

    X, y, dates = split_X_y_data_with_index(df_power_features, 'pv_power_mw')
    df_pred = generate_kfold_preds(X, y, model)
    df_pred['difference'] = df_pred.pred - df_pred.true
    df_pred['datetime'] = dates
    df_pred = df_pred.set_index('datetime')

    anomalous_idx = df_pred[(df_pred.difference > tolerance) & (df_pred.true == 0)].index

    df_pv.loc[df_pv.index.isin(anomalous_idx), 'pv_power_mw'] = np.nan

    return df_pv

# Cell
def interpolate_missing_site_power(df_pv, model=RandomForestRegressor()):
    missing_site_power_dts = df_pv.index[df_pv['pv_power_mw'].isnull()]

    if len(missing_site_power_dts) == 0: # i.e. no missing values
        return df_pv

    df_power_features = construct_df_power_features(df_pv)
    missing_dt_X = df_power_features.loc[missing_site_power_dts].drop('pv_power_mw', axis=1).values
    X, y = split_X_y_data(df_power_features, 'pv_power_mw')

    model.fit(X, y)
    df_pv.loc[missing_site_power_dts, 'pv_power_mw'] = model.predict(missing_dt_X)

    assert df_pv['pv_power_mw'].isnull().sum() == 0, 'There are still null values for the solar site power'

    return df_pv

# Cell
def interpolate_missing_weather_solar(df_pv, df_weather, weather_col='solar_location2', model=RandomForestRegressor()):
    missing_weather_solar_dts = df_weather.index[df_weather[weather_col].isnull()]

    if len(missing_weather_solar_dts) == 0: # i.e. no missing values
        return df_pv

    df_irradiance_features = construct_df_irradiance_features(df_weather, df_pv).drop('site_solar', axis=1)
    missing_dt_X = df_irradiance_features.loc[missing_weather_solar_dts].drop(weather_col, axis=1).values
    X, y = split_X_y_data(df_irradiance_features, weather_col)

    model.fit(X, y)
    df_weather.loc[missing_weather_solar_dts, weather_col] = model.predict(missing_dt_X)

    assert df_weather[weather_col].isnull().sum() == 0, 'There are still null values for the weather dataset solar observations'

    return df_weather

# Cell
def interpolate_missing_temps(df_weather, temp_variable, model=RandomForestRegressor()):
    """
    Use the other temperature locations to predict missing values of `temp_variable`.

    For test_2: a full day is missing for temp_location4 on 2018-11-18
    """
    missing_temp_dts = df_weather.index[df_weather[temp_variable].isnull()]

    if len(missing_temp_dts) == 0: # i.e. no missing values
        return df_weather

    temp_loc_cols = df_weather.columns[df_weather.columns.str.contains('temp')]

    df_temp_features = df_weather.filter(temp_loc_cols)
    missing_dt_X = df_temp_features.loc[missing_temp_dts].drop(temp_variable, axis=1).values

    X, y = split_X_y_data(df_temp_features, temp_variable)

    model.fit(X, y)

    df_weather.loc[missing_temp_dts, temp_variable] = model.predict(missing_dt_X)
    df_weather = df_weather.ffill(limit=1)

    assert df_weather[temp_variable].isnull().sum() == 0, 'There are still null values for the PV panel temperature'

    return df_weather