"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NagSuppressions = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
/**
 * Helper class with methods to add cdk-nag suppressions to cdk resources.
 *
 * @stability stable
 */
class NagSuppressions {
    /**
     * Apply cdk-nag suppressions to a Stack and optionally nested stacks.
     *
     * @param stack The Stack to apply the suppression to.
     * @param suppressions A list of suppressions to apply to the stack.
     * @param applyToNestedStacks Apply the suppressions to children stacks (default:false).
     * @stability stable
     */
    static addStackSuppressions(stack, suppressions, applyToNestedStacks = false) {
        const stacks = applyToNestedStacks
            ? stack.node.findAll().filter((x) => x instanceof monocdk_1.Stack)
            : [stack];
        stacks.forEach((s) => {
            var _b;
            const newSuppressions = [];
            for (const suppression of suppressions) {
                if (suppression.reason.length >= 10) {
                    newSuppressions.push(suppression);
                }
                else {
                    throw Error(`${s.node.id}: The cdk_nag suppression for ${suppression.id} must have a reason of 10 characters or more. See https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.`);
                }
            }
            const currentSuppressions = (_b = s.templateOptions.metadata) === null || _b === void 0 ? void 0 : _b.cdk_nag;
            if (Array.isArray(currentSuppressions === null || currentSuppressions === void 0 ? void 0 : currentSuppressions.rules_to_suppress)) {
                newSuppressions.unshift(...currentSuppressions.rules_to_suppress);
            }
            if (s.templateOptions.metadata) {
                s.templateOptions.metadata.cdk_nag = {
                    rules_to_suppress: newSuppressions,
                };
            }
            else {
                s.templateOptions.metadata = {
                    cdk_nag: { rules_to_suppress: newSuppressions },
                };
            }
        });
    }
    /**
     * Add cdk-nag suppressions to a CfnResource and optionally its children.
     *
     * @param construct The IConstruct to apply the suppression to.
     * @param suppressions A list of suppressions to apply to the resource.
     * @param applyToChildren Apply the suppressions to children CfnResources  (default:false).
     * @stability stable
     */
    static addResourceSuppressions(construct, suppressions, applyToChildren = false) {
        const newSuppressions = [];
        for (const suppression of suppressions) {
            if (suppression.reason.length >= 10) {
                newSuppressions.push(suppression);
            }
            else {
                throw Error(`${construct.node.id}: The cdk_nag suppression for ${suppression.id} must have a reason of 10 characters or more. See https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.`);
            }
        }
        const constructs = applyToChildren ? construct.node.findAll() : [construct];
        for (const child of constructs) {
            const possibleL1 = child.node.defaultChild
                ? child.node.defaultChild
                : child;
            if (possibleL1 instanceof monocdk_1.CfnResource) {
                const resource = possibleL1;
                const currentSuppressions = resource.getMetadata('cdk_nag');
                if (Array.isArray(currentSuppressions === null || currentSuppressions === void 0 ? void 0 : currentSuppressions.rules_to_suppress)) {
                    newSuppressions.unshift(...currentSuppressions.rules_to_suppress);
                }
                resource.addMetadata('cdk_nag', {
                    rules_to_suppress: newSuppressions,
                });
            }
        }
    }
    /**
     * Add cdk-nag suppressions to a CfnResource and optionally its children via its path.
     *
     * @param stack The Stack the construct belongs to.
     * @param path The path to the construct in the provided stack.
     * @param suppressions A list of suppressions to apply to the resource.
     * @param applyToChildren Apply the suppressions to children CfnResources  (default:false).
     * @stability stable
     */
    static addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren = false) {
        for (const child of stack.node.findAll()) {
            const fixedPath = path.replace(/^\//, '');
            if (child.node.path === fixedPath ||
                child.node.path + '/Resource' === fixedPath) {
                NagSuppressions.addResourceSuppressions(child, suppressions, applyToChildren);
            }
        }
    }
}
exports.NagSuppressions = NagSuppressions;
_a = JSII_RTTI_SYMBOL_1;
NagSuppressions[_a] = { fqn: "monocdk-nag.NagSuppressions", version: "0.1.35" };
//# sourceMappingURL=data:application/json;base64,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