"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveResourceFromInstrinsic = exports.resolveIfPrimitive = exports.NagPack = exports.NagMessageLevel = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const VALIDATION_FAILURE_ID = 'CdkNagValidationFailure';
const SUPPRESSION_ID = 'CdkNagSuppression';
/**
 * The level of the message that the rule applies.
 *
 * @stability stable
 */
var NagMessageLevel;
(function (NagMessageLevel) {
    NagMessageLevel[NagMessageLevel["WARN"] = 0] = "WARN";
    NagMessageLevel[NagMessageLevel["ERROR"] = 1] = "ERROR";
})(NagMessageLevel = exports.NagMessageLevel || (exports.NagMessageLevel = {}));
/**
 * Base class for all rule packs.
 *
 * @stability stable
 */
class NagPack {
    /**
     * @stability stable
     */
    constructor(props) {
        this.verbose =
            props == undefined || props.verbose == undefined ? false : props.verbose;
        this.logIgnores =
            props == undefined || props.logIgnores == undefined
                ? false
                : props.logIgnores;
    }
    /**
     * Create a rule to be used in the NagPack.
     *
     * @param params The.
     * @stability stable
     * @IApplyRule interface with rule details.
     */
    applyRule(params) {
        var _b, _c, _d;
        let resourceIgnores = (_b = params.node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
        resourceIgnores = resourceIgnores ? resourceIgnores : [];
        let stackIgnores = (_d = (_c = monocdk_1.Stack.of(params.node).templateOptions.metadata) === null || _c === void 0 ? void 0 : _c.cdk_nag) === null || _d === void 0 ? void 0 : _d.rules_to_suppress;
        stackIgnores = stackIgnores ? stackIgnores : [];
        const allIgnores = resourceIgnores.concat(stackIgnores);
        try {
            if (!params.rule(params.node)) {
                const reason = this.ignoreRule(allIgnores, params.ruleId);
                if (reason) {
                    if (this.logIgnores === true) {
                        const message = this.createMessage(SUPPRESSION_ID, `${params.ruleId} was triggered but suppressed.`, `Provided reason: "${reason}"`);
                        monocdk_1.Annotations.of(params.node).addInfo(message);
                    }
                }
                else {
                    const message = this.createMessage(params.ruleId, params.info, params.explanation);
                    if (params.level == NagMessageLevel.ERROR) {
                        monocdk_1.Annotations.of(params.node).addError(message);
                    }
                    else if (params.level == NagMessageLevel.WARN) {
                        monocdk_1.Annotations.of(params.node).addWarning(message);
                    }
                }
            }
        }
        catch (error) {
            const reason = this.ignoreRule(allIgnores, VALIDATION_FAILURE_ID);
            if (reason) {
                if (this.logIgnores === true) {
                    const message = this.createMessage(SUPPRESSION_ID, `${VALIDATION_FAILURE_ID} was triggered but suppressed.`, reason);
                    monocdk_1.Annotations.of(params.node).addInfo(message);
                }
            }
            else {
                const information = `'${params.ruleId}' threw an error during validation. This is generally caused by a parameter referencing an intrinsic function. For more details enable verbose logging.'`;
                const message = this.createMessage(VALIDATION_FAILURE_ID, information, error.message);
                monocdk_1.Annotations.of(params.node).addWarning(message);
            }
        }
    }
    /**
     * Check whether a specific rule should be ignored.
     * @param ignores The ignores listed in cdk-nag metadata.
     * @param ruleId The id of the rule to ignore.
     * @returns The reason the rule was ignored, or an empty string.
     */
    ignoreRule(ignores, ruleId) {
        for (let ignore of ignores) {
            if (ignore.id &&
                ignore.reason &&
                JSON.stringify(ignore.reason).length >= 10) {
                if (ignore.id == ruleId) {
                    return ignore.reason;
                }
            }
            else {
                throw Error(`Improperly formatted cdk_nag rule suppression detected: ${JSON.stringify(ignore)}. See https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.`);
            }
        }
        return '';
    }
    /**
     * The message to output to the console when a rule is triggered.
     * @param ruleId The id of the rule.
     * @param info Why the rule was triggered.
     * @param explanation Why the rule exists.
     * @returns The formatted message string.
     */
    createMessage(ruleId, info, explanation) {
        let message = `${ruleId}: ${info}`;
        return this.verbose ? `${message} ${explanation}\n` : `${message}\n`;
    }
}
exports.NagPack = NagPack;
_a = JSII_RTTI_SYMBOL_1;
NagPack[_a] = { fqn: "monocdk-nag.NagPack", version: "0.1.35" };
/**
 * Use in cases where a primitive value must be known to pass a rule.
 * https://developer.mozilla.org/en-US/docs/Glossary/Primitive
 * @param node The CfnResource to check.
 * @param parameter The value to attempt to resolve.
 * @returns Return a value if resolves to a primitive data type, otherwise throw an error.
 */
function resolveIfPrimitive(node, parameter) {
    const resolvedValue = monocdk_1.Stack.of(node).resolve(parameter);
    if (resolvedValue === Object(resolvedValue)) {
        throw Error(`The parameter resolved to to a non-primitive value "${JSON.stringify(resolvedValue)}", therefore the rule could not be validated.`);
    }
    else {
        return resolvedValue;
    }
}
exports.resolveIfPrimitive = resolveIfPrimitive;
/**
 * Use in cases where a token resolves to an intrinsic function and the referenced resource must be known to pass a rule.
 * @param node The CfnResource to check.
 * @param parameter The value to attempt to resolve.
 * @returns Return the Logical resource Id if resolves to a intrinsic function, otherwise the resolved provided value.
 */
function resolveResourceFromInstrinsic(node, parameter) {
    const resolvedValue = monocdk_1.Stack.of(node).resolve(parameter);
    const ref = resolvedValue === null || resolvedValue === void 0 ? void 0 : resolvedValue.Ref;
    const getAtt = resolvedValue === null || resolvedValue === void 0 ? void 0 : resolvedValue['Fn::GetAtt'];
    if (ref != undefined) {
        return ref;
    }
    else if (Array.isArray(getAtt) && getAtt.length > 0) {
        return getAtt[0];
    }
    return resolvedValue;
}
exports.resolveResourceFromInstrinsic = resolveResourceFromInstrinsic;
//# sourceMappingURL=data:application/json;base64,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