"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_iam_1 = require("monocdk/aws-iam");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * IAM Groups have at least one IAM User - (Control IDs: 7.1.2, 7.1.3, 7.2.1, 7.2.2)
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_iam_1.CfnGroup) {
        const groupLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        const groupName = monocdk_1.Stack.of(node).resolve(node.groupName);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_iam_1.CfnUser) {
                if (isMatchingUser(child, groupLogicalId, groupName)) {
                    found = true;
                    break;
                }
            }
            else if (child instanceof aws_iam_1.CfnUserToGroupAddition) {
                if (isMatchingGroupAddition(child, groupLogicalId, groupName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether the IAM User belongs to the IAM Group
 * @param node the CfnUser to check
 * @param groupLogicalId the Cfn Logical ID of the group
 * @param groupName the name of the group
 * returns whether the CfnUser is in the given group
 */
function isMatchingUser(node, groupLogicalId, groupName) {
    const groups = monocdk_1.Stack.of(node).resolve(node.groups);
    if (Array.isArray(groups)) {
        for (const group of groups) {
            const resolvedGroup = JSON.stringify(monocdk_1.Stack.of(node).resolve(group));
            if (new RegExp(`${groupLogicalId}(?![\\w])`).test(resolvedGroup) ||
                (groupName != undefined &&
                    new RegExp(`${groupName}(?![\\w\\-_\\.])`).test(resolvedGroup))) {
                return true;
            }
        }
    }
    return false;
}
/**
 * Helper function to check whether the User to Group Addition mentions the specified Group
 * @param node the CfnUserToGroupAddition to check
 * @param groupLogicalId the Cfn Logical ID of the group
 * @param groupName the name of the group
 * returns whether the CfnUserToGroupAddition references the given group
 */
function isMatchingGroupAddition(node, groupLogicalId, groupName) {
    const resolvedGroup = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.groupName));
    if (new RegExp(`${groupLogicalId}(?![\\w])`).test(resolvedGroup) ||
        (groupName != undefined &&
            new RegExp(`${groupName}(?![\\w\\-_\\.])`).test(resolvedGroup))) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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