"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PCIDSS321Checks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../nag-pack");
const apigw_1 = require("./rules/apigw");
const autoscaling_1 = require("./rules/autoscaling");
const cloudtrail_1 = require("./rules/cloudtrail");
const cloudwatch_1 = require("./rules/cloudwatch");
const codebuild_1 = require("./rules/codebuild");
const dms_1 = require("./rules/dms");
const ec2_1 = require("./rules/ec2");
const ecs_1 = require("./rules/ecs");
const efs_1 = require("./rules/efs");
const elb_1 = require("./rules/elb");
const emr_1 = require("./rules/emr");
const iam_1 = require("./rules/iam");
const kms_1 = require("./rules/kms");
const lambda_1 = require("./rules/lambda");
const opensearch_1 = require("./rules/opensearch");
const rds_1 = require("./rules/rds");
const redshift_1 = require("./rules/redshift");
const s3_1 = require("./rules/s3");
const sagemaker_1 = require("./rules/sagemaker");
const secretsmanager_1 = require("./rules/secretsmanager");
const sns_1 = require("./rules/sns");
const vpc_1 = require("./rules/vpc");
const waf_1 = require("./rules/waf");
/**
 * Check for PCI DSS 3.2.1 compliance. Based on the PCI DSS 3.2.1 AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-pci-dss.html.
 *
 * @stability stable
 */
class PCIDSS321Checks extends nag_pack_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        if (node instanceof monocdk_1.CfnResource) {
            this.checkAPIGW(node);
            this.checkAutoScaling(node);
            this.checkCloudTrail(node);
            this.checkCloudWatch(node);
            this.checkCodeBuild(node);
            this.checkDMS(node);
            this.checkEC2(node);
            this.checkECS(node);
            this.checkEFS(node);
            this.checkELB(node);
            this.checkEMR(node);
            this.checkIAM(node);
            this.checkKMS(node);
            this.checkLambda(node);
            this.checkOpenSearch(node);
            this.checkRDS(node);
            this.checkRedshift(node);
            this.checkS3(node);
            this.checkSageMaker(node);
            this.checkSecretsManager(node);
            this.checkSNS(node);
            this.checkVPC(node);
            this.checkWAF(node);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-APIGWAssociatedWithWAF',
            info: 'The REST API stage is not associated with AWS WAFv2 web ACL - (Control ID: 6.6).',
            explanation: 'AWS WAF enables you to configure a set of rules (called a web access control list (web ACL)) that allow, block, or count web requests based on customizable web security rules and conditions that you define. Ensure your Amazon API Gateway stage is associated with a WAF Web ACL to protect it from malicious attacks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.pciDss321APIGWAssociatedWithWAF,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-APIGWCacheEnabledAndEncrypted',
            info: 'The API Gateway stage does not have caching enabled and encrypted for all methods - (Control ID: 3.4).',
            explanation: "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.pciDss321APIGWCacheEnabledAndEncrypted,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-APIGWExecutionLoggingEnabled',
            info: 'The API Gateway stage does not have execution logging enabled for all methods - (Control IDs: 10.1, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6, 10.5.4).',
            explanation: 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.pciDss321APIGWExecutionLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-APIGWSSLEnabled',
            info: 'The API Gateway REST API stage is not configured with SSL certificates - (Control IDs: 2.3, 4.1, 8.2.1).',
            explanation: 'Ensure Amazon API Gateway REST API stages are configured with SSL certificates to allow backend systems to authenticate that requests originate from API Gateway.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.pciDss321APIGWSSLEnabled,
            node: node,
        });
    }
    /**
     * Check Auto Scaling Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAutoScaling(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-AutoscalingGroupELBHealthCheckRequired',
            info: 'The Auto Scaling group (which is associated with a load balancer) does not utilize ELB healthchecks - (Control ID: 2.2).',
            explanation: 'The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability. The load balancer periodically sends pings, attempts connections, or sends requests to test Amazon EC2 instances health in an auto-scaling group. If an instance is not reporting back, traffic is sent to a new Amazon EC2 instance.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.pciDss321AutoscalingGroupELBHealthCheckRequired,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-AutoscalingLaunchConfigPublicIpDisabled',
            info: 'The Auto Scaling launch configuration does not have public IP addresses disabled - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.pciDss321AutoscalingLaunchConfigPublicIpDisabled,
            node: node,
        });
    }
    /**
     * Check CloudTrail Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-CloudTrailCloudWatchLogsEnabled',
            info: 'The trail does not have CloudWatch logs enabled - (Control IDs: 2.2, 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.5, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6, 10.5.3, 10.5.4).',
            explanation: 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.pciDss321CloudTrailCloudWatchLogsEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-CloudTrailEncryptionEnabled',
            info: 'The trail does not have encryption enabled - (Control IDs: 2.2, 3.4, 10.5).',
            explanation: 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.pciDss321CloudTrailEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-CloudTrailLogFileValidationEnabled',
            info: 'The trail does not have log file validation enabled - (Control IDs: 2.2, 10.5.2, 10.5, 10.5.5, 11.5).',
            explanation: 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.pciDss321CloudTrailLogFileValidationEnabled,
            node: node,
        });
    }
    /**
     * Check CloudWatch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudWatch(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-CloudWatchLogGroupEncrypted',
            info: 'The CloudWatch Log Group is not encrypted with an AWS KMS key - (Control ID: 3.4).',
            explanation: 'To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.pciDss321CloudWatchLogGroupEncrypted,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-CloudWatchLogGroupRetentionPeriod',
            info: 'The CloudWatch Log Group does not have an explicit retention period configured - (Control IDs: 3.1, 10.7).',
            explanation: 'Ensure a minimum duration of event log data is retained for your log groups to help with troubleshooting and forensics investigations. The lack of available past event log data makes it difficult to reconstruct and identify potentially malicious events.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.pciDss321CloudWatchLogGroupRetentionPeriod,
            node: node,
        });
    }
    /**
     * Check CodeBuild Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCodeBuild(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-CodeBuildProjectEnvVarAwsCred',
            info: 'The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables - (Control ID: 8.2.1).',
            explanation: 'Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: codebuild_1.pciDss321CodeBuildProjectEnvVarAwsCred,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-CodeBuildProjectSourceRepoUrl',
            info: 'The CodeBuild project which utilizes either a GitHub or BitBucket source repository does not utilize OAUTH - (Control ID: 8.2.1).',
            explanation: 'OAUTH is the most secure method of authenticating your CodeBuild application. Use OAuth instead of personal access tokens or a user name and password to grant authorization for accessing GitHub or Bitbucket repositories.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: codebuild_1.pciDss321CodeBuildProjectSourceRepoUrl,
            node: node,
        });
    }
    /**
     * Check DMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDMS(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-DMSReplicationNotPublic',
            info: 'The DMS replication instance is public - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'DMS replication instances can contain sensitive information and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dms_1.pciDss321DMSReplicationNotPublic,
            node: node,
        });
    }
    /**
     * Check EC2 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEC2(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-EC2InstanceNoPublicIp',
            info: 'The EC2 instance is associated with a public IP address - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Manage access to the AWS Cloud by ensuring Amazon Elastic Compute Cloud (Amazon EC2) instances cannot be publicly accessed. Amazon EC2 instances can contain sensitive information and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.pciDss321EC2InstanceNoPublicIp,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-EC2InstanceProfileAttached',
            info: 'The EC2 instance does not have an instance profile attached - (Control IDs: 2.2, 7.1.1, 7.2.1).',
            explanation: 'EC2 instance profiles pass an IAM role to an EC2 instance. Attaching an instance profile to your instances can assist with least privilege and permissions management.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.pciDss321EC2InstanceProfileAttached,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-EC2InstancesInVPC',
            info: 'The EC2 instance is not within a VPC - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Deploy Amazon Elastic Compute Cloud (Amazon EC2) instances within an Amazon Virtual Private Cloud (Amazon VPC) to enable secure communication between an instance and other services within the amazon VPC, without requiring an internet gateway, NAT device, or VPN connection. All traffic remains securely within the AWS Cloud. Because of their logical isolation, domains that reside within anAmazon VPC have an extra layer of security when compared to domains that use public endpoints. Assign Amazon EC2 instances to an Amazon VPC to properly manage access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.pciDss321EC2InstancesInVPC,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-EC2RestrictedCommonPorts',
            info: 'The EC2 instance allows unrestricted inbound IPv4 TCP traffic on one or more common ports (by default these ports include 20, 21, 3389, 3309, 3306, 4333) - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 2.2, 2.2.2).',
            explanation: 'Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.pciDss321EC2RestrictedCommonPorts,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-EC2RestrictedSSH',
            info: 'The Security Group allows unrestricted SSH access - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 2.2, 2.2.2).',
            explanation: 'Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.pciDss321EC2RestrictedSSH,
            node: node,
        });
    }
    /**
     * Check ECS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkECS(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-ECSTaskDefinitionUserForHostMode',
            info: "The ECS task definition is configured for host networking and has at least one container with definitions with 'privileged' set to false or empty or 'user' set to root or empty - (Control ID: 7.2.1).",
            explanation: 'If a task definition has elevated privileges it is because you have specifically opted-in to those configurations. This rule checks for unexpected privilege escalation when a task definition has host networking enabled but the customer has not opted-in to elevated privileges.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecs_1.pciDss321ECSTaskDefinitionUserForHostMode,
            node: node,
        });
    }
    /**
     * Check EFS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEFS(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-EFSEncrypted',
            info: 'The EFS does not have encryption at rest enabled - (Control IDs: 3.4, 8.2.1).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: efs_1.pciDss321EFSEncrypted,
            node: node,
        });
    }
    /**
     * Check Elastic Load Balancer Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkELB(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-ALBHttpDropInvalidHeaderEnabled',
            info: 'The ALB does not have invalid HTTP header dropping enabled - (Control IDs: 4.1, 8.2.1).',
            explanation: 'Ensure that your Application Load Balancers (ALB) are configured to drop http headers. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.pciDss321ALBHttpDropInvalidHeaderEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-ALBHttpToHttpsRedirection',
            info: "The ALB's HTTP listeners are not configured to redirect to HTTPS - (Control IDs: 2.3, 4.1, 8.2.1).",
            explanation: 'To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.pciDss321ALBHttpToHttpsRedirection,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-ALBWAFEnabled',
            info: 'The ALB is not associated with AWS WAFv2 web ACL - (Control ID: 6.6).',
            explanation: 'A WAF helps to protect your web applications or APIs against common web exploits. These web exploits may affect availability, compromise security, or consume excessive resources within your environment.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.pciDss321ALBWAFEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-ELBACMCertificateRequired',
            info: 'The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: 4.1, 8.2.1).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.pciDss321ELBACMCertificateRequired,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-ELBLoggingEnabled',
            info: 'The ELB does not have logging enabled - (Control IDs: 10.1, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6, 10.5.4).',
            explanation: "Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client's IP address, latencies, request paths, and server responses.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.pciDss321ELBLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-ELBTlsHttpsListenersOnly',
            info: 'The CLB does not restrict its listeners to only the SSL and HTTPS protocols - (Control IDs: 2.3, 4.1, 8.2.1).',
            explanation: 'Ensure that your Classic Load Balancers (CLBs) are configured with SSL or HTTPS listeners. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.pciDss321ELBTlsHttpsListenersOnly,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-ELBv2ACMCertificateRequired',
            info: 'The ALB, NLB, or GLB listener does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control ID: 4.1).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.pciDss321ELBv2ACMCertificateRequired,
            node: node,
        });
    }
    /**
     * Check EMR Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEMR(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-EMRKerberosEnabled',
            info: 'The ALB, NLB, or GLB listener does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control ID: 7.2.1).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.pciDss321EMRKerberosEnabled,
            node: node,
        });
    }
    /**
     * Check IAM Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkIAM(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-IAMGroupHasUsers',
            info: 'The IAM Group does not have at least one IAM User - (Control IDs: 7.1.2, 7.1.3, 7.2.1, 7.2.2).',
            explanation: 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.pciDss321IAMGroupHasUsers,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-IAMNoInlinePolicy',
            info: 'The IAM Group, User, or Role contains an inline policy - (Control IDs: 2.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2).',
            explanation: 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.pciDss321IAMNoInlinePolicy,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-IAMPolicyNoStatementsWithAdminAccess',
            info: 'The IAM policy grants admin access - (Control IDs: 2.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2).',
            explanation: 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.pciDss321IAMPolicyNoStatementsWithAdminAccess,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-IAMPolicyNoStatementsWithFullAccess',
            info: 'The IAM policy grants full access - (Control IDs: 7.1.2, 7.1.3, 7.2.1, 7.2.2).',
            explanation: 'Ensure IAM Actions are restricted to only those actions that are needed. Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.pciDss321IAMPolicyNoStatementsWithFullAccess,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-IAMUserGroupMembership',
            info: 'The IAM user does not belong to any group(s) - (Control IDs: 2.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2).',
            explanation: 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.pciDss321IAMUserGroupMembership,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-IAMUserNoPolicies',
            info: 'The IAM policy is attached at the user level - (Control IDs: 2.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2).',
            explanation: 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.pciDss321IAMUserNoPolicies,
            node: node,
        });
    }
    /**
     * Check KMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkKMS(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-KMSBackingKeyRotationEnabled',
            info: 'The KMS Symmetric key does not have automatic key rotation enabled - (Control IDs: 2.2, 3.5, 3.6, 3.6.4).',
            explanation: 'Enable key rotation to ensure that keys are rotated once they have reached the end of their crypto period.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: kms_1.pciDss321KMSBackingKeyRotationEnabled,
            node: node,
        });
    }
    /**
     * Check Lambda Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkLambda(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-LambdaInsideVPC',
            info: 'The Lambda function is not VPC enabled - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 2.2.2).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: lambda_1.pciDss321LambdaInsideVPC,
            node: node,
        });
    }
    /**
     * Check OpenSearch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkOpenSearch(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-OpenSearchEncryptedAtRest',
            info: 'The OpenSearch Service domain does not have encryption at rest enabled - (Control IDs: 3.4, 8.2.1).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.pciDss321OpenSearchEncryptedAtRest,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-OpenSearchInVPCOnly',
            info: 'The OpenSearch Service domain is not running within a VPC - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'VPCs help secure your AWS resources and provide an extra layer of protection.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.pciDss321OpenSearchInVPCOnly,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-OpenSearchLogsToCloudWatch',
            info: 'The OpenSearch Service domain does not stream error logs (ES_APPLICATION_LOGS) to CloudWatch Logs - (Control IDs: 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6).',
            explanation: 'Ensure Amazon OpenSearch Service domains have error logs enabled and streamed to Amazon CloudWatch Logs for retention and response. Domain error logs can assist with security and access audits, and can help to diagnose availability issues.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.pciDss321OpenSearchLogsToCloudWatch,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-OpenSearchNodeToNodeEncryption',
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled - (Control ID: 4.1).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.pciDss321OpenSearchNodeToNodeEncryption,
            node: node,
        });
    }
    /**
     * Check RDS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRDS(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-RDSAutomaticMinorVersionUpgradeEnabled',
            info: "The route table may contain one or more unrestricted route(s) to an IGW ('0.0.0.0/0' or '::/0') - (Control ID: 6.2).",
            explanation: 'Ensure Amazon EC2 route tables do not have unrestricted routes to an internet gateway. Removing or limiting the access to the internet for workloads within Amazon VPCs can reduce unintended access within your environment.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.pciDss321RDSAutomaticMinorVersionUpgradeEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-RDSInstancePublicAccess',
            info: 'The RDS DB Instance allows public access - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.pciDss321RDSInstancePublicAccess,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-RDSLoggingEnabled',
            info: 'The RDS DB Instance does not have all CloudWatch log types exported - (Control IDs: 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6).',
            explanation: 'To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.pciDss321RDSLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-RDSStorageEncrypted',
            info: 'The RDS DB Instance or Aurora Cluster does not have storage encrypted - (Control IDs: 3.4, 8.2.1).',
            explanation: 'Because sensitive data can exist at rest in Amazon RDS instances, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.pciDss321RDSStorageEncrypted,
            node: node,
        });
    }
    /**
     * Check Redshift Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRedshift(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-RedshiftClusterConfiguration',
            info: 'The Redshift cluster does not have encryption or audit logging enabled - (Control IDs: 3.4, 8.2.1, 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6).',
            explanation: 'To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.pciDss321RedshiftClusterConfiguration,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-RedshiftClusterMaintenanceSettings',
            info: 'The Redshift cluster does not have version upgrades enabled, automated snapshot retention periods enabled, and an explicit maintenance window configured - (Control ID: 6.2).',
            explanation: 'Ensure that Amazon Redshift clusters have the preferred settings for your organization. Specifically, that they have preferred maintenance windows and automated snapshot retention periods for the database.                                                                                                                                                                                                                                                                                                                                                              ',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.pciDss321RedshiftClusterMaintenanceSettings,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-RedshiftClusterPublicAccess',
            info: 'The Redshift cluster allows public access - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.pciDss321RedshiftClusterPublicAccess,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-RedshiftEnhancedVPCRoutingEnabled',
            info: 'The Redshift cluster does not have enhanced VPC routing enabled - (Control IDs: 1.2, 1.3, 1.3.1, 1.3.2).',
            explanation: 'Enhanced VPC routing forces all COPY and UNLOAD traffic between the cluster and data repositories to go through your Amazon VPC. You can then use VPC features such as security groups and network access control lists to secure network traffic. You can also use VPC flow logs to monitor network traffic.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.pciDss321RedshiftEnhancedVPCRoutingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-RedshiftRequireTlsSSL',
            info: 'The Redshift cluster does not require TLS/SSL encryption - (Control IDs: 2.3, 4.1).',
            explanation: 'Ensure that your Amazon Redshift clusters require TLS/SSL encryption to connect to SQL clients. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.pciDss321RedshiftRequireTlsSSL,
            node: node,
        });
    }
    /**
     * Check Amazon S3 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkS3(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-S3BucketLevelPublicAccessProhibited',
            info: 'The S3 bucket does not prohibit public access through bucket level settings - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Keep sensitive data safe from unauthorized remote users by preventing public access at the bucket level.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.pciDss321S3BucketLevelPublicAccessProhibited,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-S3BucketLoggingEnabled',
            info: 'The S3 Buckets does not have server access logs enabled - (Control IDs: 2.2, 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.2.7, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.pciDss321S3BucketLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-S3BucketPublicReadProhibited',
            info: 'The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2, 2.2.2).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.pciDss321S3BucketPublicReadProhibited,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-S3BucketPublicWriteProhibited',
            info: 'The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2, 2.2.2).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.pciDss321S3BucketPublicWriteProhibited,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-S3BucketReplicationEnabled',
            info: 'The S3 Bucket does not have replication enabled - (Control IDs: 2.2, 10.5.3).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.pciDss321S3BucketReplicationEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-S3BucketServerSideEncryptionEnabled',
            info: 'The S3 Bucket does not have default server-side encryption enabled - (Control IDs: 2.2, 3.4, 8.2.1, 10.5).',
            explanation: 'Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.pciDss321S3BucketServerSideEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-S3BucketVersioningEnabled',
            info: 'The S3 Bucket does not have versioning enabled - (Control ID: 10.5.3).',
            explanation: 'Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.pciDss321S3BucketVersioningEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-S3DefaultEncryptionKMS',
            info: 'The S3 Bucket is not encrypted with a KMS Key by default - (Control IDs: 3.4, 8.2.1, 10.5).',
            explanation: 'Ensure that encryption is enabled for your Amazon Simple Storage Service (Amazon S3) buckets. Because sensitive data can exist at rest in an Amazon S3 bucket, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.pciDss321S3DefaultEncryptionKMS,
            node: node,
        });
    }
    /**
     * Check SageMaker Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSageMaker(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-SageMakerEndpointConfigurationKMSKeyConfigured',
            info: 'The SageMaker resource endpoint is not encrypted with a KMS key - (Control IDs: 3.4, 8.2.1).',
            explanation: 'Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.pciDss321SageMakerEndpointConfigurationKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-SageMakerNotebookInstanceKMSKeyConfigured',
            info: 'The SageMaker notebook is not encrypted with a KMS key - (Control IDs: 3.4, 8.2.1).',
            explanation: 'Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.pciDss321SageMakerNotebookInstanceKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-SageMakerNotebookNoDirectInternetAccess',
            info: 'The SageMaker notebook does not disable direct internet access - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.pciDss321SageMakerNotebookNoDirectInternetAccess,
            node: node,
        });
    }
    /**
     * Check Secrets Manager Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSecretsManager(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-SecretsManagerUsingKMSKey',
            info: 'The secret is not encrypted with a KMS Customer managed key - (Control IDs: 3.4, 8.2.1).',
            explanation: 'To help protect data at rest, ensure encryption with AWS Key Management Service (AWS KMS) is enabled for AWS Secrets Manager secrets. Because sensitive data can exist at rest in Secrets Manager secrets, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: secretsmanager_1.pciDss321SecretsManagerUsingKMSKey,
            node: node,
        });
    }
    /**
     * Check Amazon SNS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSNS(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-SNSEncryptedKMS',
            info: 'The SNS topic does not have KMS encryption enabled - (Control ID: 8.2.1).',
            explanation: 'To help protect data at rest, ensure that your Amazon Simple Notification Service (Amazon SNS) topics require encryption using AWS Key Management Service (AWS KMS) Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sns_1.pciDss321SNSEncryptedKMS,
            node: node,
        });
    }
    /**
     * Check VPC Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkVPC(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-VPCDefaultSecurityGroupClosed',
            info: "The VPC's default security group allows inbound or outbound traffic - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 2.1, 2.2, 2.2.2).",
            explanation: 'Amazon Elastic Compute Cloud (Amazon EC2) security groups can help in the management of network access by providing stateful filtering of ingress and egress network traffic to AWS resources. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: vpc_1.pciDss321VPCDefaultSecurityGroupClosed,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-VPCFlowLogsEnabled',
            info: 'The VPC does not have an associated Flow Log - (Control IDs: 2.2, 10.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6).',
            explanation: 'The VPC flow logs provide detailed records for information about the IP traffic going to and from network interfaces in your Amazon Virtual Private Cloud (Amazon VPC). By default, the flow log record includes values for the different components of the IP flow, including the source, destination, and protocol.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.pciDss321VPCFlowLogsEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-VPCNoUnrestrictedRouteToIGW',
            info: "The route table may contain one or more unrestricted route(s) to an IGW ('0.0.0.0/0' or '::/0') - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 2.2.2).",
            explanation: 'Ensure Amazon EC2 route tables do not have unrestricted routes to an internet gateway. Removing or limiting the access to the internet for workloads within Amazon VPCs can reduce unintended access within your environment.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.pciDss321VPCNoUnrestrictedRouteToIGW,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-VPCSubnetAutoAssignPublicIpDisabled',
            info: 'The subnet auto-assigns public IP addresses - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Manage access to the AWS Cloud by ensuring Amazon Virtual Private Cloud (VPC) subnets are not automatically assigned a public IP address. Amazon Elastic Compute Cloud (EC2) instances that are launched into subnets that have this attribute enabled have a public IP address assigned to their primary network interface.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.pciDss321VPCSubnetAutoAssignPublicIpDisabled,
            node: node,
        });
    }
    /**
     * Check WAF Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkWAF(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-WAFv2LoggingEnabled',
            info: 'The WAFv2 web ACL does not have logging enabled - (Control IDs: 10.1, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6, 10.5.4).',
            explanation: 'AWS WAF logging provides detailed information about the traffic that is analyzed by your web ACL. The logs record the time that AWS WAF received the request from your AWS resource, information about the request, and an action for the rule that each request matched.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: waf_1.pciDss321WAFv2LoggingEnabled,
            node: node,
        });
    }
}
exports.PCIDSS321Checks = PCIDSS321Checks;
_a = JSII_RTTI_SYMBOL_1;
PCIDSS321Checks[_a] = { fqn: "monocdk-nag.PCIDSS321Checks", version: "0.1.35" };
//# sourceMappingURL=data:application/json;base64,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