"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_backup_1 = require("monocdk/aws-backup");
const aws_dynamodb_1 = require("monocdk/aws-dynamodb");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * DynamoDB tables are part of AWS Backup plan(s) - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5))
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_dynamodb_1.CfnTable) {
        const tableLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        const tableName = monocdk_1.Stack.of(node).resolve(node.tableName);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_backup_1.CfnBackupSelection) {
                if (isMatchingSelection(child, tableLogicalId, tableName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether the Backup Plan Selection contains the given Table
 * @param node the CfnBackupSelection to check
 * @param tableLogicalId the Cfn Logical ID of the table
 * @param tableName the name of the table
 * returns whether the CfnBackupSelection contains the given Table
 */
function isMatchingSelection(node, tableLogicalId, tableName) {
    const backupSelection = monocdk_1.Stack.of(node).resolve(node.backupSelection);
    const resources = monocdk_1.Stack.of(node).resolve(backupSelection.resources);
    if (Array.isArray(resources)) {
        for (const resource of resources) {
            const resolvedResource = JSON.stringify(monocdk_1.Stack.of(node).resolve(resource));
            if (new RegExp(`${tableLogicalId}(?![\\w])`).test(resolvedResource) ||
                (tableName != undefined &&
                    new RegExp(`table\/${tableName}(?![\\w\\-_\\.])`).test(resolvedResource))) {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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