"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_applicationautoscaling_1 = require("monocdk/aws-applicationautoscaling");
const aws_dynamodb_1 = require("monocdk/aws-dynamodb");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * Provisioned capacity DynamoDB tables have auto-scaling enabled on their indexes - (Control IDs: CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), CP-2(6), CP-6(2), CP-10, SC-5(2), SC-6, SC-22, SC-36, SI-13(5))
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_dynamodb_1.CfnTable) {
        if (nag_pack_1.resolveResourceFromInstrinsic(node, node.billingMode) !==
            aws_dynamodb_1.BillingMode.PAY_PER_REQUEST) {
            const indexWriteMatches = [''];
            const indexReadMatches = [''];
            const tableLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
            const tableName = monocdk_1.Stack.of(node).resolve(node.tableName);
            const globalSecondaryIndexes = monocdk_1.Stack.of(node).resolve(node.globalSecondaryIndexes);
            if (Array.isArray(globalSecondaryIndexes)) {
                globalSecondaryIndexes.forEach((gsi) => {
                    const resolvedGsi = monocdk_1.Stack.of(node).resolve(gsi);
                    indexWriteMatches.push(resolvedGsi.indexName);
                    indexReadMatches.push(resolvedGsi.indexName);
                });
            }
            for (const child of monocdk_1.Stack.of(node).node.findAll()) {
                if (child instanceof aws_applicationautoscaling_1.CfnScalableTarget) {
                    const writeMatchIndex = isMatchingScalableTarget(child, 'WriteCapacityUnits', tableLogicalId, tableName, indexWriteMatches);
                    if (writeMatchIndex !== -1) {
                        indexWriteMatches.splice(writeMatchIndex, 1);
                        continue;
                    }
                    const readMatchIndex = isMatchingScalableTarget(child, 'ReadCapacityUnits', tableLogicalId, tableName, indexReadMatches);
                    if (readMatchIndex !== -1) {
                        indexReadMatches.splice(readMatchIndex, 1);
                    }
                }
            }
            if (indexWriteMatches.length != 0 || indexReadMatches.length != 0) {
                return false;
            }
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether the CfnScalableTarget is related to the given Table index
 * @param node the CfnScalableTarget to check
 * @param dimension the dimension of the CfnScalableTarget to check
 * @param tableLogicalId the Cfn Logical ID of the table
 * @param tableName the name of the table
 * @param indexNames the names of the indexes to check against
 * returns the location in the indexNames array of the matching index or -1 if no match is found
 */
function isMatchingScalableTarget(node, dimension, tableLogicalId, tableName, indexNames) {
    if (node.serviceNamespace === 'dynamodb') {
        let scalableDimension = '';
        const resourceId = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.resourceId));
        for (let i = 0; i < indexNames.length; i++) {
            const regexes = Array();
            const indexName = indexNames[i];
            if (indexName === '') {
                regexes.push(`${tableLogicalId}.*`);
                if (tableName !== undefined) {
                    regexes.push(`${tableName}.*`);
                }
                scalableDimension = `dynamodb:table:${dimension}`;
            }
            else {
                regexes.push(`${tableLogicalId}.*index\/${indexName}(?![\\w\\-_\\.])`);
                if (tableName !== undefined) {
                    regexes.push(`${tableName}.*index\/${indexName}(?![\\w\\-_\\.])`);
                }
                scalableDimension = `dynamodb:index:${dimension}`;
            }
            const regex = new RegExp(regexes.join('|'), 'gm');
            if (node.scalableDimension === scalableDimension &&
                regex.test(resourceId)) {
                return i;
            }
        }
    }
    return -1;
}
//# sourceMappingURL=data:application/json;base64,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