"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
const aws_elasticloadbalancing_1 = require("monocdk/aws-elasticloadbalancing");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * CLBs utilize secure ACM-managed certificates - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13)
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_elasticloadbalancing_1.CfnLoadBalancer) {
        //For each listener, ensure that it's utilizing an ACM SSL/HTTPS cert
        const listeners = monocdk_1.Stack.of(node).resolve(node.listeners);
        if (listeners != undefined) {
            //Iterate through listeners, checking if secured ACM certs are used
            for (const listener of listeners) {
                const resolvedListener = monocdk_1.Stack.of(node).resolve(listener);
                const listenerARN = nag_pack_1.resolveIfPrimitive(node, resolvedListener.sslCertificateId);
                //Use the ARN to check if this is an ACM managed cert
                if (listenerARN == undefined) {
                    return false;
                }
                else {
                    const acmRegex = /^arn:[^:]+:acm:.+$/;
                    if (!acmRegex.test(listenerARN)) {
                        return false;
                    }
                }
            }
        }
    }
    return true;
}
exports.default = default_1;
//# sourceMappingURL=data:application/json;base64,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