"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ec2_1 = require("monocdk/aws-ec2");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
const BLOCKED_PORTS = [20, 21, 3389, 3309, 3306, 4333];
/**
 * EC2 instances have all common TCP ports restricted for ingress IPv4 traffic - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)))
 * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/default-custom-security-groups.html
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_ec2_1.CfnSecurityGroup) {
        const ingressRules = monocdk_1.Stack.of(node).resolve(node.securityGroupIngress);
        if (ingressRules != undefined) {
            //For each ingress rule, ensure that it does not allow unrestricted SSH traffic.
            for (const rule of ingressRules) {
                const resolvedRule = monocdk_1.Stack.of(node).resolve(rule);
                for (const portNum of BLOCKED_PORTS) {
                    if (!testPort(node, resolvedRule, portNum)) {
                        return false;
                    }
                }
            }
        }
    }
    else if (node instanceof aws_ec2_1.CfnSecurityGroupIngress) {
        for (const portNum of BLOCKED_PORTS) {
            if (!testPort(node, node, portNum)) {
                return false;
            }
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to identify if the given port number is unrestricted
 * @param node the CfnResource to check
 * @param rule the CfnSecurityGroupIngress rule to check
 * @param portNum the number of the port to check
 */
function testPort(node, rule, portNum) {
    //Does this rule apply to TCP traffic?
    const ipProtocol = nag_pack_1.resolveIfPrimitive(node, rule.ipProtocol);
    const cidrIp = nag_pack_1.resolveIfPrimitive(node, rule.cidrIp);
    const fromPort = nag_pack_1.resolveIfPrimitive(node, rule.fromPort);
    const toPort = nag_pack_1.resolveIfPrimitive(node, rule.toPort);
    if (ipProtocol === 'tcp') {
        //Does this rule allow all IPv4 addresses (unrestricted access)?
        if (cidrIp != undefined && cidrIp.includes('/0')) {
            //Is a port range specified?
            if (fromPort != undefined && toPort != undefined) {
                if ((fromPort <= portNum && toPort >= portNum) ||
                    fromPort == -1 ||
                    toPort == -1) {
                    return false;
                }
            }
            else {
                if (fromPort == portNum) {
                    return false;
                }
            }
        }
    }
    //Are all ports allowed?
    if (ipProtocol === '-1') {
        return false;
    }
    return true;
}
//# sourceMappingURL=data:application/json;base64,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