"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ec2_1 = require("monocdk/aws-ec2");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
const EBS_OPTIMIZED_SUPPORTED = [
    'c1.xlarge',
    'c3.xlarge',
    'c3.2xlarge',
    'c3.4xlarge',
    'g2.2xlarge',
    'i2.xlarge',
    'i2.2xlarge',
    'i2.4xlarge',
    'm1.large',
    'm1.xlarge',
    'm2.2xlarge',
    'm2.4xlarge',
    'm3.xlarge',
    'm3.2xlarge',
    'r3.xlarge',
    'r3.2xlarge',
    'r3.4xlarge',
];
const DEFAULT_TYPE = 'm1.small';
/**
 * EC2 instance types that support EBS optimization and are not EBS optimized by default have EBS optimization enabled - (Control ID: 164.308(a)(7)(i))
 * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html#previous
 *  @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_ec2_1.CfnInstance) {
        const instanceType = node.instanceType
            ? nag_pack_1.resolveIfPrimitive(node, node.instanceType)
            : DEFAULT_TYPE;
        const ebsOptimized = monocdk_1.Stack.of(node).resolve(node.ebsOptimized);
        if (EBS_OPTIMIZED_SUPPORTED.includes(instanceType) &&
            ebsOptimized !== true) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
//# sourceMappingURL=data:application/json;base64,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