"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_applicationautoscaling_1 = require("monocdk/aws-applicationautoscaling");
const aws_dynamodb_1 = require("monocdk/aws-dynamodb");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * Provisioned capacity DynamoDB tables have auto-scaling enabled on their indexes - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(C))
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_dynamodb_1.CfnTable) {
        if (nag_pack_1.resolveResourceFromInstrinsic(node, node.billingMode) !==
            aws_dynamodb_1.BillingMode.PAY_PER_REQUEST) {
            const indexWriteMatches = [''];
            const indexReadMatches = [''];
            const tableLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
            const tableName = monocdk_1.Stack.of(node).resolve(node.tableName);
            const globalSecondaryIndexes = monocdk_1.Stack.of(node).resolve(node.globalSecondaryIndexes);
            if (Array.isArray(globalSecondaryIndexes)) {
                globalSecondaryIndexes.forEach((gsi) => {
                    const resolvedGsi = monocdk_1.Stack.of(node).resolve(gsi);
                    indexWriteMatches.push(resolvedGsi.indexName);
                    indexReadMatches.push(resolvedGsi.indexName);
                });
            }
            for (const child of monocdk_1.Stack.of(node).node.findAll()) {
                if (child instanceof aws_applicationautoscaling_1.CfnScalableTarget) {
                    const writeMatchIndex = isMatchingScalableTarget(child, 'WriteCapacityUnits', tableLogicalId, tableName, indexWriteMatches);
                    if (writeMatchIndex !== -1) {
                        indexWriteMatches.splice(writeMatchIndex, 1);
                        continue;
                    }
                    const readMatchIndex = isMatchingScalableTarget(child, 'ReadCapacityUnits', tableLogicalId, tableName, indexReadMatches);
                    if (readMatchIndex !== -1) {
                        indexReadMatches.splice(readMatchIndex, 1);
                    }
                }
            }
            if (indexWriteMatches.length != 0 || indexReadMatches.length != 0) {
                return false;
            }
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether the CfnScalableTarget is related to the given Table index
 * @param node the CfnScalableTarget to check
 * @param dimension the dimension of the CfnScalableTarget to check
 * @param tableLogicalId the Cfn Logical ID of the table
 * @param tableName the name of the table
 * @param indexNames the names of the indexes to check against
 * returns the location in the indexNames array of the matching index or -1 if no match is found
 */
function isMatchingScalableTarget(node, dimension, tableLogicalId, tableName, indexNames) {
    if (node.serviceNamespace === 'dynamodb') {
        let scalableDimension = '';
        const resourceId = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.resourceId));
        for (let i = 0; i < indexNames.length; i++) {
            const regexes = Array();
            const indexName = indexNames[i];
            if (indexName === '') {
                regexes.push(`${tableLogicalId}.*`);
                if (tableName !== undefined) {
                    regexes.push(`${tableName}.*`);
                }
                scalableDimension = `dynamodb:table:${dimension}`;
            }
            else {
                regexes.push(`${tableLogicalId}.*index\/${indexName}(?![\\w\\-_\\.])`);
                if (tableName !== undefined) {
                    regexes.push(`${tableName}.*index\/${indexName}(?![\\w\\-_\\.])`);
                }
                scalableDimension = `dynamodb:index:${dimension}`;
            }
            const regex = new RegExp(regexes.join('|'), 'gm');
            if (node.scalableDimension === scalableDimension &&
                regex.test(resourceId)) {
                return i;
            }
        }
    }
    return -1;
}
//# sourceMappingURL=data:application/json;base64,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