"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
const aws_secretsmanager_1 = require("monocdk/aws-secretsmanager");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * Secrets have automatic rotation scheduled
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_secretsmanager_1.CfnSecret) {
        const secretLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        const secretTargetAttachmentLogicalIds = Array();
        const cfnSecretTargetAttachments = Array();
        const cfnRotationSchedules = Array();
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_secretsmanager_1.CfnSecretTargetAttachment) {
                cfnSecretTargetAttachments.push(child);
            }
            else if (child instanceof aws_secretsmanager_1.CfnRotationSchedule) {
                cfnRotationSchedules.push(child);
            }
        }
        if (cfnRotationSchedules.length === 0) {
            return false;
        }
        let found = false;
        for (const child of cfnSecretTargetAttachments) {
            const attachmentLogicalId = getMatchingSecretTargetAttachment(child, secretLogicalId);
            if (attachmentLogicalId) {
                secretTargetAttachmentLogicalIds.push(attachmentLogicalId);
            }
        }
        for (const child of cfnRotationSchedules) {
            if (isMatchingRotationSchedule(child, secretLogicalId, secretTargetAttachmentLogicalIds)) {
                found = true;
                break;
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether a given Secret Target Attachment is associated with the given secret.
 * @param node The CfnTargetAttachment to check.
 * @param secretLogicalId The Cfn Logical ID of the secret.
 * Returns the Logical ID if the attachment if is associated with the secret, otherwise and empty string.
 */
function getMatchingSecretTargetAttachment(node, secretLogicalId) {
    const resourceSecretId = nag_pack_1.resolveResourceFromInstrinsic(node, node.secretId);
    if (secretLogicalId === resourceSecretId) {
        return nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
    }
    return '';
}
/**
 * Helper function to check whether a given Rotation Schedule is associated with the given secret.
 * @param node The CfnRotationSchedule to check.
 * @param secretLogicalId The Cfn Logical ID of the secret.
 * @param secretTargetAttachmentLogicalIds The Cfn Logical IDs of any Secret Target Attachments associated with the given secret.
 * Returns whether the CfnRotationSchedule is associated with the given secret.
 */
function isMatchingRotationSchedule(node, secretLogicalId, secretTargetAttachmentLogicalIds) {
    const resourceSecretId = nag_pack_1.resolveResourceFromInstrinsic(node, node.secretId);
    if (secretLogicalId === resourceSecretId ||
        secretTargetAttachmentLogicalIds.includes(resourceSecretId)) {
        if (monocdk_1.Stack.of(node).resolve(node.hostedRotationLambda) === undefined &&
            monocdk_1.Stack.of(node).resolve(node.rotationLambdaArn) === undefined) {
            return false;
        }
        const rotationRules = monocdk_1.Stack.of(node).resolve(node.rotationRules);
        if (rotationRules !== undefined) {
            const automaticallyAfterDays = monocdk_1.Stack.of(node).resolve(rotationRules.automaticallyAfterDays);
            if (automaticallyAfterDays !== undefined) {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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