"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsSolutionsChecks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../nag-pack");
const analytics_1 = require("./rules/analytics");
const application_integration_1 = require("./rules/application_integration");
const compute_1 = require("./rules/compute");
const databases_1 = require("./rules/databases");
const developer_tools_1 = require("./rules/developer_tools");
const machine_learning_1 = require("./rules/machine_learning");
const management_and_governance_1 = require("./rules/management_and_governance");
const media_services_1 = require("./rules/media_services");
const network_and_delivery_1 = require("./rules/network_and_delivery");
const security_and_compliance_1 = require("./rules/security_and_compliance");
const serverless_1 = require("./rules/serverless");
const storage_1 = require("./rules/storage");
/**
 * Check Best practices based on AWS Solutions Security Matrix.
 *
 * @stability stable
 */
class AwsSolutionsChecks extends nag_pack_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        if (node instanceof monocdk_1.CfnResource) {
            this.checkCompute(node);
            this.checkStorage(node);
            this.checkDatabases(node);
            this.checkNetworkDelivery(node);
            this.checkManagementGovernance(node);
            this.checkMachineLearning(node);
            this.checkAnalytics(node);
            this.checkSecurityCompliance(node);
            this.checkServerless(node);
            this.checkApplicationIntegration(node);
            this.checkMediaServices(node);
            this.checkDeveloperTools(node);
        }
    }
    /**
     * Check Compute Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCompute(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-EB1',
            info: 'The Elastic Beanstalk environment is not configured to use a specific VPC.',
            explanation: 'Use a non-default in order to seperate your environment from default resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: compute_1.awsSolutionsEb1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EB3',
            info: 'The Elastic Beanstalk environment does not have managed updates enabled.',
            explanation: 'Enable managed platform updates for beanstalk environments in order to receive bug fixes, software updates and new features. Managed platform updates perform immutable environment updates.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: compute_1.awsSolutionsEb3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EB4',
            info: 'The Elastic Beanstalk environment does not upload EC2 Instance logs to S3.',
            explanation: 'Beanstalk environment logs should be retained and uploaded to Amazon S3 in order to keep the logging data for future audits, historical purposes or to track and analyze the EB application environment behavior for a long period of time.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: compute_1.awsSolutionsEb4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EC23',
            info: 'The Security Group allows for 0.0.0.0/0 or ::/0 inbound access.',
            explanation: 'Large port ranges, when open, expose instances to unwanted attacks. More than that, they make traceability of vulnerabilities very difficult. For instance, your web servers may only require 80 and 443 ports to be open, but not all. One of the most common mistakes observed is when  all ports for 0.0.0.0/0 range are open in a rush to access the instance. EC2 instances must expose only to those ports enabled on the corresponding security group level.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: compute_1.awsSolutionsEc23,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EC26',
            info: 'The EBS volume has encryption disabled.',
            explanation: "With EBS encryption, you aren't required to build, maintain, and secure your own key management infrastructure. EBS encryption uses KMS keys when creating encrypted volumes and snapshots. This helps protect data at rest.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: compute_1.awsSolutionsEc26,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EC27',
            info: 'The Security Group does not have a description.',
            explanation: 'Descriptions help simplify operations and remove any opportunities for operator errors.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: compute_1.awsSolutionsEc27,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EC28',
            info: 'The EC2 instance/AutoScaling launch configuration does not have detailed monitoring enabled.',
            explanation: 'Monitoring data helps make better decisions on architecting and managing compute resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: compute_1.awsSolutionsEc28,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EC29',
            info: 'The EC2 instance is not part of an ASG and has Termination Protection disabled.',
            explanation: 'Termination Protection safety feature enabled in order to protect the instances from being accidentally terminated.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: compute_1.awsSolutionsEc29,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ECR1',
            info: 'The ECR Repository allows open access.',
            explanation: 'Removing * principals in an ECR Repository helps protect against unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: compute_1.awsSolutionsEcr1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ECS4',
            info: 'The ECS Cluster has CloudWatch Container Insights disabled.',
            explanation: 'CloudWatch Container Insights allow operators to gain a better perspective on how the cluster’s applications and microservices are performing.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: compute_1.awsSolutionsEcs4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ECS7',
            info: 'The ECS Task Definition does not have awslogs logging enabled at the minimum.',
            explanation: 'Container logging allows operators to view and aggregate the logs from the container.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: compute_1.awsSolutionsEcs7,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB1',
            info: 'The CLB is used for incoming HTTP/HTTPS traffic. Use ALBs instead.',
            explanation: 'HTTP/HTTPS applications (monolithic or containerized) should use the ALB instead of the CLB for enhanced incoming traffic distribution, better performance and lower costs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: compute_1.awsSolutionsElb1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB2a',
            info: 'The ALB does not have access logs enabled.',
            explanation: 'Access logs allow operators to to analyze traffic patterns and identify and troubleshoot security issues.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: compute_1.awsSolutionsElb2a,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB2e',
            info: 'The CLB does not have access logs enabled.',
            explanation: 'Access logs allow operators to to analyze traffic patterns and identify and troubleshoot security issues.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: compute_1.awsSolutionsElb2e,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB3',
            info: 'The CLB does not have connection draining enabled.',
            explanation: 'With Connection Draining feature enabled, if an EC2 backend instance fails health checks The CLB will not send any new requests to the unhealthy instance. However, it will still allow existing (in-flight) requests to complete for the duration of the configured timeout.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: compute_1.awsSolutionsElb3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB4',
            info: 'The CLB does not use at least two AZs with the Cross-Zone Load Balancing feature enabled.',
            explanation: 'CLBs can distribute the traffic evenly across all backend instances. To use Cross-Zone Load Balancing at optimal level, the system should maintain an equal EC2 capacity distribution in each of the AZs registered with the load balancer.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: compute_1.awsSolutionsElb4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB5',
            info: 'The CLB listener is not configured for secure (HTTPs or SSL) protocols for client communication.',
            explanation: 'The HTTPs or SSL protocols enable secure communication by encrypting the communication between the client and the load balancer.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: compute_1.awsSolutionsElb5,
            node: node,
        });
    }
    /**
     * Check Storage Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkStorage(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-S1',
            info: 'The S3 Bucket has server access logs disabled.',
            explanation: 'The bucket should have server access logging enabled to provide detailed records for the requests that are made to the bucket.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: storage_1.awsSolutionsS1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-S2',
            info: 'The S3 Bucket does not have public access restricted and blocked.',
            explanation: 'The bucket should have public access restricted and blocked to prevent unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: storage_1.awsSolutionsS2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-S3',
            info: 'The S3 Bucket does not default encryption enabled.',
            explanation: 'The bucket should minimally have SSE enabled to help protect data-at-rest.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: storage_1.awsSolutionsS3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EFS1',
            info: 'The EFS is not configured for encryption at rest.',
            explanation: 'By using an encrypted file system, data and metadata are automatically encrypted before being written to the file system. Similarly, as data and metadata are read, they are automatically decrypted before being presented to the application. These processes are handled transparently by EFS without requiring modification of applications.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: storage_1.awsSolutionsEfs1,
            node: node,
        });
    }
    /**
     * Check Database Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDatabases(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-RDS2',
            info: 'The RDS instance or Aurora DB cluster does not have storage encryption enabled.',
            explanation: 'Storage encryption helps protect data-at-rest by encrypting the underlying storage, automated backups, read replicas, and snapshots for the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRds2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS6',
            info: 'The RDS Aurora MySQL/PostgresSQL cluster does not have IAM Database Authentication enabled.',
            explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the MySQL/PostgreSQL database instances, instead it uses an authentication token.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRds6,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS10',
            info: 'The RDS instance or Aurora DB cluster does not have deletion protection enabled.',
            explanation: 'The deletion protection feature helps protect the database from being accidentally deleted.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRds10,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS11',
            info: 'The RDS instance or Aurora DB cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MySQL/Aurora port 3306, SQL Server port 1433, PostgreSQL port 5432, etc).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRds11,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS13',
            info: 'The RDS instance is not configured for automated backups.',
            explanation: 'Automated backups allow for point-in-time recovery.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRds13,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS14',
            info: 'The RDS Aurora MySQL cluster does not have Backtrack enabled.',
            explanation: 'Backtrack helps order to rewind cluster tables to a specific time, without using backups.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRds14,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS15',
            info: 'The RDS Aurora cluster does not have deletion protection enabled.',
            explanation: 'Enabling Deletion Protection at the cluster level helps protect Amazon Aurora dtabases from accidental deletion.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRds15,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS16',
            info: 'The RDS Aurora MySQL serverless cluster does not have audit, error, general, and slowquery Log Exports enabled.',
            explanation: 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRds16,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DDB3',
            info: 'The DynamoDB table does not have Point-in-time Recovery enabled.',
            explanation: 'DynamoDB continuous backups represent an additional layer of insurance against accidental loss of data on top of on-demand backups. The DynamoDB service can back up the data with per-second granularity and restore it to any single second from the time PITR was enabled up to the prior 35 days.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: databases_1.awsSolutionsDdb3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DDB4',
            info: 'The DAX cluster does not have server-side encryption enabled.',
            explanation: 'Data in cache, configuration data and log files should be encrypted using Server-Side Encryption in order to protect from unauthorized access to the underlying storage.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsDdb4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AEC1',
            info: 'The ElastiCache cluster is not provisioned in a VPC.',
            explanation: 'Provisioning the cluster within a VPC allows for better flexibility and control over the cache clusters security, availability, traffic routing and more.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsAec1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AEC3',
            info: 'The ElastiCache Redis cluster does not have both encryption in transit and at rest enabled.',
            explanation: 'Encryption in transit helps secure communications to the cluster. Encryption at rest helps protect data at rest from unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsAec3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AEC4',
            info: 'The ElastiCache Redis cluster is not deployed in a Multi-AZ configuration.',
            explanation: 'The cluster should use a Multi-AZ deployment configuration for high availability.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsAec4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AEC5',
            info: 'The ElastiCache cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redis port 6379 and Memcached port 11211).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsAec5,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AEC6',
            info: 'The ElastiCache Redis cluster does not use Redis AUTH for user authentication.',
            explanation: 'Redis authentication tokens enable Redis to require a token (password) before allowing clients to execute commands, thereby improving data security.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsAec6,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-N1',
            info: 'The Neptune DB cluster is not deployed in a Multi-AZ configuration.',
            explanation: 'The cluster should use a Multi-AZ deployment configuration for high availability.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsN1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-N2',
            info: 'The Neptune DB instance does have Auto Minor Version Upgrade enabled.',
            explanation: 'The Neptune service regularly releases engine updates. Enabling Auto Minor Version Upgrade will allow the service to automatically apply these upgrades to DB Instances.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsN2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-N3',
            info: 'The Neptune DB cluster does not have a reasonable minimum backup retention period configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsN3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-N4',
            info: 'The Neptune DB cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsN4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-N5',
            info: 'The Neptune DB cluster does not have IAM Database Authentication enabled.',
            explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the cluster.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsN5,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS1',
            info: 'The Redshift cluster parameter group must have the "require_ssl" parameter enabled.',
            explanation: 'Enabling the "require_ssl" parameter secures data-in-transit by encrypting the connection between the clients and the Redshift clusters.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRs1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS2',
            info: 'The Redshift cluster is not provisioned in a VPC.',
            explanation: 'Provisioning the cluster within a VPC allows for better flexibility and control over the Redshift clusters security, availability, traffic routing and more.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRs2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS3',
            info: 'The Redshift cluster uses the default "awsuser" username.',
            explanation: 'Using a custom master user name instead of the default master user name (i.e. "awsuser") provides an additional layer of defense against non-targeted attacks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRs3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS4',
            info: 'The Redshift cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redshift port 5439).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRs4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS5',
            info: 'The Redshift cluster does not have audit logging enabled.',
            explanation: 'Audit logging helps operators troubleshoot issues and ensure security.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRs5,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS6',
            info: 'The Redshift cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRs6,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS8',
            info: 'The Redshift cluster is publicly accessible.',
            explanation: 'Disabling public accessibility helps minimize security risks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRs8,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS9',
            info: 'The Redshift cluster does not have version upgrade enabled.',
            explanation: 'Version Upgrade must enabled to enable the cluster to automatically receive upgrades during the maintenance window.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRs9,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS10',
            info: 'The Redshift cluster does not have a retention period for automated snapshots configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A positive retention period should be set to configure this feature.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRs10,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS11',
            info: 'The Redshift cluster does not have user activity logging enabled.',
            explanation: 'User activity logging logs each query before it is performed on the clusters databse. To enable this feature associate a Resdhsift Cluster Parameter Group with the "enable_user_activity_logging" parameter set to "true".',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsRs11,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DOC1',
            info: 'The Document DB cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsDoc1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DOC2',
            info: 'The Document DB cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MongoDB port 27017).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsDoc2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DOC3',
            info: 'The Document DB cluster does not have the username and password stored in Secrets Manager.',
            explanation: "Secrets Manager enables operators to replace hardcoded credentials in your code, including passwords, with an API call to Secrets Manager to retrieve the secret programmatically. This helps ensure the secret can't be compromised by someone examining system code, because the secret no longer exists in the code. Also, operators can configure Secrets Manager to automatically rotate the secret for you according to a specified schedule. This enables you to replace long-term secrets with short-term ones, significantly reducing the risk of compromise.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsDoc3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DOC4',
            info: 'The Document DB cluster does not have a reasonable minimum backup retention period configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsDoc4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DOC5',
            info: 'The Document DB cluster does not have authenticate, createIndex, and dropCollection Log Exports enabled.',
            explanation: 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database. The events recorded by the AWS DocumentDB audit logs include successful and failed authentication attempts, creating indexes or dropping a collection in a database within the DocumentDB cluster.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: databases_1.awsSolutionsDoc5,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-TS3',
            info: 'The Timestream database does not use a Customer Managed KMS Key for at rest encryption.',
            explanation: 'All Timestream tables in a database are encrypted at rest by default using AWS Managed Key. These keys are rotated every three years. Data at rest must be encrypted using CMKs if you require more control over the permissions and lifecycle of your keys, including the ability to have them automatically rotated on an annual basis.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: databases_1.awsSolutionsTs3,
            node: node,
        });
    }
    /**
     * Check Network and Delivery Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkNetworkDelivery(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-VPC3',
            info: 'A Network ACL or Network ACL entry has been implemented.',
            explanation: 'Network ACLs should be used sparingly for the following reasons: they can be complex to manage, they are stateless, every IP address must be explicitly opened in each (inbound/outbound) direction, and they affect a complete subnet. Use security groups when possible as they are stateful and easier to manage.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: network_and_delivery_1.awsSolutionsVpc3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-VPC7',
            info: 'The VPC does not have an associated Flow Log.',
            explanation: 'VPC Flow Logs capture network flow information for a VPC, subnet, or network interface and stores it in Amazon CloudWatch Logs. Flow log data can help customers troubleshoot network issues; for example, to diagnose why specific traffic is not reaching an instance, which might be a result of overly restrictive security group rules.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: network_and_delivery_1.awsSolutionsVpc7,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CFR1',
            info: 'The CloudFront distribution may require Geo restrictions.',
            explanation: 'Geo restriction may need to be enabled for the distribution in order to allow or deny a country in order to allow or restrict users in specific locations from accessing content.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: network_and_delivery_1.awsSolutionsCfr1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CFR2',
            info: 'The CloudFront distribution may require integration with AWS WAF.',
            explanation: 'The Web Application Firewall can help protect against application-layer attacks that can compromise the security of the system or place unnecessary load on them.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: network_and_delivery_1.awsSolutionsCfr2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CFR3',
            info: 'The CloudFront distributions does not have access logging enabled.',
            explanation: 'Enabling access logs helps operators track all viewer requests for the content delivered through the Content Delivery Network.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: network_and_delivery_1.awsSolutionsCfr3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CFR5',
            info: 'The CloudFront distributions uses SSLv3 or TLSv1 for communication to the origin.',
            explanation: 'Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Using a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS helps protect viewer connections.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: network_and_delivery_1.awsSolutionsCfr5,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CFR6',
            info: 'The CloudFront distribution does not use an origin access identity an S3 origin.',
            explanation: 'Origin access identities help with security by restricting any direct access to objects through S3 URLs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: network_and_delivery_1.awsSolutionsCfr6,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-APIG1',
            info: 'The API does not have access logging enabled.',
            explanation: 'Enabling access logs helps operators view who accessed an API and how the caller accessed the API.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: network_and_delivery_1.awsSolutionsApig1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-APIG2',
            info: 'The REST API does not have request validation enabled.',
            explanation: 'The API should have basic request validation enabled. If the API is integrated with custom source (Lambda, ECS, etc..) in the backend, deeper input validation should be considered for implementation.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: network_and_delivery_1.awsSolutionsApig2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-APIG3',
            info: 'The REST API stage is not associated with AWS WAFv2 web ACL.',
            explanation: 'AWS WAFv2 is a web application firewall that helps protect web applications and APIs from attacks by allowing configured rules to allow, block, or monitor (count) web requests based on customizable rules and conditions that are defined.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: network_and_delivery_1.awsSolutionsApig3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-APIG4',
            info: 'The API does not implement authorization.',
            explanation: 'In most cases an API needs to have an authentication and authorization implementation strategy. This includes using such approaches as IAM, Cognito User Pools, Custom authorizer, etc.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: network_and_delivery_1.awsSolutionsApig4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-APIG6',
            info: 'The REST API Stage does not have CloudWatch logging enabled for all methods.',
            explanation: 'Enabling CloudWatch logs at the stage level helps operators to track and analyze execution behavior at the API stage level.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: network_and_delivery_1.awsSolutionsApig6,
            node: node,
        });
    }
    /**
     * Check Management and Governance Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkManagementGovernance(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-AS1',
            info: 'The Auto Scaling Group does not have a cooldown period.',
            explanation: 'A cooldown period temporarily suspends any scaling activities in order to allow the newly launched EC2 instance(s) some time to start handling the application traffic.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: management_and_governance_1.awsSolutionsAs1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AS2',
            info: 'The Auto Scaling Group does not have properly configured health checks.',
            explanation: 'The health check feature enables the service to detect whether its registered EC2 instances are healthy or not.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: management_and_governance_1.awsSolutionsAs2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AS3',
            info: 'The Auto Scaling Group does not have notifications configured for all scaling events.',
            explanation: 'Notifications on EC2 instance launch, launch error, termination, and termination errors allow operators to gain better insights into systems attributes such as activity and health.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: management_and_governance_1.awsSolutionsAs3,
            node: node,
        });
    }
    /**
     * Check Machine Learning Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkMachineLearning(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-SM1',
            info: 'The SageMaker notebook instance is not provisioned inside a VPC.',
            explanation: 'Provisioning the notebook instances inside a VPC enables the notebook to access VPC-only resources such as EFS file systems.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: machine_learning_1.awsSolutionsSm1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-SM2',
            info: 'The SageMaker notebook instance does not have an encrypted storage volume.',
            explanation: 'Encrypting storage volumes helps protect SageMaker data-at-rest.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: machine_learning_1.awsSolutionsSm2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-SM3',
            info: 'The SageMaker notebook instance has direct internet access enabled.',
            explanation: 'Disabling public accessibility helps minimize security risks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: machine_learning_1.awsSolutionsSm3,
            node: node,
        });
    }
    /**
     * Check Analytics Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAnalytics(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-ATH1',
            info: 'The Athena workgroup does not encrypt query results.',
            explanation: 'Encrypting query results stored in S3 helps secure data to meet compliance requirements for data-at-rest encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsAth1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EMR2',
            info: 'The EMR cluster does not have S3 logging enabled.',
            explanation: 'Uploading logs to S3 enables the system to keep the logging data for historical purposes or to track and analyze the clusters behavior.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsEmr2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EMR6',
            info: 'The EMR cluster does not implement authentication via an EC2 Key Pair or Kerberos.',
            explanation: 'SSH clients can use an EC2 key pair to authenticate to cluster instances. Alternatively, with EMR release version 5.10.0 or later, solutions can configure Kerberos to authenticate users and SSH connections to the master node.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsEmr6,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-KDA3',
            info: 'The Kinesis Data Analytics Flink Application does not have checkpointing enabled.',
            explanation: 'Checkpoints are backups of application state that KDA automatically creates periodically and uses to restore from faults.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: analytics_1.awsSolutionsKda3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-KDS1',
            info: 'The Kinesis Data Stream does not has server-side encryption enabled.',
            explanation: "Data is encrypted before it's written to the Kinesis stream storage layer, and decrypted after it’s retrieved from storage. This allows the system to meet strict regulatory requirements and enhance the security of system data.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsKds1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-KDS3',
            info: 'The Kinesis Data Stream specifies server-side encryption and does not use the "aws/kinesis" key.',
            explanation: 'Customer Managed Keys can incur additional costs that scale with the amount of consumers and producers. Ensure that Customer Managed Keys are required for compliance before using them (https://docs.aws.amazon.com/streams/latest/dev/costs-performance.html).',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: analytics_1.awsSolutionsKds3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-KDF1',
            info: 'The Kinesis Data Firehose delivery stream does have server-side encryption enabled.',
            explanation: 'This allows the system to meet strict regulatory requirements and enhance the security of system data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsKdf1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MSK2',
            info: 'The MSK cluster uses plaintext communication between clients and brokers.',
            explanation: 'TLS only communication secures data-in-transit by encrypting the connection between the clients and brokers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsMsk2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MSK3',
            info: 'The MSK cluster uses plaintext communication between brokers.',
            explanation: 'TLS communication secures data-in-transit by encrypting the connection between brokers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsMsk3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MSK6',
            info: 'The MSK cluster does not send broker logs to a supported destination.',
            explanation: 'Broker logs enable operators to troubleshoot Apache Kafka applications and to analyze their communications with the MSK cluster. The cluster can deliver logs to the following resources: a CloudWatch log group, an S3 bucket, a Kinesis Data Firehose delivery stream.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsMsk6,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS1',
            info: 'The OpenSearch Service domain is not provisioned inside a VPC.',
            explanation: 'Provisioning the domain within a VPC enables better flexibility and control over the clusters access and security as this feature keeps all traffic between the VPC and OpenSearch domains within the AWS network instead of going over the public Internet.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS2',
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled.',
            explanation: 'Enabling the node-to-node encryption feature adds an extra layer of data protection on top of the existing ES security features such as HTTPS client to cluster encryption and data-at-rest encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS3',
            info: 'The OpenSearch Service domain does not only grant access via allowlisted IP addresses.',
            explanation: 'Using allowlisted IP addresses helps protect the domain against unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS4',
            info: 'The OpenSearch Service domain does not use dedicated master nodes.',
            explanation: 'Using dedicated master nodes helps improve environmental stability by offloading all the management tasks from the data nodes.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS5',
            info: 'The OpenSearch Service domain does not allow for unsigned requests or anonymous access.',
            explanation: 'Restricting public access helps prevent unauthorized access and prevents any unsigned requests to be made to the resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs5,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS7',
            info: 'The OpenSearch Service domain does not have Zone Awareness enabled.',
            explanation: 'Enabling cross-zone replication (Zone Awareness) increases the availability of the OpenSearch Service domain by allocating the nodes and replicate the data across two AZs in the same region in order to prevent data loss and minimize downtime in the event of node or AZ failure.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs7,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS8',
            info: 'The OpenSearch Service domain does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs8,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS9',
            info: 'The OpenSearch Service domain does not minimally publish SEARCH_SLOW_LOGS and INDEX_SLOW_LOGS to CloudWatch Logs.',
            explanation: 'These logs enable operators to gain full insight into the performance of these operations.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs9,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-QS1',
            info: 'The Quicksight data sources connection is not configured to use SSL.',
            explanation: 'SSL secures communications to data sources, especially when using public networks. Using SSL with QuickSight requires the use of certificates signed by a publicly-recognized certificate authority.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsQs1,
            node: node,
        });
    }
    /**
     * Check Security and Compliance Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSecurityCompliance(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-IAM4',
            info: 'The IAM user, role, or group uses AWS managed policies.',
            explanation: 'An AWS managed policy is a standalone policy that is created and administered by AWS. Currently, many AWS managed policies do not restrict resource scope. Replace AWS managed policies with system specific (customer) managed policies.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsIam4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-IAM5',
            info: 'The IAM entity contains wildcard permissions and does not have a cdk_nag rule suppression with evidence for those permission.',
            explanation: 'Metadata explaining the evidence (e.g. via supporting links) for wildcard permissions allows for transparency to operators.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsIam5,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-COG1',
            info: 'The Cognito user pool does not have a password policy that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters.',
            explanation: 'Strong password policies increase system security by encouraging users to create reliable and secure passwords.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsCog1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-COG2',
            info: 'The Cognito user pool does not require MFA.',
            explanation: 'Multi-factor authentication (MFA) increases security for the application by adding another authentication method, and not relying solely on user name and password.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: security_and_compliance_1.awsSolutionsCog2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-COG3',
            info: 'The Cognito user pool does not have AdvancedSecurityMode set to ENFORCED.',
            explanation: 'Advanced security features enable the system to detect and act upon malicious sign-in attempts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsCog3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-COG4',
            info: 'The API GW method does not use a Cognito user pool authorizer.',
            explanation: 'API Gateway validates the tokens from a successful user pool authentication, and uses them to grant your users access to resources including Lambda functions, or your own API.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsCog4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-COG7',
            info: 'The Cognito identity pool allows for unauthenticated logins and does not have a cdk_nag rule suppression with a reason.',
            explanation: 'In many cases applications do not warrant unauthenticated guest access applications. Metadata explaining the use case allows for transparency to operators.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsCog7,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-KMS5',
            info: 'The KMS Symmetric key does not have automatic key rotation enabled.',
            explanation: 'KMS key rotation allow a system to set an yearly rotation schedule for a KMS key so when a AWS KMS key is required to encrypt new data, the KMS service can automatically use the latest version of the HSA backing key to perform the encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsKms5,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-SMG4',
            info: 'The secret does not have automatic rotation scheduled.',
            explanation: 'AWS Secrets Manager can be configured to automatically rotate the secret for a secured service or database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsSmg4,
            node: node,
        });
    }
    /**
     * Check Serverless Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkServerless(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-ASC3',
            info: 'The GraphQL API does not have request leveling logging enabled.',
            explanation: 'It is important to use CloudWatch Logs to log metrics such as who has accessed the GraphQL API, how the caller accessed the API, and invalid requests.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: serverless_1.awsSolutionsAsc3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-SF1',
            info: 'The Step Function does not log "ALL" events to CloudWatch Logs.',
            explanation: 'Logging "ALL" events to CloudWatch logs help operators troubleshoot and audit systems.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: serverless_1.awsSolutionsSf1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-SF2',
            info: 'The Step Function does not have X-Ray tracing enabled.',
            explanation: 'X-ray provides an end-to-end view of how an application is performing. This helps operators to discover performance issues, detect permission problems, and track requests made to and from other AWS services.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: serverless_1.awsSolutionsSf2,
            node: node,
        });
    }
    /**
     * Check Application Integration Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkApplicationIntegration(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-SNS2',
            info: 'The SNS Topic does not have server-side encryption enabled.',
            explanation: 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: application_integration_1.awsSolutionsSns2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-SQS2',
            info: 'The SQS Queue does not have server-side encryption enabled.',
            explanation: 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: application_integration_1.awsSolutionsSqs2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-SQS3',
            info: 'The SQS queue does not have a dead-letter queue (DLQ) enabled or have a cdk_nag rule suppression indicating it is a DLQ.',
            explanation: 'Using a DLQ helps maintain the queue flow and avoid losing data by detecting and mitigating failures and service disruptions on time.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: application_integration_1.awsSolutionsSqs3,
            node: node,
        });
    }
    /**
     * Check Media Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkMediaServices(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-MS1',
            info: 'The MediaStore container does not have container access logging enabled.',
            explanation: 'The container should have access logging enabled to provide detailed records for the requests that are made to the container.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: media_services_1.awsSolutionsMs1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MS4',
            info: 'The MediaStore container does not define a metric policy to send metrics to CloudWatch.',
            explanation: 'Using a combination of MediaStore metrics and CloudWatch alarms helps operators gain better insights into container operations.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: media_services_1.awsSolutionsMs4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MS7',
            info: 'The MediaStore container does not define a container policy.',
            explanation: 'Using a container policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: media_services_1.awsSolutionsMs7,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MS8',
            info: 'The MediaStore container does not define a CORS policy.',
            explanation: 'Using a CORS policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: media_services_1.awsSolutionsMs8,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MS10',
            info: 'The MediaStore container does not define a lifecycle policy.',
            explanation: 'Many use cases warrant the usage of lifecycle configurations to manage container objects during their lifetime.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: media_services_1.awsSolutionsMs10,
            node: node,
        });
    }
    /**
     * Check Developer Tools Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDeveloperTools(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-CB3',
            info: 'The CodeBuild project has privileged mode enabled.',
            explanation: 'Privileged grants elevated rights to the system, which introduces additional risk. Privileged mode should only be set to true only if the build project is used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: developer_tools_1.awsSolutionsCb3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CB4',
            info: 'The CodeBuild project does not use an AWS KMS key for encryption.',
            explanation: 'Using an AWS KMS key helps follow the standard security advice of granting least privilege to objects generated by the project.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: developer_tools_1.awsSolutionsCb4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CB5',
            info: 'The Codebuild project does not use images provided by the CodeBuild service or have a cdk_nag suppression rule explaining the need for a custom image.',
            explanation: 'Explaining differences/edits to Docker images helps operators better understand system dependencies.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: developer_tools_1.awsSolutionsCb5,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-C91',
            info: 'The Cloud9 instance does not use a no-ingress EC2 instance with AWS Systems Manager.',
            explanation: 'SSM adds an additional layer of protection as it allows operators to control access through IAM permissions and does not require opening inbound ports.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: developer_tools_1.awsSolutionsC91,
            node: node,
        });
    }
}
exports.AwsSolutionsChecks = AwsSolutionsChecks;
_a = JSII_RTTI_SYMBOL_1;
AwsSolutionsChecks[_a] = { fqn: "monocdk-nag.AwsSolutionsChecks", version: "0.1.35" };
//# sourceMappingURL=data:application/json;base64,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