"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CDN = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const aws_cloudfront_origins_1 = require("aws-cdk-lib/aws-cloudfront-origins");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const constructs_1 = require("constructs");
class CDN extends constructs_1.Construct {
    constructor(scope, id, { domainName, certificate, remixPublicPath, httpApi }) {
        super(scope, id);
        const domainNames = domainName ? [domainName] : undefined;
        const httpApiHost = (httpApi.url || '').split('/')[2];
        const staticBucket = new aws_s3_1.Bucket(this, 'static', {
            accessControl: aws_s3_1.BucketAccessControl.PRIVATE,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            publicReadAccess: false,
            versioned: true,
        });
        new aws_s3_deployment_1.BucketDeployment(this, 'staticDeployment', {
            sources: [aws_s3_deployment_1.Source.asset(remixPublicPath)],
            destinationBucket: staticBucket,
            memoryLimit: 2048,
        });
        const cachePolicy = new aws_cloudfront_1.CachePolicy(this, 'cachePolicy', {
            cookieBehavior: aws_cloudfront_1.CacheCookieBehavior.all(),
            defaultTtl: aws_cdk_lib_1.Duration.seconds(0),
            minTtl: aws_cdk_lib_1.Duration.seconds(0),
            maxTtl: aws_cdk_lib_1.Duration.days(10),
            queryStringBehavior: aws_cloudfront_1.CacheQueryStringBehavior.all(),
            enableAcceptEncodingGzip: true,
            enableAcceptEncodingBrotli: true,
        });
        const remixRequestPolicy = new aws_cloudfront_1.OriginRequestPolicy(this, 'originRequestPolicy', {
            cookieBehavior: aws_cloudfront_1.OriginRequestCookieBehavior.all(),
            headerBehavior: aws_cloudfront_1.OriginRequestHeaderBehavior.allowList('Accept', 'origin'),
            queryStringBehavior: aws_cloudfront_1.OriginRequestQueryStringBehavior.all(),
        });
        const s3Origin = new aws_cloudfront_origins_1.S3Origin(staticBucket);
        this.distribution = new aws_cloudfront_1.Distribution(this, 'distribution', {
            domainNames,
            certificate,
            defaultBehavior: {
                allowedMethods: aws_cloudfront_1.AllowedMethods.ALLOW_ALL,
                cachePolicy,
                origin: new aws_cloudfront_origins_1.HttpOrigin(httpApiHost),
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                originRequestPolicy: remixRequestPolicy,
                responseHeadersPolicy: aws_cloudfront_1.ResponseHeadersPolicy.SECURITY_HEADERS,
            },
            additionalBehaviors: {
                '/build/*': {
                    origin: s3Origin,
                    viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                },
                'favicon.ico': {
                    origin: s3Origin,
                    viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                },
            },
        });
    }
}
exports.CDN = CDN;
//# sourceMappingURL=data:application/json;base64,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