import json
import logging
import os
import re
from pathlib import Path
from typing import Dict, Text, List, Any, Union, Tuple

import convo.shared.data
from convo.shared.nlu.constants import TXT, KEY_INTENT_NAME
from convo.shared.nlu.training_data.message import Msg
from convo.shared.constants import (
    INTENT_MSG_PREFIX ,
    DOMAIN_DOCUMENTS_URL,
    LEGACY_DOCUMENTS_BASE_URL,
    DEFAULT_E2E_TEST_PATH ,
    STORIES_DOCUMENTS_URL,
)
from convo.shared.core.events import UserUttered
from convo.shared.nlu.interpreter import RegexInterpreter
from convo.shared.core.training_data.story_reader.story_reader import (
    storyReviewer,
    StoryAnalysedError,
)
from convo.shared.core.training_data.structures import StoryStage, formPrefix
import convo.shared.utils.io

log = logging.getLogger(__name__)


class MarkdownStoryReviewer(storyReviewer):
    """Class that reads the core training data in a Markdown format"""

    def readFromFile(self, filename: Union[Text, Path]) -> List[StoryStage]:
        """Given a md file reads the contained stories."""

        try:
            with open(
                filename, "r", encoding=convo.shared.utils.io.ENCODING_DEFAULT
            ) as f:
                statement = f.readlines()

            return self.processLines(statement)
        except ValueError as err:
            fileInfo = "Invalid story file format. Failed to parse '{}'".format(
                os.path.abspath(filename)
            )
            log.exception(fileInfo)
            if not err.args:
                err.args = ("",)
            err.args = err.args + (fileInfo,)
            raise

    def processLines(self, lines: List[Text]) -> List[StoryStage]:
        multilineComment = False

        for idx, statement in enumerate(lines):
            lineNum = idx + 1
            try:
                statement = self.replaceTemplateVariables(self.cleanUpLine(statement))
                if statement.strip() == "":
                    continue
                elif statement.startswith("<!--"):
                    multilineComment = True
                    continue
                elif multilineComment and statement.endswith("-->"):
                    multilineComment = False
                    continue
                elif multilineComment:
                    continue
                elif statement.startswith(">>"):
                    # reached a new rule block
                    ruleName = statement.lstrip(">> ")
                    self.newRulePart(ruleName, self.source_name)
                elif statement.startswith("#"):
                    # reached a new story block
                    fetch_name = statement[1:].strip("# ")
                    self.newStoryPart(fetch_name, self.source_name)
                elif statement.startswith(">"):
                    # reached a checkpoint
                    fetch_name, conditions = self.parseEventLine(statement[1:].strip())
                    self.addCheckpoint(fetch_name, conditions)
                elif re.match(fr"^[*\-]\s+{formPrefix}", statement):
                    log.debug(
                        "Skipping line {}, "
                        "because it was generated by "
                        "form action".format(statement)
                    )
                elif statement.startswith("-"):
                    # reached a slot, event, or executed action
                    event_name, parameters = self.parseEventLine(statement[1:])
                    self.addEvent(event_name, parameters)
                elif statement.startswith("*"):
                    # reached a user message
                    userMessages = [el.strip() for el in statement[1:].split(" OR ")]
                    if self.use_e2e:
                        self.addE2eMessages(userMessages, lineNum)
                    else:
                        self.addUserMessages(userMessages, lineNum)
                else:
                    # reached an unknown type of line
                    log.warning(
                        f"Skipping line {lineNum}. "
                        "No valid command found. "
                        f"Line Content: '{statement}'"
                    )
            except Exception as e:
                message = f"Error in line {lineNum}: {e}"
                log.error(message, exc_info=1)  # pytype: disable=wrong-arg-types
                raise ValueError(message) from e
        self.addCurrentStoriesToResult()
        return self.story_steps

    @staticmethod
    def parametersFromJsonString(s: Text, line: Text) -> Dict[Text, Any]:
        """Parse the passed string as json and create a parameter dict."""

        if s is None or not s.strip():
            # if there is no strings there are not going to be any parameters
            return {}

        try:
            parsedSlots = json.loads(s)
            if isinstance(parsedSlots, dict):
                return parsedSlots
            else:
                raise Exception(
                    "Parsed value isn't a json object "
                    "(instead parser found '{}')"
                    ".".format(type(parsedSlots))
                )
        except Exception as e:
            raise ValueError(
                "Invalid to parse arguments in line "
                "'{}'. Failed to decode parameters"
                "as a json object. Make sure the event"
                "name is followed by a proper json "
                "object. Error: {}".format(line, e)
            )

    def replaceTemplateVariables(self, line: Text) -> Text:
        def processMatch(matchobject):
            variableName = matchobject.group(1)
            if variableName in self.template_variables:
                return self.template_variables[variableName]
            else:
                raise ValueError(
                    "Unknown variable `{var}` "
                    "in template line '{line}'"
                    "".format(var=variableName, line=line)
                )

        templateRx = re.compile(r"`([^`]+)`")
        return templateRx.sub(processMatch, line)

    @staticmethod
    def cleanUpLine(line: Text) -> Text:
        """Removes comments and trailing spaces"""

        return re.sub(r"<!--.*?-->", "", line).strip()

    @staticmethod
    def parseEventLine(line: Text) -> Tuple[Text, Dict[Text, Text]]:
        """Tries to parse a single line as an event with arguments."""

        # the regex matches "slot{"a": 1}"
        n = re.search("^([^{]+)([{].+)?", line)
        if n is not None:
            eventName = n.group(1).strip()
            slotsStr = n.group(2)
            parameters = MarkdownStoryReviewer.parametersFromJsonString(
                slotsStr, line
            )
            return eventName, parameters
        else:
            convo.shared.utils.io.raising_warning(
                f"Failed to parse action line '{line}'. Ignoring this line.",
                docs=STORIES_DOCUMENTS_URL,
            )
            return "", {}

    def addUserMessages(self, messages: List[Text], line_num: int) -> None:
        if not self.current_step_builder:
            raise StoryAnalysedError(
                "User message '{}' at invalid location. "
                "Expected story start.".format(messages)
            )
        parsedMessages = [self.parseMessage(m, line_num) for m in messages]
        self.current_step_builder.addUserMessages(
            parsedMessages, self.is_used_for_conversion
        )

    def addE2eMessages(self, e2e_messages: List[Text], line_num: int) -> None:
        if not self.current_step_builder:
            raise StoryAnalysedError(
                "End-to-end message '{}' at invalid "
                "location. Expected story start."
                "".format(e2e_messages)
            )

        parsed_messages = []
        for m in e2e_messages:
            analysed = self.parseMessage(m, line_num)
            parsed_messages.append(analysed)
        self.current_step_builder.addUserMessages(parsed_messages)

    @staticmethod
    def parse_e2e_message(line: Text, is_used_for_conversion: bool = False) -> Msg:
        """Parses an md list item line based on the current section type.

        Matches expressions of the form `<intent>:<example>`. For the
        syntax of `<example>` see the Convo docs on NLU training data."""

        # Match three groups:
        # 1) Potential "form" annotation
        # 2) The correct intent
        # 3) Optional entities
        # 4) The message text
        formGroup = fr"({formPrefix}\s*)*"
        itemRegex = re.compile(r"\s*" + formGroup + r"([^{}]+?)({.*})*:\s*(.*)")
        compare = re.match(itemRegex, line)

        if not compare:
            raise ValueError(
                "Encountered invalid test story format for message "
                "`{}`. Please visit the documentation page on "
                "end-to-end testing at {}/user-guide/testing-your-assistant/"
                "#end-to-end-testing/".format(line, LEGACY_DOCUMENTS_BASE_URL)
            )
        from convo.shared.nlu.training_data import entities_parser

        intention = compare.group(2)
        msg = compare.group(4)
        eg = entities_parser.parsing_training_example(msg, intention)
        if is_used_for_conversion:
            # In case this is a simple conversion from Markdown we should copy over
            # the original text and not parse the entities
            eg.data[convo.shared.nlu.constants.TXT] = msg
            eg.data[convo.shared.nlu.constants.ENTITIES_NAME] = []

        # If the message starts with the `INTENT_MSG_PREFIX ` potential entities
        # are annotated in the json format (e.g. `/greet{"name": "Convo"})
        if msg.startswith(INTENT_MSG_PREFIX):
            analysed = RegexInterpreter().synch_parsing(msg)
            eg.data["entities"] = analysed["entities"]

        return eg

    def parseMessage(self, message: Text, line_num: int) -> UserUttered:

        if self.use_e2e:
            analysed = self.parse_e2e_message(message, self.is_used_for_conversion)
            txt = analysed.get("text")
            intention = {KEY_INTENT_NAME: analysed.get("intent")}
            entityList = analysed.get("entityList")
            parseData = {
                "text": txt,
                "intent": intention,
                "intent_ranking": [{KEY_INTENT_NAME: analysed.get("intent")}],
                "entityList": entityList,
            }
        else:
            parseData = RegexInterpreter().synch_parsing(message)
            txt = None
            intention = parseData.get("intent")

        change = UserUttered(txt, intention, parseData.get("entityList"), parseData)

        name_of_intent = change.intent.get(KEY_INTENT_NAME)

        if self.domain and name_of_intent not in self.domain.fetch_intents:
            convo.shared.utils.io.raising_warning(
                f"Found unknown intent '{name_of_intent}' on line {line_num}. "
                "Please, make sure that all convo_intents are "
                "listed in your domain yaml.",
                UserWarning,
                docs=DOMAIN_DOCUMENTS_URL,
            )
        return change

    @staticmethod
    def isStoriesFile(file_path: Union[Text, Path]) -> bool:
        """Check if file contains Core training data or rule data in Markdown format.

        Args:
            file_path: Path of the file to check.

        Returns:
            `True` in case the file is a Core Markdown training data or rule data file,
            `False` otherwise.
        """
        if not convo.shared.data.is_mark_down_file (
            file_path
        ) or convo.shared.data.nlu_file_check(file_path):
            return False

        try:
            with open(
                file_path,
                encoding=convo.shared.utils.io.ENCODING_DEFAULT,
                errors="surrogateescape",
            ) as lines:
                return any(
                    MarkdownStoryReviewer.containsStoryOrRulePattern(line)
                    for line in lines
                )
        except Exception as e:
            # catch-all because we might be loading files we are not expecting to load
            log.error(
                f"Tried to check if '{file_path}' is a story file, but failed to "
                f"read it. If this file contains story or rule data, you should "
                f"investigate this error, otherwise it is probably best to "
                f"move the file to a different location. Error: {e}"
            )
            return False

    @staticmethod
    def isTestStoriesFile(file_path: Union[Text, Path]) -> bool:
        """Checks if a file contains test stories.

        Args:
            file_path: Path of the file which should be checked.

        Returns:
            `True` if it's a file containing test stories, otherwise `False`.
        """
        if not convo.shared.data.is_mark_down_file (file_path):
            return False

        dirname = os.path.dirname(file_path)
        return (
            DEFAULT_E2E_TEST_PATH in dirname
            and convo.shared.data.story_file_check(file_path)
            and not convo.shared.data.nlu_file_check(file_path)
        )

    @staticmethod
    def containsStoryOrRulePattern(text: Text) -> bool:
        storyPattern = r".*##.+"
        rulePattern = r".*>>.+"

        return any(re.match(pattern, text) for pattern in [storyPattern, rulePattern])
