"""
.. module:: PyTorchMalConv
	:synopsis: MalConv CClassifierPytorch wrapper

.. moduleauthor:: Luca Demetrio <luca.demetrio@dibris.unige.it>

"""
import os
import sys

import numpy as np
import torch
import torchvision.transforms as transforms
from secml.array import CArray
from secml.ml.classifiers.pytorch.c_classifier_pytorch import CClassifierPyTorch
from secml.settings import SECML_PYTORCH_USE_CUDA

from secml_malware.models.basee2e import End2EndModel

use_cuda = torch.cuda.is_available() and SECML_PYTORCH_USE_CUDA


class CClassifierEnd2EndMalware(CClassifierPyTorch):

	def __init__(
			self,
			model: End2EndModel,
			epochs=100,
			batch_size=256,
			train_transform=None,
			preprocess=None,
			softmax_outputs=False,
			random_state=None,
			plus_version=False,
			input_shape=(1, 2 ** 20),
			verbose=0,
	):
		super(CClassifierEnd2EndMalware, self).__init__(
			model,
			loss="binary_crossentropy",
			epochs=epochs,
			batch_size=batch_size,
			preprocess=preprocess,
			input_shape=(1, model.max_input_size),
			softmax_outputs=softmax_outputs,
			random_state=random_state,
		)
		self.plus_version = plus_version
		self.verbose = verbose
		self.train_transform = (
			train_transform
			if train_transform is not None
			else transforms.Lambda(lambda p: p.reshape(input_shape[1]))
		)

	def _apply_constraints(self):
		# remove zeros from weights, allowing only biases to be lower than 0
		self._model.embedding_1.weight.data.clamp_(0)
		self._model.conv1d_1.weight.data.clamp_(0)
		self._model.conv1d_2.weight.data.clamp_(0)
		self._model.dense_1.weight.data.clamp_(0)
		self._model.dense_2.weight.data.clamp_(0)

	def gradient(self, x, w=None):
		"""Compute gradient at x by doing a forward and a backward pass.

		The gradient is pre-multiplied by w.

		"""
		return self._gradient_f(x)


	def gradient_f_x(self, x, **kwargs):
		"""Returns the gradient of the function on point x.
		
		Arguments:
			x {CArray} -- The point
		
		Raises:
			NotImplementedError: Model do not support gradient
		
		Returns:
			CArray -- the gradient computed on x
		"""
		if self.preprocess is not None:
			# Normalize data before compute the classifier gradient
			x_pre = self.preprocess.normalize(x)
		else:  # Data will not be preprocessed
			x_pre = x
		try:  # Get the derivative of decision_function
			grad_f = self._gradient_f(x_pre, **kwargs)
		except NotImplementedError:
			raise NotImplementedError(
				"{:} does not implement `gradient_f_x`".format(self.__class__.__name__)
			)
		return grad_f

	def _gradient_f(self, x, y=None, w=None, layer=None, sum_embedding=True):
		penalty_term = torch.zeros(1)
		penalty_term.requires_grad_()
		gradient = self.compute_embedding_gradient(x.tondarray(), penalty_term)
		if sum_embedding:
			gradient = torch.mean(gradient, dim=1)
		if gradient.is_cuda:
			gradient = gradient.cpu()
		return CArray(gradient)

	def load_pretrained_model(
			self, path=None
	):
		"""Load pretrained model
	
			Keyword Arguments:
				path {str} -- The path of the model, default is None, and it will load the internal default one
		"""
		root = os.path.dirname(
			os.path.dirname(os.path.abspath(sys.modules["secml_malware"].__file__))
		)
		self._model.load_simplified_model(
			os.path.join(root, "secml_malware/data/trained/pretrained_malconv.pth") if path is None else path)
		self._classes = np.array([0, 1])
		self._n_features = 2 ** 20

	def get_embedding_size(self):
		return self._model.embedding_size

	def get_input_max_length(self):
		return self._model.max_input_size

	def get_embedding_value(self):
		return self._model.embedding_value

	def get_is_shifting_values(self):
		return self._model.shift_values

	def embed(self, x, transpose=True):
		return self._model.embed(x, transpose=transpose)

	def compute_embedding_gradient(self, x, penalty_term):
		"""Compute the gradient w.r.t. embedding layer.
		
		Arguments:
			x {CArray} -- point whenre gradient will be computed
		
		Returns:
			cArray -- the gradient w.r.t. the embedding
		"""
		data = x
		if isinstance(x, CArray):
			data = x.tondarray()
		emb_x = self.embed(data)
		y = self.model.embedd_and_forward(emb_x)
		output = y + penalty_term
		if use_cuda:
			output = output.cuda()
		g = torch.autograd.grad(output, emb_x)[0]
		g = torch.transpose(g, 1, 2)[0]
		return g

	def embedding_predict(self, x):
		# if type(x) is CArray:
		# 	x = torch.Tensor(x.tondarray()).long()
		return self._model.embedd_and_forward(x)

	def _forward(self, x):
		confidence = super(CClassifierEnd2EndMalware, self)._forward(x)
		confidence[0, 0] = 1 - confidence[0, 1]
		return confidence
