import numpy
from secml.array import CArray
from secml_malware.attack.whitebox.c_discretized_bytes_evasion import CDiscreteBytesEvasion
from secml_malware.models import CClassifierEnd2EndMalware, End2EndModel

from secml_malware.utils.extend_pe import shift_section_by, shift_pe_header_by, create_int_list_from_x_adv


class CFormatExploitEvasion(CDiscreteBytesEvasion):

	def __init__(self,
			end2end_model : CClassifierEnd2EndMalware,
			preferable_extension_amount : int = 0x200,
			pe_header_extension : int = 0x200,
			iterations : int =100,
			is_debug: bool = False,
			random_init: bool = False,
			threshold: float = 0.5,
			penalty_regularizer: float = 0,
			chunk_hyper_parameter : int = None,
	):
		super(CFormatExploitEvasion, self).__init__(
			end2end_model=end2end_model,
			index_to_perturb=[],
			iterations=iterations,
			is_debug=is_debug,
			random_init=random_init,
			threshold=threshold,
			penalty_regularizer=penalty_regularizer,
			chunk_hyper_parameter=chunk_hyper_parameter
		)
		self.preferable_extension_amount = preferable_extension_amount
		self.pe_header_extension = pe_header_extension

	def _run(self, x0, y0, x_init=None):
		x_init, _ = self._craft_perturbed_c_array(x0)
		return super(CFormatExploitEvasion, self)._run(x0, y0, x_init)

	def _craft_perturbed_c_array(self, x0 : CArray):
		x_init, indexes_to_perturb = self._generate_list_adv_example(x0)
		self.indexes_to_perturb = indexes_to_perturb
		x_init = CArray(
			[
				End2EndModel.list_to_numpy(
					x_init,
					self.classifier.get_input_max_length(),
					self.classifier.get_embedding_value(),
					self.classifier.get_is_shifting_values())
			]
		)
		return x_init, indexes_to_perturb

	def _generate_list_adv_example(self, x0):
		x_init = create_int_list_from_x_adv(x0, self.classifier.get_embedding_value(), self.classifier.get_is_shifting_values())
		x_init, index_to_perturb_sections = shift_section_by(x_init, preferable_extension_amount=self.preferable_extension_amount)
		x_init, index_to_perturb_pe = shift_pe_header_by(x_init, preferable_extension_amount=self.pe_header_extension)
		indexes_to_perturb = index_to_perturb_pe + [i + len(index_to_perturb_pe) for i in
														 index_to_perturb_sections]
		return x_init, indexes_to_perturb

	def create_real_sample_from_adv(self, original_file_path : str, x_adv : CArray, new_file_path : str = None):
		with open(original_file_path, 'rb') as f:
			code = bytearray(f.read())
		original_x = CArray([numpy.frombuffer(code, dtype=numpy.uint8).astype(numpy.uint16)])
		if self.classifier.get_is_shifting_values():
			original_x += self.classifier.get_is_shifting_values()
		x_init, index_to_perturb = self._generate_list_adv_example(original_x)
		x_init = CArray([x_init]).astype(numpy.uint8)
		x_init[0, index_to_perturb] = x_adv[0, index_to_perturb] - self.classifier.get_is_shifting_values()
		x_real = x_init[0,:].tolist()[0]
		x_real_adv = b''.join([bytes([i]) for i in x_real])
		if new_file_path:
			with open(new_file_path, 'wb') as f:
				f.write(x_real_adv)
		return x_real_adv


