import json
import argparse


def _read_tasks(task_file):
    """Read in the tasks to a list
    Read in the json tasks file generated by the dispatcher
    and create a lists of tasks to be processed

    Args:
        task_file (str): Path the the task.json file

    Returns:
        list: List of tasks to be processed
    """
    tasks = []
    with open(task_file) as f:
        tasks = json.load(f)

    if not isinstance(tasks, (list, tuple)):
        tasks = [tasks]

    return tasks


###
# Shared Arguments
###
shared_parser = argparse.ArgumentParser(add_help=False)
shared_parser.add_argument('--local', action='store_true',
                           help="Run/save everything locally")

###
# Shared between Dispatch & Download
###
dispatch_download_parser = argparse.ArgumentParser(add_help=False)
dispatch_download_parser.add_argument('--standalone', action='store_true',
                                      help="Do not trigger the next action")

###
# Parser for subparsers
###
parser = argparse.ArgumentParser()
subparsers = parser.add_subparsers(dest='action')
subparsers.required = True

###
# Validate Config
###
parser_validate = subparsers.add_parser('validate',
                                        help="Validate the yaml/env vars")

###
# Test Source files
###
parser_test = subparsers.add_parser('create-test',
                                    help=("Move files into test directory and "
                                          "generate an extracted qa file"))
parser_test.add_argument('metadata',
                         help=("Path to a sources metadata file "
                               "to create a test for"))

###
# Dispatch Arguments
###
parser_dispatch = subparsers.add_parser('dispatch',
                                        help="Run the Dispatcher",
                                        parents=[shared_parser,
                                                 dispatch_download_parser])
parser_dispatch.add_argument('--dump-tasks', action='store_true',
                             help="Save the tasks as json")
parser_dispatch.add_argument('-l', '--limit', type=int,
                             help='Limit the amount of tasks dispatch')
parser_dispatch.add_argument('--tasks',
                             type=_read_tasks,
                             help=("Output file from dispatch's --dump-tasks"))
ratelimit_group = parser_dispatch.add_mutually_exclusive_group()
ratelimit_group.add_argument('--qps',
                             help='Number of tasks to dispatch a second')
ratelimit_group.add_argument('--period', type=float,
                             help='Dispatch all tasks in n hours')
###
# Download Arguments
###
parser_download = subparsers.add_parser('download',
                                        help="Run the Downloader",
                                        parents=[shared_parser,
                                                 dispatch_download_parser])
parser_download.add_argument('--tasks',
                             type=_read_tasks,
                             required=True,
                             help=("Output file from dispatch's --dump-tasks"))
###
# Extract Arguments
###
parser_extract = subparsers.add_parser('extract',
                                       help="Run the Extractor",
                                       parents=[shared_parser,
                                                ])
parser_extract.add_argument('source',
                            help="Local dir or source file to extract")
