"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Use an Amazon SQS queue as an event source for AWS Lambda.
 */
class SqsEventSource {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        this._eventSourceMappingId = undefined;
        if (this.props.batchSize !== undefined && (this.props.batchSize < 1 || this.props.batchSize > 10)) {
            throw new Error(`Maximum batch size must be between 1 and 10 inclusive (given ${this.props.batchSize})`);
        }
    }
    bind(target) {
        const eventSourceMapping = target.addEventSourceMapping(`SqsEventSource:${this.queue.node.uniqueId}`, {
            batchSize: this.props.batchSize,
            eventSourceArn: this.queue.queueArn,
        });
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.queue.grantConsumeMessages(target);
    }
    /**
     * The identifier for this EventSourceMapping
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('SqsEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.SqsEventSource = SqsEventSource;
//# sourceMappingURL=data:application/json;base64,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