from datetime import datetime
from typing import ClassVar, Optional

from pydantic import BaseModel


class Power(BaseModel):
    """Power utility"""

    MICROJOULES_TO_WATT_FACTOR: ClassVar[int] = 1000000
    WH_TO_KWH_FACTOR: ClassVar[int] = 1000
    SECONDS_TO_HOURS_FACTOR: ClassVar[int] = 3600

    @staticmethod
    def watts_to_watt_hours(
        watts: float, previous_energy_measurement_time: Optional[datetime] = None
    ) -> float:
        """
        Convert current watts to watt-hours W/h using the previous energy measurement.

        :param watts: the wattage in W
        :param previous_energy_measurement_time: the previous measurement time
        :return: watt-hours W/h
        """
        now = datetime.now()
        if previous_energy_measurement_time:
            time_difference_in_seconds = (
                now - previous_energy_measurement_time
            ).total_seconds()
        else:
            time_difference_in_seconds = 1
        return watts * (time_difference_in_seconds / Power.SECONDS_TO_HOURS_FACTOR)

    @staticmethod
    def co2g_from_watts_hour(watts_hour: float, co2g_per_kwh: float) -> float:
        """
        Calculate the CO2g generated using watt hours and the CO2g/kwh.

        :return: the CO2g generated by the energy consumption
        """
        return (watts_hour / Power.WH_TO_KWH_FACTOR) * co2g_per_kwh

    @staticmethod
    def watts_from_microjoules(microjoules: float) -> float:
        """
        Create watts from microjoules

        :param: microjoules
        :return watts
        """
        return microjoules / Power.MICROJOULES_TO_WATT_FACTOR
