# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/00_api.ipynb (unless otherwise specified).

__all__ = ["authorize", "predict"]

# Cell

from typing import *

# Internal Cell

import requests
import pandas as pd
import json
from .testing import get_test_dataframe
import os

# Internal Cell

servers_base_url = dict(
    production="https://api.nanobit.captn.ai",
    staging="https://api.staging.nanobit.captn.ai",
    local=os.environ.get("webservice_ip") or "http://captn_nanobit_server:6006",
)

# Cell


def authorize(
    username: str, password: str, *, server: str = "production"
) -> Dict[str, str]:
    """
    In order to use this API you first need to do an authorization.
    You need to set the username, password and server to which the requests will be sent.

    Args:
        username (str): Your username.
        password (str): Your password.
        server (str): Server is one of "local", "staging" or "production".

    Returns:
        Dict[str, str]: Returns an access token.
        Access token contains keys: access_token, token_type and url.
    """

    base_url = servers_base_url[server]
    oauth_data = dict(username=username, password=password)
    response = requests.post(base_url + "/token", oauth_data)
    token = response.json()
    token["url"] = base_url
    return token


# Internal Cell


def convert_df_to_dict(df: pd.DataFrame) -> dict:
    """
    Converts pandas dataframe into the dictionary.

    Args:
        df (pd.DataFrame): Previous campaign data.

    Returns:
        Dict[str, Any]: Returns the dictionary so that the data can be sent in the request.
    """
    default_dict = df.to_dict(orient="list")

    revenue_dict = {k: v for k, v in default_dict.items() if "revenue_" in k}

    default_dict["date"] = [d.isoformat() for d in default_dict["date"]]
    [default_dict.pop(key) for key in list(default_dict.keys()) if "revenue_" in key]
    default_dict["revenues"] = revenue_dict

    for c in [
        "network",
        "campaign",
        "campaign_id",
        "adgroup",
        "adgroup_id",
        "game_name",
        "platform",
    ]:
        if (df[c] != df[c].iloc[0]).any():
            raise ValueError(f"df[{c}] is not constant: {df[c]}")
        default_dict[c] = df[c].iloc[0]

    return default_dict


# Cell


def predict(df: pd.DataFrame, token: Dict[str, str]) -> pd.DataFrame:
    """
    Predicts future revenues of a campaign.
    To make a prediction you need to have a valid access token and previous campaign data.

    Args:
        df (pd.DataFrame): Previous campaign data.
        token (Dict[str,str]): Access token.

    Returns:
        pd.DataFrame: Returns pandas dataframe with future revenues.
    """
    data = convert_df_to_dict(df)
    data = json.dumps(data)  # type:ignore
    url = token["url"]

    token_type = token["token_type"]
    access_token = token["access_token"]
    headers = {"Authorization": f"{token_type} {access_token}"}

    response = requests.post(
        f"{url}/predict",
        data=data,
        headers=headers,
    )
    return pd.DataFrame.from_dict(response.json()).round(decimals=2)
