"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RelationalDBSchemaTransformer = exports.TemplateContext = exports.TableContext = void 0;
const graphql_1 = require("graphql");
const RelationalDBSchemaTransformerUtils_1 = require("./RelationalDBSchemaTransformerUtils");
const RelationalDBParsingException_1 = require("./RelationalDBParsingException");
const graphql_transformer_common_1 = require("graphql-transformer-common");
class TableContext {
    constructor(typeDefinition, createDefinition, updateDefinition, primaryKeyField, primaryKeyType, stringFieldList, intFieldList) {
        this.tableTypeDefinition = typeDefinition;
        this.tableKeyField = primaryKeyField;
        this.createTypeDefinition = createDefinition;
        this.updateTypeDefinition = updateDefinition;
        this.tableKeyFieldType = primaryKeyType;
        this.stringFieldList = stringFieldList;
        this.intFieldList = intFieldList;
    }
}
exports.TableContext = TableContext;
class TemplateContext {
    constructor(schemaDoc, typePrimaryKeyMap, stringFieldMap, intFieldMap, typePrimaryKeyTypeMap) {
        this.schemaDoc = schemaDoc;
        this.typePrimaryKeyMap = typePrimaryKeyMap;
        this.stringFieldMap = stringFieldMap;
        this.intFieldMap = intFieldMap;
        this.typePrimaryKeyTypeMap = typePrimaryKeyTypeMap;
    }
}
exports.TemplateContext = TemplateContext;
class RelationalDBSchemaTransformer {
    constructor(dbReader, database) {
        this.introspectDatabaseSchema = async () => {
            let tableNames = null;
            try {
                tableNames = await this.dbReader.listTables();
            }
            catch (err) {
                throw new RelationalDBParsingException_1.RelationalDBParsingException(`Failed to list tables in ${this.database}`, err.stack);
            }
            let typeContexts = new Array();
            let types = new Array();
            let pkeyMap = new Map();
            let pkeyTypeMap = new Map();
            let stringFieldMap = new Map();
            let intFieldMap = new Map();
            for (const tableName of tableNames) {
                let type = null;
                try {
                    type = await this.dbReader.describeTable(tableName);
                }
                catch (err) {
                    throw new RelationalDBParsingException_1.RelationalDBParsingException(`Failed to describe table ${tableName}`, err.stack);
                }
                if (type.tableKeyField) {
                    typeContexts.push(type);
                    types.push(type.createTypeDefinition);
                    types.push(type.tableTypeDefinition);
                    types.push(type.updateTypeDefinition);
                    stringFieldMap.set(tableName, type.stringFieldList);
                    intFieldMap.set(tableName, type.intFieldList);
                    pkeyMap.set(tableName, type.tableKeyField);
                    pkeyTypeMap.set(tableName, type.tableKeyFieldType);
                }
                else {
                    console.warn(`Skipping table ${type.tableTypeDefinition.name.value} because it does not have a single PRIMARY KEY.`);
                }
            }
            types.push(this.getMutations(typeContexts));
            types.push(this.getQueries(typeContexts));
            types.push(this.getSubscriptions(typeContexts));
            types.push(this.getSchemaType());
            let context = this.dbReader.hydrateTemplateContext(new TemplateContext({ kind: graphql_1.Kind.DOCUMENT, definitions: types }, pkeyMap, stringFieldMap, intFieldMap, pkeyTypeMap));
            return context;
        };
        this.dbReader = dbReader;
        this.database = database;
    }
    getSchemaType() {
        return {
            kind: graphql_1.Kind.SCHEMA_DEFINITION,
            directives: [],
            operationTypes: [
                RelationalDBSchemaTransformerUtils_1.getOperationTypeDefinition('query', RelationalDBSchemaTransformerUtils_1.getNamedType('Query')),
                RelationalDBSchemaTransformerUtils_1.getOperationTypeDefinition('mutation', RelationalDBSchemaTransformerUtils_1.getNamedType('Mutation')),
                RelationalDBSchemaTransformerUtils_1.getOperationTypeDefinition('subscription', RelationalDBSchemaTransformerUtils_1.getNamedType('Subscription')),
            ],
        };
    }
    getMutations(types) {
        const fields = [];
        for (const typeContext of types) {
            const type = typeContext.tableTypeDefinition;
            const formattedTypeValue = graphql_transformer_common_1.toUpper(type.name.value);
            fields.push(RelationalDBSchemaTransformerUtils_1.getOperationFieldDefinition(`delete${formattedTypeValue}`, [RelationalDBSchemaTransformerUtils_1.getInputValueDefinition(RelationalDBSchemaTransformerUtils_1.getNonNullType(RelationalDBSchemaTransformerUtils_1.getNamedType(typeContext.tableKeyFieldType)), typeContext.tableKeyField)], RelationalDBSchemaTransformerUtils_1.getNamedType(`${type.name.value}`), null));
            fields.push(RelationalDBSchemaTransformerUtils_1.getOperationFieldDefinition(`create${formattedTypeValue}`, [RelationalDBSchemaTransformerUtils_1.getInputValueDefinition(RelationalDBSchemaTransformerUtils_1.getNonNullType(RelationalDBSchemaTransformerUtils_1.getNamedType(`Create${formattedTypeValue}Input`)), `create${formattedTypeValue}Input`)], RelationalDBSchemaTransformerUtils_1.getNamedType(`${type.name.value}`), null));
            fields.push(RelationalDBSchemaTransformerUtils_1.getOperationFieldDefinition(`update${formattedTypeValue}`, [RelationalDBSchemaTransformerUtils_1.getInputValueDefinition(RelationalDBSchemaTransformerUtils_1.getNonNullType(RelationalDBSchemaTransformerUtils_1.getNamedType(`Update${formattedTypeValue}Input`)), `update${formattedTypeValue}Input`)], RelationalDBSchemaTransformerUtils_1.getNamedType(`${type.name.value}`), null));
        }
        return RelationalDBSchemaTransformerUtils_1.getTypeDefinition(fields, 'Mutation');
    }
    getSubscriptions(types) {
        const fields = [];
        for (const typeContext of types) {
            const type = typeContext.tableTypeDefinition;
            const formattedTypeValue = graphql_transformer_common_1.toUpper(type.name.value);
            fields.push(RelationalDBSchemaTransformerUtils_1.getOperationFieldDefinition(`onCreate${formattedTypeValue}`, [], RelationalDBSchemaTransformerUtils_1.getNamedType(`${type.name.value}`), [
                RelationalDBSchemaTransformerUtils_1.getDirectiveNode(`create${formattedTypeValue}`),
            ]));
        }
        return RelationalDBSchemaTransformerUtils_1.getTypeDefinition(fields, 'Subscription');
    }
    getQueries(types) {
        const fields = [];
        for (const typeContext of types) {
            const type = typeContext.tableTypeDefinition;
            const formattedTypeValue = graphql_transformer_common_1.toUpper(type.name.value);
            fields.push(RelationalDBSchemaTransformerUtils_1.getOperationFieldDefinition(`get${formattedTypeValue}`, [RelationalDBSchemaTransformerUtils_1.getInputValueDefinition(RelationalDBSchemaTransformerUtils_1.getNonNullType(RelationalDBSchemaTransformerUtils_1.getNamedType(typeContext.tableKeyFieldType)), typeContext.tableKeyField)], RelationalDBSchemaTransformerUtils_1.getNamedType(`${type.name.value}`), null));
            const nameListType = RelationalDBSchemaTransformerUtils_1.getSingletonListTypeNode(type.name.value);
            fields.push(RelationalDBSchemaTransformerUtils_1.getOperationFieldDefinition(`list${formattedTypeValue}s`, [], nameListType, null));
        }
        return RelationalDBSchemaTransformerUtils_1.getTypeDefinition(fields, 'Query');
    }
    getConnectionType(tableName) {
        return RelationalDBSchemaTransformerUtils_1.getTypeDefinition([RelationalDBSchemaTransformerUtils_1.getFieldDefinition('items', RelationalDBSchemaTransformerUtils_1.getNamedType(`[${tableName}]`)), RelationalDBSchemaTransformerUtils_1.getFieldDefinition('nextToken', RelationalDBSchemaTransformerUtils_1.getNamedType('String'))], `${tableName}Connection`);
    }
}
exports.RelationalDBSchemaTransformer = RelationalDBSchemaTransformer;
//# sourceMappingURL=RelationalDBSchemaTransformer.js.map