"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.filter = exports.isEmpty = exports.ConfigurationMapper = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const toml_1 = require("@iarna/toml");
const param_case_1 = require("param-case");
const pascal_case_1 = require("pascal-case");
const snake_case_1 = require("snake-case");
class ConfigurationMapper {
    constructor(props) {
        this.props = props;
    }
    static withDefaults(props) {
        const { globalConfiguration, runnersConfiguration } = props;
        return new ConfigurationMapper({
            globalConfiguration: {
                concurrent: 10,
                checkInterval: 0,
                logFormat: "runner",
                logLevel: "info",
                ...globalConfiguration,
            },
            runnersConfiguration: runnersConfiguration.map((item) => {
                return {
                    url: "https://gitlab.com",
                    limit: 10,
                    outputLimit: 52428800,
                    executor: "docker+machine",
                    environment: [
                        "DOCKER_DRIVER=overlay2",
                        "DOCKER_TLS_CERTDIR=/certs",
                        'DOCKER_AUTH_CONFIG={"credsStore": "ecr-login"}',
                    ],
                    ...item,
                    docker: {
                        tlsVerify: false,
                        image: "docker:19.03.5",
                        privileged: true,
                        capAdd: ["CAP_SYS_ADMIN"],
                        waitForServicesTimeout: 300,
                        disableCache: false,
                        volumes: ["/certs/client", "/cache"],
                        shmSize: 0,
                        ...item.docker,
                    },
                    machine: {
                        idleCount: 0,
                        idleTime: 300,
                        maxBuilds: 20,
                        machineDriver: "amazonec2",
                        machineName: "gitlab-runner-%s",
                        ...item.machine,
                        machineOptions: {
                            requestSpotInstance: true,
                            spotPrice: 0.03,
                            metadataToken: "required",
                            metadataTokenResponseHopLimit: 2,
                            ...item.machine?.machineOptions,
                        },
                        autoscaling: item.machine?.autoscaling
                            ? item.machine?.autoscaling
                            : [
                                {
                                    periods: ["* * 7-22 * * mon-fri *"],
                                    idleCount: 1,
                                    idleTime: 1800,
                                    timezone: "Etc/UTC",
                                },
                            ],
                    },
                    cache: item.cache?.s3 && Object.keys(item.cache?.s3).length
                        ? {
                            type: "s3",
                            shared: true,
                            ...item.cache,
                        }
                        : undefined,
                };
            }),
        });
    }
    static fromProps(props) {
        return new ConfigurationMapper(props);
    }
    toToml() {
        return toml_1.stringify(this._toJsonMap());
    }
    /**
     * @internal
     */
    _toJsonMap() {
        const { globalConfiguration, runnersConfiguration } = this.props;
        const result = toJsonMap(globalConfiguration, snake_case_1.snakeCase);
        result.runners = [];
        for (const config of runnersConfiguration) {
            const runner = toJsonMap(config, snake_case_1.snakeCase);
            // Fix naming convention inconsistencies
            runner["tls-ca-file"] = runner.tls_ca_file;
            delete runner.tls_ca_file;
            runner["tls-cert-file"] = runner.tls_cert_file;
            delete runner.tls_ca_file;
            runner["tls-key-file"] = runner.tls_key_file;
            delete runner.tls_ca_file;
            if (config.docker) {
                runner.docker = toJsonMap(config.docker, snake_case_1.snakeCase);
            }
            runner.machine = toJsonMap(config.machine, pascal_case_1.pascalCase);
            if (config.machine?.machineOptions) {
                runner.machine.MachineOptions = this._mapMachineOptions(config.machine?.machineOptions);
            }
            if (config.machine?.autoscaling?.length) {
                runner.machine.autoscaling = config.machine?.autoscaling.map((autoscaling) => toJsonMap(autoscaling, pascal_case_1.pascalCase));
            }
            delete runner.machine.Autoscaling;
            if (config?.cache?.s3 && Object.keys(config.cache.s3).length) {
                runner.cache = toJsonMap(config.cache, pascal_case_1.pascalCase);
                runner.cache.s3 = toJsonMap(config.cache.s3, pascal_case_1.pascalCase);
                delete runner.cache.s3;
            }
            else {
                delete runner.cache;
            }
            result.runners.push(runner);
        }
        return filter(result, (item) => !isEmpty(item));
    }
    _mapMachineOptions(machineOptions) {
        return toProperties(machineOptions, (key) => {
            switch (true) {
                case key == "engineInstallUrl":
                    return param_case_1.paramCase(key);
                default:
                    return `amazonec2-${param_case_1.paramCase(key)}`;
            }
        });
    }
}
exports.ConfigurationMapper = ConfigurationMapper;
_a = JSII_RTTI_SYMBOL_1;
ConfigurationMapper[_a] = { fqn: "@pepperize/cdk-autoscaling-gitlab-runner.ConfigurationMapper", version: "0.2.425" };
/**
 * Transforms configuration objects to JsonMap. Pass an inflector function to transform object keys.
 *
 * @param configuration
 * @param inflector A function to transform the object key
 */
function toJsonMap(configuration, inflector) {
    const result = {};
    for (const key in configuration) {
        const value = configuration[key];
        if (value === undefined) {
            continue;
        }
        result[inflector(key)] = value;
    }
    return result;
}
/**
 * Transforms configuration objects to a property array. Pass an inflector function to transform object keys.
 *
 * @param configuration
 * @param inflector A function to transform the object key
 * @example
 * // returns ["foo=bar"]
 * toProperties({foo: "bar", (s) => s});
 */
function toProperties(configuration, inflector) {
    const result = [];
    for (const key in configuration) {
        const value = configuration[key];
        if (value === undefined) {
            continue;
        }
        result.push(`${inflector(key)}=${value}`);
    }
    return result;
}
function isEmpty(subject) {
    if (Array.isArray(subject)) {
        return !subject.length;
    }
    if (typeof subject === "object" && !(subject instanceof Date)) {
        return !Object.keys(subject).length;
    }
    if (subject === undefined) {
        return true;
    }
    if (subject === null) {
        return true;
    }
    return false;
}
exports.isEmpty = isEmpty;
function filter(subject, predicate) {
    if (Array.isArray(subject)) {
        const result = [];
        subject.forEach((element) => {
            const filtered = filter(element, predicate);
            if (predicate.call(subject, filtered)) {
                result.push(filtered);
            }
        });
        return result;
    }
    if (typeof subject === "object" && !(subject instanceof Date)) {
        const result = {};
        for (const key in subject) {
            const value = subject[key];
            const filtered = filter(value, predicate);
            if (predicate.call(subject, filtered)) {
                result[key] = filtered;
            }
        }
        return result;
    }
    return subject;
}
exports.filter = filter;
//# sourceMappingURL=data:application/json;base64,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