"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_rds_1 = require("monocdk/aws-rds");
const nag_rules_1 = require("../../nag-rules");
/**
 * RDS Aurora MySQL serverless clusters have audit, error, general, and slowquery Log Exports enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_rds_1.CfnDBCluster) {
        const engine = nag_rules_1.NagRules.resolveIfPrimitive(node, node.engine).toLowerCase();
        const engineMode = nag_rules_1.NagRules.resolveIfPrimitive(node, node.engineMode).toLowerCase();
        if (engineMode != undefined &&
            engineMode.toLowerCase() == 'serverless' &&
            (engine.toLowerCase() == 'aurora' ||
                engine.toLowerCase() == 'aurora-mysql')) {
            if (node.enableCloudwatchLogsExports == undefined) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
            const needed = ['audit', 'error', 'general', 'slowquery'];
            const exports = node.enableCloudwatchLogsExports.map((i) => {
                return i.toLowerCase();
            });
            const compliant = needed.every((i) => exports.includes(i));
            if (compliant !== true) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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