"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cloudwatch_1 = require("monocdk/aws-cloudwatch");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * CloudWatch alarms have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_cloudwatch_1.CfnAlarm) {
        const actionsEnabled = monocdk_1.Stack.of(node).resolve(node.actionsEnabled);
        if (actionsEnabled === false) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        // Actions can be an array with a token that then resolves to an empty array or undefined
        const alarmActions = monocdk_1.Stack.of(node).resolve(node.alarmActions);
        const insufficientDataActions = monocdk_1.Stack.of(node).resolve(node.insufficientDataActions);
        const okActions = monocdk_1.Stack.of(node).resolve(node.okActions);
        const totalAlarmActions = alarmActions ? alarmActions.length : 0;
        const totalInsufficientDataActions = insufficientDataActions
            ? insufficientDataActions.length
            : 0;
        const totalOkActions = okActions ? okActions.length : 0;
        const totalActions = totalAlarmActions + totalInsufficientDataActions + totalOkActions;
        if (totalActions == 0) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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