"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NIST80053R4Checks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../nag-pack");
const apigw_1 = require("../rules/apigw");
const autoscaling_1 = require("../rules/autoscaling");
const cloudtrail_1 = require("../rules/cloudtrail");
const cloudwatch_1 = require("../rules/cloudwatch");
const codebuild_1 = require("../rules/codebuild");
const dms_1 = require("../rules/dms");
const dynamodb_1 = require("../rules/dynamodb");
const ec2_1 = require("../rules/ec2");
const efs_1 = require("../rules/efs");
const elasticache_1 = require("../rules/elasticache");
const elb_1 = require("../rules/elb");
const emr_1 = require("../rules/emr");
const iam_1 = require("../rules/iam");
const kms_1 = require("../rules/kms");
const lambda_1 = require("../rules/lambda");
const opensearch_1 = require("../rules/opensearch");
const rds_1 = require("../rules/rds");
const redshift_1 = require("../rules/redshift");
const s3_1 = require("../rules/s3");
const sagemaker_1 = require("../rules/sagemaker");
const sns_1 = require("../rules/sns");
const vpc_1 = require("../rules/vpc");
const waf_1 = require("../rules/waf");
/**
 * Check for NIST 800-53 rev 4 compliance.
 *
 * Based on the NIST 800-53 rev 4 AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-nist-800-53_rev_4.html
 *
 * @stability stable
 */
class NIST80053R4Checks extends nag_pack_1.NagPack {
    /**
     * @stability stable
     */
    constructor(props) {
        super(props);
        this.packName = 'NIST.800.53.R4';
    }
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        if (node instanceof monocdk_1.CfnResource) {
            this.checkAPIGW(node);
            this.checkAutoScaling(node);
            this.checkCloudTrail(node);
            this.checkCloudWatch(node);
            this.checkCodeBuild(node);
            this.checkDMS(node);
            this.checkDynamoDB(node);
            this.checkEC2(node);
            this.checkEFS(node);
            this.checkElastiCache(node);
            this.checkELB(node);
            this.checkEMR(node);
            this.checkIAM(node);
            this.checkKMS(node);
            this.checkLambda(node);
            this.checkOpenSearch(node);
            this.checkRDS(node);
            this.checkRedshift(node);
            this.checkS3(node);
            this.checkSageMaker(node);
            this.checkSNS(node);
            this.checkVPC(node);
            this.checkWAF(node);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node) {
        this.applyRule({
            info: 'The API Gateway stage does not have caching enabled and encrypted for all methods - (Control IDs: SC-13, SC-28).',
            explanation: "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWCacheEnabledAndEncrypted,
            node: node,
        });
        this.applyRule({
            info: 'The API Gateway stage does not have execution logging enabled for all methods - (Control IDs: AU-2(a)(d), AU-3, AU-12(a)(c)).',
            explanation: 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWExecutionLoggingEnabled,
            node: node,
        });
    }
    /**
     * Check Auto Scaling Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAutoScaling(node) {
        this.applyRule({
            info: 'The Auto Scaling group utilizes a load balancer and does not have an ELB health check configured - (Control IDs: SC-5).',
            explanation: 'The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupELBHealthCheckRequired,
            node: node,
        });
    }
    /**
     * Check CloudTrail Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node) {
        this.applyRule({
            info: 'The trail does not have CloudWatch logs enabled - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-6(1)(3), AU-7(1), AU-12(a)(c), CA-7(a)(b), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c)).',
            explanation: 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailCloudWatchLogsEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The trail does not have a KMS key ID or have encryption enabled - (Control ID: AU-9).',
            explanation: 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The trail does not have log file validation enabled - (Control ID: AC-6).',
            explanation: 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailLogFileValidationEnabled,
            node: node,
        });
    }
    /**
     * Check CloudWatch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudWatch(node) {
        this.applyRule({
            info: 'The CloudWatch alarm does not have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled - (Control IDs: AC-2(4), AU-6(1)(3), AU-7(1), CA-7(a)(b), IR-4(1), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c)).',
            explanation: 'Amazon CloudWatch alarms alert when a metric breaches the threshold for a specified number of evaluation periods. The alarm performs one or more actions based on the value of the metric or expression relative to a threshold over a number of time periods.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.CloudWatchAlarmAction,
            node: node,
        });
        this.applyRule({
            info: 'The CloudWatch Log Group is not encrypted with an AWS KMS key - (Control IDs: AU-9, SC-13, SC-28).',
            explanation: 'To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.CloudWatchLogGroupEncrypted,
            node: node,
        });
        this.applyRule({
            info: 'The CloudWatch Log Group does not have an explicit retention period configured - (Control IDs: AU-11, SI-12).',
            explanation: 'Ensure a minimum duration of event log data is retained for your log groups to help with troubleshooting and forensics investigations. The lack of available past event log data makes it difficult to reconstruct and identify potentially malicious events.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.CloudWatchLogGroupRetentionPeriod,
            node: node,
        });
    }
    /**
     * Check CodeBuild Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCodeBuild(node) {
        this.applyRule({
            info: 'The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables - (Control IDs: AC-6, IA-5(7), SA-3(a)).',
            explanation: 'Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: codebuild_1.CodeBuildProjectEnvVarAwsCred,
            node: node,
        });
        this.applyRule({
            info: 'The CodeBuild project which utilizes either a GitHub or BitBucket source repository does not utilize OAUTH - (Control ID: SA-3(a)).',
            explanation: 'OAUTH is the most secure method of authenticating your CodeBuild application. Use OAuth instead of personal access tokens or a user name and password to grant authorization for accessing GitHub or Bitbucket repositories.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: codebuild_1.CodeBuildProjectSourceRepoUrl,
            node: node,
        });
    }
    /**
     * Check DMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDMS(node) {
        this.applyRule({
            info: 'The DMS replication instance is public - (Control IDs: AC-3).',
            explanation: 'DMS replication instances can contain sensitive information and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dms_1.DMSReplicationNotPublic,
            node: node,
        });
    }
    /**
     * Check DynamoDB Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDynamoDB(node) {
        this.applyRule({
            info: "The provisioned capacity DynamoDB table does not have Auto Scaling enabled on it's indexes - (Control IDs: CP-10, SC-5).",
            explanation: 'Amazon DynamoDB auto scaling uses the AWS Application Auto Scaling service to adjust provisioned throughput capacity that automatically responds to actual traffic patterns. This enables a table or a global secondary index to increase its provisioned read/write capacity to handle sudden increases in traffic, without throttling.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DynamoDBAutoScalingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The DynamoDB table is not in an AWS Backup plan - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'To help with data back-up processes, ensure your Amazon DynamoDB tables are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DynamoDBInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: 'The DynamoDB table does not have Point-in-time Recovery enabled - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'The recovery maintains continuous backups of your table for the last 35 days.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DynamoDBPITREnabled,
            node: node,
        });
    }
    /**
     * Check EC2 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEC2(node) {
        this.applyRule({
            info: 'The EBS volume is not in an AWS Backup plan - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'To help with data back-up processes, ensure your Amazon Elastic Block Store (Amazon EBS) volumes are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2EBSInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance does not have detailed monitoring enabled - (Control IDs: CA-7(a)(b), SI-4(2), SI-4(a)(b)(c)).',
            explanation: 'Detailed monitoring provides additional monitoring information (such as 1-minute period graphs) on the AWS console.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceDetailedMonitoringEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance is not within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstancesInVPC,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance is associated with a public IP address - (Control IDs: AC-4, AC-6, AC-21(b), SC-7, SC-7(3)). ',
            explanation: 'Amazon EC2 instances can contain sensitive information and access control is required for such resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceNoPublicIp,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance allows unrestricted inbound IPv4 TCP traffic on common ports (20, 21, 3389, 3306, 4333) - (Control IDs: AC-4, CM-2, SC-7, SC-7(3)).',
            explanation: 'Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2RestrictedCommonPorts,
            node: node,
        });
        this.applyRule({
            info: 'The Security Group allows unrestricted SSH access - (Control IDs: AC-4, SC-7, SC-7(3)).',
            explanation: 'Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2RestrictedSSH,
            node: node,
        });
    }
    /**
     * Check EFS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEFS(node) {
        this.applyRule({
            info: 'The EFS is not in an AWS Backup plan - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'To help with data back-up processes, ensure your Amazon Elastic File System (Amazon EFS) file systems are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: efs_1.EFSInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: 'The EFS does not have encryption at rest enabled - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: efs_1.EFSEncrypted,
            node: node,
        });
    }
    /**
     * Check ElastiCache Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkElastiCache(node) {
        this.applyRule({
            info: 'The ElastiCache Redis cluster does not retain automatic backups for at least 15 days - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'Automatic backups can help guard against data loss. If a failure occurs, you can create a new cluster, which restores your data from the most recent backup.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterAutomaticBackup,
            node: node,
        });
    }
    /**
     * Check Elastic Load Balancer Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkELB(node) {
        this.applyRule({
            info: 'The ALB does not have invalid HTTP header dropping enabled - (Control ID: AC-17(2)).',
            explanation: 'Ensure that your Application Load Balancers (ALB) are configured to drop http headers. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ALBHttpDropInvalidHeaderEnabled,
            node: node,
        });
        this.applyRule({
            info: "The ALB's HTTP listeners are not configured to redirect to HTTPS - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13, SC-23).",
            explanation: 'To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ALBHttpToHttpsRedirection,
            node: node,
        });
        this.applyRule({
            info: 'The ALB is not associated with AWS WAFv2 web ACL - (Control IDs: SC-7, SI-4(a)(b)(c)).',
            explanation: 'A WAF helps to protect your web applications or APIs against common web exploits. These web exploits may affect availability, compromise security, or consume excessive resources within your environment.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ALBWAFEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBACMCertificateRequired,
            node: node,
        });
        this.applyRule({
            info: 'The CLB does not balance traffic between at least 2 Availability Zones - (Control IDs: SC-5, CP-10).',
            explanation: 'The cross-zone load balancing reduces the need to maintain equivalent numbers of instances in each enabled availability zone.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBCrossZoneLoadBalancingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The ALB, NLB, or GLB does not have deletion protection enabled - (Control IDs: CM-2, CP-10).',
            explanation: 'Use this feature to prevent your load balancer from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The ELB does not have logging enabled - (Control ID: AU-2(a)(d)).',
            explanation: "Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client's IP address, latencies, request paths, and server responses.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The CLB does not restrict its listeners to only the SSL and HTTPS protocols - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-23).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBTlsHttpsListenersOnly,
            node: node,
        });
    }
    /**
     * Check EMR Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEMR(node) {
        this.applyRule({
            info: 'The EMR cluster does not have Kerberos enabled - (Control IDs: AC-2(j), AC-3, AC-5c, AC-6).',
            explanation: 'The access permissions and authorizations can be managed and incorporated with the principles of least privilege and separation of duties, by enabling Kerberos for Amazon EMR clusters.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRKerberosEnabled,
            node: node,
        });
    }
    /**
     * Check IAM Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkIAM(node) {
        this.applyRule({
            info: 'The IAM Group does not have at least one IAM User - (Control ID: AC-2(j)).',
            explanation: 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMGroupHasUsers,
            node: node,
        });
        this.applyRule({
            info: 'The IAM Group, User, or Role contains an inline policy - (Control ID: AC-6).',
            explanation: 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMNoInlinePolicy,
            node: node,
        });
        this.applyRule({
            info: 'The IAM policy grants admin access - (Control IDs: AC-2(1), AC-2(j), AC-3, AC-6).',
            explanation: 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, restricting policies from containing "Effect": "Allow" with "Action": "*" over "Resource": "*". Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMPolicyNoStatementsWithAdminAccess,
            node: node,
        });
        this.applyRule({
            info: 'The IAM user does not belong to any group(s) - (Control IDs: AC-2(1), AC-2(j), AC-3, AC-6).',
            explanation: 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations, by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMUserGroupMembership,
            node: node,
        });
        this.applyRule({
            info: 'The IAM policy is attached at the user level - (Control IDs: AC-2(j), AC-3, AC-5c, AC-6).',
            explanation: 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMUserNoPolicies,
            node: node,
        });
    }
    /**
     * Check KMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkKMS(node) {
        this.applyRule({
            info: 'The KMS Symmetric key does not have automatic key rotation enabled - (Control ID: SC-12).',
            explanation: 'Enable key rotation to ensure that keys are rotated once they have reached the end of their crypto period.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: kms_1.KMSBackingKeyRotationEnabled,
            node: node,
        });
    }
    /**
     * Check Lambda Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkLambda(node) {
        this.applyRule({
            info: 'The Lambda function is not VPC enabled - (Control IDs: AC-4, SC-7, SC-7(3)).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: lambda_1.LambdaInsideVPC,
            node: node,
        });
    }
    /**
     * Check OpenSearch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkOpenSearch(node) {
        this.applyRule({
            info: 'The OpenSearch Service domain does not have encryption at rest enabled - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchEncryptedAtRest,
            node: node,
        });
        this.applyRule({
            info: 'The OpenSearch Service domain is not running within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).',
            explanation: 'VPCs help secure your AWS resources and provide an extra layer of protection.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchInVPCOnly,
            node: node,
        });
        this.applyRule({
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled - (Control IDs: SC-7, SC-8, SC-8(1)).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchNodeToNodeEncryption,
            node: node,
        });
    }
    /**
     * Check Amazon RDS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRDS(node) {
        this.applyRule({
            info: 'The RDS DB instance does not enhanced monitoring enabled - (Control ID: CA-7(a)(b)).',
            explanation: 'Enable enhanced monitoring to help monitor Amazon RDS availability. This provides detailed visibility into the health of your Amazon RDS database instances.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSEnhancedMonitoringEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB instance is not in an AWS Backup plan - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'To help with data back-up processes, ensure your Amazon Relational Database Service (Amazon RDS) instances are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB instance does not have backups enabled - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'The backup feature of Amazon RDS creates backups of your databases and transaction logs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceBackupEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB instance or Aurora DB cluster does not have deletion protection enabled - (Control ID: SC-5).',
            explanation: 'Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS DB instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB instance allows public access - (Control IDs: AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstancePublicAccess,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB instance does not have all CloudWatch log types exported - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c)).',
            explanation: 'To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The non-Aurora RDS DB instance does not have multi-AZ support enabled - (Control IDs: CP-10, SC-5, SC-36).',
            explanation: 'Multi-AZ support in Amazon Relational Database Service (Amazon RDS) provides enhanced availability and durability for database instances. When you provision a Multi-AZ database instance, Amazon RDS automatically creates a primary database instance, and synchronously replicates the data to a standby instance in a different Availability Zone. In case of an infrastructure failure, Amazon RDS performs an automatic failover to the standby so that you can resume database operations as soon as the failover is complete.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSMultiAZSupport,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB instance or Aurora DB cluster does not have storage encrypted - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist at rest in Amazon RDS DB instances and clusters, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSStorageEncrypted,
            node: node,
        });
    }
    /**
     * Check Redshift Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRedshift(node) {
        this.applyRule({
            info: 'The Redshift cluster does not have encryption or audit logging enabled - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c), SC-13).',
            explanation: 'To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterConfiguration,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster allows public access - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterPublicAccess,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster does not require TLS/SSL encryption - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13).',
            explanation: 'Ensure that your Amazon Redshift clusters require TLS/SSL encryption to connect to SQL clients. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftRequireTlsSSL,
            node: node,
        });
    }
    /**
     * Check Amazon S3 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkS3(node) {
        this.applyRule({
            info: 'The S3 Bucket does not have object lock enabled - (Control ID: SC-28).',
            explanation: 'Because sensitive data can exist at rest in S3 buckets, enforce object locks at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketDefaultLockEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not have server access logs enabled - (Control IDs: AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c)).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketPublicReadProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketPublicWriteProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not have replication enabled - (Control IDs: AU-9(2), CP-9(b), CP-10, SC-5, SC-36).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketReplicationEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not have default server-side encryption enabled - (Control IDs: AU-9(2), CP-9(b), CP-10, SC-5, SC-36).',
            explanation: 'Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketServerSideEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not require requests to use SSL - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13).',
            explanation: 'To help protect data in transit, ensure that your Amazon Simple Storage Service (Amazon S3) buckets require requests to use Secure Socket Layer (SSL). Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketSSLRequestsOnly,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not have versioning enabled - (Control IDs: CP-10, SI-12).',
            explanation: 'Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketVersioningEnabled,
            node: node,
        });
    }
    /**
     * Check SageMaker Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSageMaker(node) {
        this.applyRule({
            info: 'The SageMaker endpoint is not encrypted with a KMS key - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerEndpointConfigurationKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            info: 'The SageMaker notebook is not encrypted with a KMS key - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookInstanceKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            info: 'The SageMaker notebook does not disable direct internet access - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookNoDirectInternetAccess,
            node: node,
        });
    }
    /**
     * Check Amazon SNS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSNS(node) {
        this.applyRule({
            info: 'The SNS topic does not have KMS encryption enabled - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sns_1.SNSEncryptedKMS,
            node: node,
        });
    }
    /**
     * Check VPC Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkVPC(node) {
        this.applyRule({
            info: "The VPC's default security group allows inbound or outbound traffic - (Control IDs: AC-4, SC-7, SC-7(3)).",
            explanation: 'When creating a VPC through CloudFormation, the default security group will always be open. Therefore it is important to always close the default security group after stack creation whenever a VPC is created. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: vpc_1.VPCDefaultSecurityGroupClosed,
            node: node,
        });
        this.applyRule({
            info: 'The VPC does not have an associated Flow Log - (Control IDs: AU-2(a)(d), AU-3, AU-12(a)(c)).',
            explanation: 'The VPC flow logs provide detailed records for information about the IP traffic going to and from network interfaces in your Amazon Virtual Private Cloud (Amazon VPC). By default, the flow log record includes values for the different components of the IP flow, including the source, destination, and protocol.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCFlowLogsEnabled,
            node: node,
        });
    }
    /**
     * Check WAF Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkWAF(node) {
        this.applyRule({
            info: 'The WAFv2 web ACL does not have logging enabled - (Control IDs: AU-2(a)(d), AU-3, AU-12(a)(c), SC-7, SI-4(a)(b)(c)).',
            explanation: 'AWS WAF logging provides detailed information about the traffic that is analyzed by your web ACL. The logs record the time that AWS WAF received the request from your AWS resource, information about the request, and an action for the rule that each request matched.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: waf_1.WAFv2LoggingEnabled,
            node: node,
        });
    }
}
exports.NIST80053R4Checks = NIST80053R4Checks;
_a = JSII_RTTI_SYMBOL_1;
NIST80053R4Checks[_a] = { fqn: "monocdk-nag.NIST80053R4Checks", version: "1.4.17" };
//# sourceMappingURL=data:application/json;base64,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