"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NagPack = exports.NagMessageLevel = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const fs_1 = require("fs");
const path_1 = require("path");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("./nag-rules");
const VALIDATION_FAILURE_ID = 'CdkNagValidationFailure';
const SUPPRESSION_ID = 'CdkNagSuppression';
/**
 * The level of the message that the rule applies.
 *
 * @stability stable
 */
var NagMessageLevel;
(function (NagMessageLevel) {
    NagMessageLevel["WARN"] = "Warning";
    NagMessageLevel["ERROR"] = "Error";
})(NagMessageLevel = exports.NagMessageLevel || (exports.NagMessageLevel = {}));
/**
 * Base class for all rule packs.
 *
 * @stability stable
 */
class NagPack {
    /**
     * @stability stable
     */
    constructor(props) {
        /**
         * @stability stable
         */
        this.reportStacks = new Array();
        /**
         * @stability stable
         */
        this.packName = '';
        this.verbose =
            props == undefined || props.verbose == undefined ? false : props.verbose;
        this.logIgnores =
            props == undefined || props.logIgnores == undefined
                ? false
                : props.logIgnores;
        this.reports =
            props == undefined || props.reports == undefined ? true : props.reports;
    }
    /**
     * @stability stable
     */
    get readPackName() {
        return this.packName;
    }
    /**
     * @stability stable
     */
    get readReportStacks() {
        return this.reportStacks;
    }
    /**
     * Create a rule to be used in the NagPack.
     *
     * @param params The.
     * @stability stable
     * @IApplyRule interface with rule details.
     */
    applyRule(params) {
        var _b, _c, _d;
        if (this.packName === '') {
            throw Error('The NagPack does not have a pack name, therefore the rule could not be applied. Set a packName in the NagPack constructor.');
        }
        let resourceIgnores = (_b = params.node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
        resourceIgnores = resourceIgnores ? resourceIgnores : [];
        let stackIgnores = (_d = (_c = monocdk_1.Stack.of(params.node).templateOptions.metadata) === null || _c === void 0 ? void 0 : _c.cdk_nag) === null || _d === void 0 ? void 0 : _d.rules_to_suppress;
        stackIgnores = stackIgnores ? stackIgnores : [];
        const allIgnores = resourceIgnores.concat(stackIgnores);
        const ruleSuffix = params.ruleSuffixOverride
            ? params.ruleSuffixOverride
            : params.rule.name;
        const ruleId = `${this.packName}-${ruleSuffix}`;
        try {
            const ruleCompliance = params.rule(params.node);
            if (this.reports === true &&
                ruleCompliance === nag_rules_1.NagRuleCompliance.COMPLIANT) {
                this.writeToStackComplianceReport(params, ruleId, ruleCompliance);
            }
            else if (ruleCompliance === nag_rules_1.NagRuleCompliance.NON_COMPLIANT) {
                const reason = this.ignoreRule(allIgnores, ruleId);
                if (this.reports === true) {
                    this.writeToStackComplianceReport(params, ruleId, ruleCompliance, reason);
                }
                if (reason) {
                    if (this.logIgnores === true) {
                        const message = this.createMessage(SUPPRESSION_ID, `${ruleId} was triggered but suppressed.`, `Provided reason: "${reason}"`);
                        monocdk_1.Annotations.of(params.node).addInfo(message);
                    }
                }
                else {
                    const message = this.createMessage(ruleId, params.info, params.explanation);
                    if (params.level == NagMessageLevel.ERROR) {
                        monocdk_1.Annotations.of(params.node).addError(message);
                    }
                    else if (params.level == NagMessageLevel.WARN) {
                        monocdk_1.Annotations.of(params.node).addWarning(message);
                    }
                }
            }
        }
        catch (error) {
            const reason = this.ignoreRule(allIgnores, VALIDATION_FAILURE_ID);
            if (this.reports === true) {
                this.writeToStackComplianceReport(params, ruleId, 'UNKNOWN', reason);
            }
            if (reason) {
                if (this.logIgnores === true) {
                    const message = this.createMessage(SUPPRESSION_ID, `${VALIDATION_FAILURE_ID} was triggered but suppressed.`, reason);
                    monocdk_1.Annotations.of(params.node).addInfo(message);
                }
            }
            else {
                const information = `'${ruleId}' threw an error during validation. This is generally caused by a parameter referencing an intrinsic function. For more details enable verbose logging.'`;
                const message = this.createMessage(VALIDATION_FAILURE_ID, information, error.message);
                monocdk_1.Annotations.of(params.node).addWarning(message);
            }
        }
    }
    /**
     * Check whether a specific rule should be ignored.
     *
     * @param ignores The ignores listed in cdk-nag metadata.
     * @param ruleId The id of the rule to ignore.
     * @returns The reason the rule was ignored, or an empty string.
     * @stability stable
     */
    ignoreRule(ignores, ruleId) {
        for (let ignore of ignores) {
            if (ignore.id &&
                ignore.reason &&
                JSON.stringify(ignore.reason).length >= 10) {
                if (ignore.id == ruleId) {
                    return ignore.reason;
                }
            }
            else {
                throw Error(`Improperly formatted cdk_nag rule suppression detected: ${JSON.stringify(ignore)}. See https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.`);
            }
        }
        return '';
    }
    /**
     * The message to output to the console when a rule is triggered.
     *
     * @param ruleId The id of the rule.
     * @param info Why the rule was triggered.
     * @param explanation Why the rule exists.
     * @returns The formatted message string.
     * @stability stable
     */
    createMessage(ruleId, info, explanation) {
        let message = `${ruleId}: ${info}`;
        return this.verbose ? `${message} ${explanation}\n` : `${message}\n`;
    }
    /**
     * Write a line to the rule packs compliance report for the resource's Stack.
     *
     * @param params The.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     * @stability stable
     * @IApplyRule interface with rule details.
     */
    writeToStackComplianceReport(params, ruleId, compliance, explanation = '') {
        var _b;
        const line = this.createComplianceReportLine(params, ruleId, compliance, explanation);
        let outDir = (_b = monocdk_1.App.of(params.node)) === null || _b === void 0 ? void 0 : _b.outdir;
        const fileName = `${this.packName}-${params.node.stack.stackName}-NagReport.csv`;
        const filePath = path_1.join(outDir ? outDir : '', fileName);
        if (!this.reportStacks.includes(fileName)) {
            this.reportStacks.push(fileName);
            fs_1.writeFileSync(filePath, 'Rule ID,Resource ID,Compliance,Exception Reason,Rule Level,Rule Info\n');
        }
        fs_1.appendFileSync(filePath, line);
    }
    /**
     * Helper function to create a line for the compliance report.
     *
     * @param params The.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     * @stability stable
     * @IApplyRule interface with rule details.
     */
    createComplianceReportLine(params, ruleId, compliance, explanation = '') {
        //| Rule ID | Resource ID | Compliance | Exception Reason | Rule Level | Rule Info
        const line = Array();
        line.push(ruleId);
        line.push(params.node.node.path);
        if ((compliance === nag_rules_1.NagRuleCompliance.NON_COMPLIANT ||
            compliance === 'UNKNOWN') &&
            explanation !== '') {
            line.push('Suppressed');
            line.push(explanation);
        }
        else {
            line.push(compliance);
            line.push('N/A');
        }
        line.push(params.level);
        line.push(params.info);
        return line.map((i) => '"' + i.replace(/"/g, '""') + '"').join(',') + '\n';
    }
}
exports.NagPack = NagPack;
_a = JSII_RTTI_SYMBOL_1;
NagPack[_a] = { fqn: "monocdk-nag.NagPack", version: "1.4.17" };
//# sourceMappingURL=data:application/json;base64,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