"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = exports.TargetDependencyProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const syncpack_options_1 = require("./syncpack-options");
const NX_MONOREPO_PLUGIN_PATH = ".nx/plugins/nx-monorepo-plugin.js";
/**
 * Supported enums for a TargetDependency.
 */
var TargetDependencyProject;
(function (TargetDependencyProject) {
    /**
     * Only rely on the package where the target is called.
     *
     * This is usually done for test like targets where you only want to run unit
     * tests on the target packages without testing all dependent packages.
     */
    TargetDependencyProject["SELF"] = "self";
    /**
     * Target relies on executing the target against all dependencies first.
     *
     * This is usually done for build like targets where you want to build all
     * dependant projects first.
     */
    TargetDependencyProject["DEPENDENCIES"] = "dependencies";
})(TargetDependencyProject = exports.TargetDependencyProject || (exports.TargetDependencyProject = {}));
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            github: options.github ?? false,
            package: options.package ?? false,
            prettier: options.prettier ?? true,
            projenrcTs: true,
            release: options.release ?? false,
            jest: options.jest ?? false,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "mainline",
            sampleCode: false,
        });
        this.targetOverrides = {};
        this.nxConfig = options.nxConfig;
        this.workspaceConfig = options.workspaceConfig;
        this.implicitDependencies = this.nxConfig?.implicitDependencies || {};
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // No need to compile or test a monorepo root package.
        this.compileTask.reset();
        this.testTask.reset();
        this.addDevDeps("@nrwl/cli", "@nrwl/workspace");
        this.addDeps("aws-cdk-lib", "constructs", "cdk-nag"); // Needed as this can be bundled in aws-prototyping-sdk
        if (options.monorepoUpgradeDeps !== false) {
            this.addDevDeps("npm-check-updates", "syncpack");
            const upgradeDepsTask = this.addTask(options.monorepoUpgradeDepsOptions?.taskName || "upgrade-deps");
            upgradeDepsTask.exec("npx npm-check-updates --deep --rejectVersion 0.0.0 -u");
            upgradeDepsTask.exec("npx syncpack fix-mismatches");
            upgradeDepsTask.exec(`${this.package.packageManager} install`);
            upgradeDepsTask.exec("npx projen");
            new projen_1.JsonFile(this, ".syncpackrc.json", {
                obj: options.monorepoUpgradeDepsOptions?.syncpackConfig || syncpack_options_1.DEFAULT_CONFIG,
                readonly: true,
            });
        }
        options.nxConfig?.nxCloudReadOnlyAccessToken &&
            this.addDevDeps("@nrwl/nx-cloud");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(this.nxConfig?.nxIgnore || []));
        new projen_1.TextFile(this, NX_MONOREPO_PLUGIN_PATH, {
            readonly: true,
            lines: fs.readFileSync(getPluginPath()).toString("utf-8").split("\n"),
        });
        new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                plugins: [`./${NX_MONOREPO_PLUGIN_PATH}`],
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: options.nxConfig?.nxCloudReadOnlyAccessToken
                            ? "@nrwl/nx-cloud"
                            : "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: ["build", "test"],
                            accessToken: options.nxConfig?.nxCloudReadOnlyAccessToken,
                        },
                    },
                },
                implicitDependencies: this.implicitDependencies,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(this.nxConfig?.targetDependencies || {}),
                },
                affected: {
                    defaultBase: "mainline",
                },
            },
        });
    }
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent, dependee) {
        if (this.implicitDependencies[dependent.name]) {
            this.implicitDependencies[dependent.name].push(dependee.name);
        }
        else {
            this.implicitDependencies[dependent.name] = [dependee.name];
        }
    }
    /**
     * Allow project specific target overrides.
     */
    overrideProjectTargets(project, targets) {
        const _package = project.tryFindObjectFile("package.json");
        _package?.addOverride("nx", {
            targets: targets,
        });
        !_package && (this.targetOverrides[project.outdir] = targets);
    }
    // Remove this hack once subProjects is made public in Projen
    get subProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects.sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.updateWorkspace();
        this.synthesizeNonNodePackageJson();
        super.synth();
    }
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    validateSubProjects() {
        this.subProjects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (isNodeProject(subProject) &&
                subProject.package.packageManager !== this.package.packageManager) {
                throw new Error(`${subProject.name} packageManager does not match the monorepo packageManager: ${this.package.packageManager}.`);
            }
        });
    }
    /**
     * For non-node projects, a package.json is required in order to be discovered by NX.
     */
    synthesizeNonNodePackageJson() {
        this.subProjects
            .filter((subProject) => !isNodeProject(subProject))
            .filter((subProject) => !subProject.tryFindFile("package.json"))
            .forEach((subProject) => {
            // generate a package.json if not found
            const manifest = {};
            (manifest.nx = this.targetOverrides[subProject.outdir]
                ? { targets: this.targetOverrides[subProject.outdir] }
                : undefined),
                (manifest.name = subProject.name);
            manifest.private = true;
            manifest.__pdk__ = true;
            manifest.scripts = subProject.tasks.all.reduce((p, c) => ({
                [c.name]: `npx projen ${c.name}`,
                ...p,
            }), {});
            manifest.version = "0.0.0";
            new projen_1.JsonFile(subProject, "package.json", {
                obj: manifest,
                readonly: true,
            });
        });
    }
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    updateWorkspace() {
        // Add workspaces for each subproject
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                readonly: true,
                obj: {
                    packages: this.subProjects.map((subProject) => path.relative(this.outdir, subProject.outdir)),
                },
            });
        }
        else {
            this.package.addField("workspaces", {
                packages: this.subProjects.map((subProject) => path.relative(this.outdir, subProject.outdir)),
                nohoist: this.workspaceConfig?.noHoist,
            });
        }
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject", version: "0.9.3" };
/**
 * Determines if the passed in project is of type NodeProject.
 *
 * @param project Project instance.
 * @returns true if the project instance is of type NodeProject.
 */
function isNodeProject(project) {
    return project instanceof javascript_1.NodeProject || project.package;
}
function getPluginPath() {
    return path.join(__dirname, "plugin/nx-monorepo-plugin.js");
}
//# sourceMappingURL=data:application/json;base64,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