# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['osxmetadata', 'osxmetadata.attribute_data']

package_data = \
{'': ['*']}

install_requires = \
['bitstring>=3.1.9,<4.0.0',
 'click>=8.0,<9.0',
 'py-applescript>=1.0.3,<2.0.0',
 'pyobjc-core>=9.0.0,<10.0.0',
 'pyobjc-framework-AVFoundation>=9.0.0,<10.0.0',
 'pyobjc-framework-AppleScriptKit>=9.0.0,<10.0.0',
 'pyobjc-framework-AppleScriptObjC>=9.0.0,<10.0.0',
 'pyobjc-framework-CoreMedia>=9.0.0,<10.0.0',
 'pyobjc-framework-CoreServices>=9.0.0,<10.0.0',
 'pyobjc-framework-Quartz>=9.0.0,<10.0.0',
 'pyobjc-framework-ScriptingBridge>=9.0.0,<10.0.0',
 'twine>=4.0.1,<5.0.0',
 'wheel>=0.37.1,<0.38.0',
 'xattr>=0.9.9,<0.10.0']

entry_points = \
{'console_scripts': ['osxmetadata = osxmetadata.__main__:cli']}

setup_kwargs = {
    'name': 'osxmetadata',
    'version': '1.2.0',
    'description': 'Read and write meta data, such as tags/keywords, Finder comments, etc. on MacOS files',
    'long_description': '# osxmetadata\n\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/python/black)\n[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)\n[![All Contributors](https://img.shields.io/badge/all_contributors-4-orange.svg?style=flat)](#contributors-)\n\n## What is osxmetadata?\n\nosxmetadata provides a simple interface to access various metadata about MacOS / OS X files.  Currently supported metadata attributes include tags/keywords, Finder comments, authors, etc.\n\n## Why osxmetadata?\n\nApple provides rich support for file metadata through the [MDItem](https://developer.apple.com/documentation/coreservices/file_metadata/mditem) class and the [NSURL getResourceValue:forKey:error:](https://developer.apple.com/documentation/foundation/nsurl/1408874-getresourcevalue?language=objc) method. However, Apple does not provide a way to easily set much of the metadata. For example, while there is a documented [MDItem MDItemCopyAttribute](https://developer.apple.com/documentation/coreservices/1427080-mditemcopyattribute?language=objc) to copy metadata attributes such as [kMDItemAuthors](https://developer.apple.com/documentation/coreservices/kmditemauthors?language=objc), Apple does not provide a public interface to set this data.  Other data, such as Finder comments, can only be set through sending AppleScript commands to the Finder and others, like Finder tags can be retrieved but cannot be set through the public API.\n\nosxmetadata provides a unified interface to get and set most of the metadata available on your Mac from python.  It uses a combination of documented and undocumented APIs to access the metadata.  It also provides a simple interface to set Finder tags and Finder comments.\n\nMacOS provides some tools to view these various metadata attributes.  For example, `mdls` lists the MDItem Spotlight metadata associated with a file but doesn\'t let you edit the data. osxmetadata makes it easy to to both view and manipulate the macOS metadata attributes, either programmatically or through an included `osxmetadata` command line tool.\n\n## Supported operating systems\n\nOnly works on MacOS.  Requires Python 3.8+. Tested on macOS 10.15.7 (Catalina); should work on all versions of macOS 10.15 and later.\n\n## Installation instructions\n\n### Installation using pipx\n\nIf you aren\'t familiar with installing python applications, I recommend you install `osxmetadata` with [pipx](https://github.com/pipxproject/pipx). If you use `pipx`, you will not need to create a virtual environment as `pipx` takes care of this. The easiest way to do this on a Mac is to use [homebrew](https://brew.sh/):\n\n- Open `Terminal` (search for `Terminal` in Spotlight or look in `Applications/Utilities`)\n- Install `homebrew` according to instructions at [https://brew.sh/](https://brew.sh/)\n- Type the following into Terminal: `brew install pipx`\n- Then type this: `pipx install osxmetadata`\n- Now you should be able to run `osxmetadata` by typing: `osxmetadata`\n\nOnce you\'ve installed osxmetadata with pipx, to upgrade to the latest version:\n\n    pipx upgrade osxmetadata\n\n### Installation using pip\n\nYou can also install directly from [pypi](https://pypi.org/project/osxmetadata/):\n\n    pip install osxmetadata\n\nOnce you\'ve installed osxmetadata with pip, to upgrade to the latest version:\n\n    pip install --upgrade osxmetadata\n\n### Installation from git repository\n\nOSXMetaData uses setuptools, thus simply run:\n\n    git clone https://github.com/RhetTbull/osxmetadata.git\n    cd osxmetadata \n    pip install poetry\n    poetry install\n\nI recommend you create a [virtual environment](https://docs.python.org/3/tutorial/venv.html) before installing osxmetadata.\n\n## Using the API\n\n```pycon\n>>> import datetime\n>>> import pathlib\n>>> from osxmetadata import *\n>>> pathlib.Path("test_file.txt").touch()\n>>> md = OSXMetaData("test_file.txt")\n>>> md.set(kMDItemAuthors, ["Jane Smith", "John Doe"])\n>>> md.get(kMDItemAuthors)\n[\'Jane Smith\', \'John Doe\']\n>>> md.kMDItemFinderComment = "This is my comment"\n>>> md.kMDItemFinderComment\n\'This is my comment\'\n>>> md.tags = [Tag("Test", FINDER_COLOR_NONE), Tag("ToDo", FINDER_COLOR_RED)]\n>>> md.tags\n[Tag(name=\'Test\', color=0), Tag(name=\'ToDo\', color=6)]\n>>> md[kMDItemDueDate] = datetime.datetime(2022,10,1)\n>>> md[kMDItemDueDate]\ndatetime.datetime(2022, 10, 1, 0, 0)\n>>>\n```\n\nSomewhat contrary to the [Zen of Python](https://peps.python.org/pep-0020/), osxmetadata provides more than one way to access metadata attributes.  You can get and set metadata attributes using the `get()`/`set()` getter/setter methods, using the attribute name as a dictionary key on the OSXMetaData object, or using the attribute name as an attribute on the `OSXMetaData()` object.  For example, the following are all equivalent:\n\n- `OSXMetaData.get(attribute)` - get the value of the metadata attribute `attribute`\n- `OSXMetaData[attribute]` - get the value of the metadata attribute `attribute`\n- `OSXMetaData.attribute` - get the value of the metadata attribute `attribute`\n\nAs are the following:\n\n- `OSXMetaData.set(attribute, value)` - set the value of the metadata attribute `attribute` to `value`\n- `OSXMetaData[attribute] = value` - set the value of the metadata attribute `attribute` to `value`\n- `OSXMetaData.attribute = value` - set the value of the metadata attribute `attribute` to `value`\n\nThis allows you to use osxmetadata in accordance with your own code style preferences.\n\nSupported attribute names include all attributes defined for [MDItem](https://developer.apple.com/documentation/coreservices/file_metadata/mditem) and all resource keys defined for [NSURL](https://developer.apple.com/documentation/foundation/nsurl?language=objc). Additionally, the metadata constants defined in the [MDImporter](https://developer.apple.com/documentation/coreservices/file_metadata/mdimporter?language=objc) are supported as well as the following additional attributes:\n\n- `_kMDItemUserTags` - list of Finder tags\n- `kMDItemDownloadedDate` - list of datetime objects for when the file was downloaded\n\nAdditionally, osxmetadata defines a "shortcut name" attribute for each MDItem attribute that can be used as a shortcut `OSXMetaData` class attribute.  The shortcut name is the lowercase value of text following `kMDItem` for each attribute. For example, `kMDItemAuthors` has a short name of `authors` so you can set the authors like this:\n\n```pycon\n>>> from osxmetadata import *\n>>> md = OSXMetaData("test_file.txt")\n>>> md.authors = ["Jane Smith", "John Doe"]\n>>> md.authors\n[\'Jane Smith\', \'John Doe\']\n>>>\n```\n\nand `kMDItemDueDate` would have a short name of `duedate`:\n\n```pycon\n>>> from osxmetadata import *\n>>> import datetime\n>>> md = OSXMetaData("test_file.txt")\n>>> md.duedate = datetime.datetime(2022, 10, 1)\n>>> md.duedate\ndatetime.datetime(2022, 10, 1, 0, 0)\n>>>\n```\n\nThe names of all supported attributes are available in the `osxmetadata.ALL_ATTRIBUTES` set:\n\n```pycon\n>>> from osxmetadata import ALL_ATTRIBUTES\n>>> "kMDItemDueDate" in ALL_ATTRIBUTES\nTrue\n>>> "NSURLTagNamesKey" in ALL_ATTRIBUTES\nTrue\n>>> "findercomment" in ALL_ATTRIBUTES\nTrue\n>>>\n```\n\nThe class attributes are handled dynamically which, unfortunately, means that IDEs like PyCharm and Visual Studio Code cannot provide tab-completion for them.\n\n## Finder Tags\n\nUnlike other attributes, which are mapped to native Python types appropriate for the source Objective C type, Finder tags (`_kMDItemUserTags` or `tags`) have two components: a name (str) and a color ID (unsigned int in range 0 to 7) representing a color tag in the Finder.  Reading tags returns a list of `Tag` namedtuples and setting tags requires a list of `Tag` namedtuples.  \n\n```pycon\n>>> from osxmetadata import *\n>>> md = OSXMetaData("test_file.txt")\n>>> md.tags = [Tag("Test", FINDER_COLOR_NONE), Tag("ToDo", FINDER_COLOR_RED)]\n>>> md.tags\n[Tag(name=\'Test\', color=0), Tag(name=\'ToDo\', color=6)]\n>>> md.get("_kMDItemUserTags")\n[Tag(name=\'Test\', color=0), Tag(name=\'ToDo\', color=6)]\n>>> \n```\n\nTag names (but not colors) can also be accessed through the [NSURLTagNamesKey](https://developer.apple.com/documentation/foundation/nsurltagnameskey) resource key and the label color ID is accessible through `NSURLLabelNumberKey`; the localized label color name is accessible through `NSURLLocalizedLabelKey` though these latter two resource keys only return a single color whereas a file may have more than one color tag. For most purposes, I recommend using the `tags` attribute as it is more convenient and provides access to both the name and color ID of the tag.\n\n```pycon\n>>> from osxmetadata import *\n>>> md = OSXMetaData("test_file.txt")\n>>> md.tags = [Tag("Test", FINDER_COLOR_NONE), Tag("ToDo", FINDER_COLOR_RED)]\n>>> md.tags\n[Tag(name=\'Test\', color=0), Tag(name=\'ToDo\', color=6)]\n>>> md.NSURLTagNamesKey\n(\n    Test,\n    ToDo\n)\n>>> md.NSURLLabelNumberKey\n6\n>>> md.NSURLLocalizedLabelKey\n\'Red\'\n>>> md.NSURLTagNamesKey = ["NewTag"]\n>>> md.NSURLTagNamesKey\n(\n    NewTag\n)\n>>> md.tags\n[Tag(name=\'NewTag\', color=0)]\n>>>\n```\n\n### Create a Tag namedtuple\n\n`Tag(name, color)`\n\n- `name`: tag name (str)\n- `color`: color ID for Finder color label associated with tag (int)\n\nValid color constants (exported by osxmetadata):\n\n- `FINDER_COLOR_NONE` = 0\n- `FINDER_COLOR_GRAY` = 1\n- `FINDER_COLOR_GREEN` = 2\n- `FINDER_COLOR_PURPLE` = 3\n- `FINDER_COLOR_BLUE` = 4\n- `FINDER_COLOR_YELLOW` = 5\n- `FINDER_COLOR_RED` = 6\n- `FINDER_COLOR_ORANGE` = 7\n\n## Finder Comments\n\nFinder comments can be access via the `kMDItemFinderComment` attribute or the `findercomment` shortcut attribute. Apple provides a public API for getting Finder comments but does not provide a programmatic method for setting Finder comments and I have not been able to find a private API for doing so. osxmetadata works around this by send AppleScript events to the Finder to set the Finder comment. This means that setting Finder comments is slower than setting other attributes and may not work in all circumstances. The first time you set a Finder comment, your terminal app may need to prompt you to allow AppleScript to control the Finder. If you include osxmetadata in a standalone app, for example, one created with [py2app](https://py2app.readthedocs.io/en/latest/), you will need to include the `com.apple.security.automation.apple-events` entitlement and the `NSAppleEventsUsageDescription` key in your app\'s `Info.plist` file. See the [Apple Developer Documentation](https://developer.apple.com/documentation/bundleresources/information_property_list/nsappleeventsusagedescription?language=objc) for more information.\n\n```pycon\n>>> from osxmetadata import *\n>>> md = OSXMetaData("test_file.txt")\n>>> md.kMDItemFinderComment = "Hello World!"\n>>> md.kMDItemFinderComment\n\'Hello World!\'\n>>> md.findercomment\n\'Hello World!\'\n>>>\n```\n\n## Dates/Times\n\nMetadata attributes which return date/times such as `kMDItemDueDate` or `kMDItemDownloadedDate` return a `datetime.datetime` object.  The `datetime.datetime` object is timezone-naive (does not contain timezone) and returns the time in the local timezone.  Internally, Apple appears to store these as [CFDate](https://developer.apple.com/documentation/corefoundation/cfdate?language=objc) objects in the UTC timezone but when retrieved, they are returned in the local time.  You may pass a timezone-aware datetime object to set these attributes and it will be converted appropriately.\n\n```pycon\n>>> from osxmetadata import *\n>>> md = OSXMetaData("test_file.txt")\n>>> import datetime\n>>> md.kMDItemDueDate = datetime.datetime(2022, 10, 1)\n>>> md.kMDItemDueDate\ndatetime.datetime(2022, 10, 1, 0, 0)\n>>> md.kMDItemDownloadedDate = datetime.datetime(2022, 10, 1, tzinfo=datetime.timezone.utc)\n>>> \n```\n\n## Extended Attributes\n\nIn addition to `MDItem` and `NSURL` metadata attributes, osxmetadata can also read & write metadata saved in extended attributes. For many MDItem attributes, Apple stores the same data in both the MDItem and extended attribute (with name `com.apple.metadata:AttributeName`).  For example, the `kMDItemWhereFroms` attribute can be accessed both via MDItemCopyAttribute (exposed via osxmetadata\'s `get()` method) and via the `com.apple.metadata:kMDItemWhereFroms` extended attribute.  The extended attribute is a binary plist (BPLIST) and can be read using the `xattr` command line tool.  The `get_xattr()` method will return the value of the extended attribute and the `set_xattr()` method will set it.  Extended attributes can be removed with the `remove_xattr()` method.  `get_xattr()` provides for an optional callable argument, `decode`, which will be called on the returned value.  `set_xattr()` provides an optional callable argument `encode`. This is useful for encoding/decoding binary plist data.  For example, to decode the `com.apple.metadata:kMDItemWhereFroms` extended attribute, you can use the `plistlib.loads()` function:\n\n```pycon\n>>> from osxmetadata import *\n>>> import plistlib\n>>> from plistlib import FMT_BINARY\n>>> from functools import partial\n>>> md = OSXMetaData("test_file.txt")\n>>> md.kMDItemWhereFroms = ["apple.com"]\n>>> md.kMDItemWhereFroms\n[\'apple.com\']\n>>> decode = partial(plistlib.loads, fmt=FMT_BINARY)\n>>> encode = partial(plistlib.dumps, fmt=FMT_BINARY)\n>>> md.get_xattr("com.apple.metadata:kMDItemWhereFroms")\nb\'bplist00\\xa1\\x01Yapple.com\\x08\\n\\x00\\x00\\x00\\x00\\x00\\x00\\x01\\x01\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x02\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x14\'\n>>> md.get_xattr("com.apple.metadata:kMDItemWhereFroms", decode=decode)\n[\'apple.com\']\n>>> md.set_xattr("com.apple.metadata:kMDItemWhereFroms", ["google.com"], encode=encode)\n>>> md.get_xattr("com.apple.metadata:kMDItemWhereFroms", decode=decode)\n[\'google.com\']\n>>> md.remove_xattr("com.apple.metadata:kMDItemWhereFroms")\n>>>\n```\n\nFor most use cases, it is recommended you do not directly access the Apple metadata related extended attributes and instead use the getter/setter methods provided by osxmetadata.\n\n## Finder Info\n\nThe Finder keeps some legacy Finder info data about files in a bitstring stored in the `com.apple.FinderInfo` extended attribute. osxmetadata provides some attributes for working with this data.\n\n- `stationerypad`: True if the file is a stationery pad (file template) otherwise False; setting this attribute has the same effect as setting the `Stationery pad` checkbox in the Finder\'s `Get Info` window.\n- `findercolor`: The color of the file as an integer; setting this attribute has the same effect as applying a color label in the Finder\'s `Get Info` window. osxmetadata will set this attribute automatically when setting user tags; it is recommended you do not set this attribute directly.\n- `finderinfo`: The raw Finder info data as a bytes object; you should only manipulate this attribute if you know what you are doing.\n\n## Temporary Files\n\nSpotlight does not appear to index temporary files (those in `/tmp` or `/private/var/tmp`). Setting metadata using osxmetadata on temporary files in these locations will not fail but but it appears the metadata will not be indexed and a subsequent read will return the default value as if the metadata had not been written. This is not a limitation of osxmetadata but rather a limitation of Spotlight. If you need to set metadata on temporary files, you should use a different location.\n\n## Command Line Usage\n\nInstalls command line tool called `osxmetadata` which provides a simple interface to view/edit metadata supported by osxmetadata.\n\nIf you only care about the command line tool, I recommend installing with [pipx](https://github.com/pipxproject/pipx)\n\nThe command line tool can also be run via `python -m osxmetadata`.  Running it with no arguments or with --help option will print a help message:\n\n<!-- [[[cog\nimport cog\nfrom osxmetadata.__main__ import cli\nfrom click.testing import CliRunner\nrunner = CliRunner()\nresult = runner.invoke(cli, ["--help"])\nhelp = result.output.replace("Usage: cli", "Usage: osxmetadata")\ncog.out(\n    "```\\n{}\\n```".format(help)\n)\n]]] -->\n```\nUsage: osxmetadata [OPTIONS] FILE\n\n  Read/write metadata from file(s).\n\nOptions:\n  -v, --version                   Show the version and exit.\n  -w, --walk                      Walk directory tree, processing each file in\n                                  the tree.\n  -j, --json                      Print output in JSON format, for use with\n                                  --list and --get.\n  -X, --wipe                      Wipe all metadata attributes from FILE.\n  -s, --set ATTRIBUTE VALUE       Set ATTRIBUTE to VALUE.\n  -l, --list                      List all metadata attributes for FILE.\n  -c, --clear ATTRIBUTE           Remove attribute from FILE.\n  -a, --append ATTRIBUTE VALUE    Append VALUE to ATTRIBUTE; for multi-valued\n                                  attributes, appends only if VALUE is not\n                                  already present.\n  -g, --get ATTRIBUTE             Get value of ATTRIBUTE.\n  -r, --remove ATTRIBUTE VALUE    Remove VALUE from ATTRIBUTE; only applies to\n                                  multi-valued attributes.\n  -m, --mirror ATTRIBUTE1 ATTRIBUTE2\n                                  Mirror values between ATTRIBUTE1 and\n                                  ATTRIBUTE2 so that ATTRIBUTE1 = ATTRIBUTE2;\n                                  for multi-valued attributes, merges values;\n                                  for string attributes, sets ATTRIBUTE1 =\n                                  ATTRIBUTE2 overwriting any value in\n                                  ATTRIBUTE1.  For example: \'--mirror keywords\n                                  tags\' sets tags and keywords to same values.\n  -B, --backup                    Backup FILE attributes.  Backup file\n                                  \'.osxmetadata.json\' will be created in same\n                                  folder as FILE. Only backs up attributes known\n                                  to osxmetadata unless used with --all.\n  -R, --restore                   Restore FILE attributes from backup file.\n                                  Restore will look for backup file\n                                  \'.osxmetadata.json\' in same folder as FILE.\n                                  Only restores attributes known to osxmetadata\n                                  unless used with --all.\n  -V, --verbose                   Print verbose output.\n  -f, --copyfrom SOURCE_FILE      Copy attributes from file SOURCE_FILE (only\n                                  updates destination attributes that are not\n                                  null in SOURCE_FILE).\n  --files-only                    Do not apply metadata commands to directories\n                                  themselves, only files in a directory.\n  -p, --pattern PATTERN           Only process files matching PATTERN; only\n                                  applies to --walk. If specified, only files\n                                  matching PATTERN will be processed as each\n                                  directory is walked. May be used for than once\n                                  to specify multiple patterns. For example, tag\n                                  all *.pdf files in projectdir and subfolders\n                                  with tag \'project\': osxmetadata --append tags\n                                  \'project\' --walk projectdir/ --pattern \'*.pdf\'\n  --help                          Show this message and exit.\n\nValid attributes for ATTRIBUTE: Each attribute has a short name, a constant\nname, and a long constant name. Any of these may be used for ATTRIBUTE\n\nFor example: --set findercomment "Hello world"\nor:          --set kMDItemFinderComment "Hello world"\nor:          --set com.apple.metadata:kMDItemFinderComment "Hello world"\n\nAttributes that are strings can only take one value for --set; --append will\nappend to the existing value.  Attributes that are arrays can be set multiple\ntimes to add to the array: e.g. --set keywords \'foo\' --set keywords \'bar\' will\nset keywords to [\'foo\', \'bar\']\n\nOptions are executed in the following order regardless of order passed on the\ncommand line: restore, wipe, copyfrom, clear, set, append, remove, mirror, get,\nlist, backup.  --backup and --restore are mutually exclusive.  Other options may\nbe combined or chained together.\n\nFinder tags (tags attribute) contain both a name and an optional color. To\nspecify the color, append comma + color name (e.g. \'red\') after the tag name.\nFor example --set tags Foo,red. Valid color names are: gray, green, purple,\nblue, yellow, red, orange. If color is not specified but a tag of the same name\nhas already been assigned a color in the Finder, the same color will\nautomatically be assigned.\n\ncom.apple.FinderInfo (finderinfo) value is a key:value dictionary. To set\nfinderinfo, pass value in format key1:value1,key2:value2,etc. For example:\n\'osxmetadata --set finderinfo color:2 file.ext\'.\n\nShort Name                 Description\nacquisitionmake            kMDItemAcquisitionMake;\n                           com.apple.metadata:kMDItemAcquisitionMake; The\n                           manufacturer of the device used to aquire the\n                           document contents.; string\nacquisitionmodel           kMDItemAcquisitionModel;\n                           com.apple.metadata:kMDItemAcquisitionModel; The\n                           model of the device used to aquire the document\n                           contents. For example, 100, 200, 400, etc.; string\nalbum                      kMDItemAlbum; com.apple.metadata:kMDItemAlbum; The\n                           title for a collection of media. This is analagous\n                           to a record album, or photo album.; string\naltitude                   kMDItemAltitude;\n                           com.apple.metadata:kMDItemAltitude; The altitude of\n                           the item in meters above sea level, expressed using\n                           the WGS84 datum. Negative values lie below sea\n                           level.; string\naperture                   kMDItemAperture;\n                           com.apple.metadata:kMDItemAperture; The aperture\n                           setting used to acquire the document contents. This\n                           unit is the APEX value.; number\nappleloopdescriptors       kMDItemAppleLoopDescriptors;\n                           com.apple.metadata:kMDItemAppleLoopDescriptors;\n                           Specifies multiple pieces of descriptive\n                           information about a loop.; list of strings\nappleloopskeyfiltertype    kMDItemAppleLoopsKeyFilterType;\n                           com.apple.metadata:kMDItemAppleLoopsKeyFilterType;\n                           Specifies key filtering information about a loop.\n                           Loops are matched against projects that often in a\n                           major or minor key.; string\nappleloopsloopmode         kMDItemAppleLoopsLoopMode;\n                           com.apple.metadata:kMDItemAppleLoopsLoopMode;\n                           Specifies how a file should be played.; string\nappleloopsrootkey          kMDItemAppleLoopsRootKey;\n                           com.apple.metadata:kMDItemAppleLoopsRootKey;\n                           Specifies the loop\'s original key. The key is the\n                           root note or tonic for the loop, and does not\n                           include the scale type.; string\nattributechangedate        kMDItemAttributeChangeDate;\n                           com.apple.metadata:kMDItemAttributeChangeDate; The\n                           date and time of the last change made to a metadata\n                           attribute.; date/time\naudiences                  kMDItemAudiences;\n                           com.apple.metadata:kMDItemAudiences; The audience\n                           for which the file is intended. The audience may be\n                           determined by the creator or the publisher or by a\n                           third party.; list of strings\naudiobitrate               kMDItemAudioBitRate;\n                           com.apple.metadata:kMDItemAudioBitRate; The audio\n                           bit rate.; number\naudiochannelcount          kMDItemAudioChannelCount;\n                           com.apple.metadata:kMDItemAudioChannelCount; Number\n                           of channels in the audio data contained in the\n                           file.; number\naudioencodingapplication   kMDItemAudioEncodingApplication;\n                           com.apple.metadata:kMDItemAudioEncodingApplication;\n                           The name of the application that encoded the data\n                           contained in the audio file.; string\naudiosamplerate            kMDItemAudioSampleRate;\n                           com.apple.metadata:kMDItemAudioSampleRate; Sample\n                           rate of the audio data contained in the file. The\n                           sample rate is a float value representing hz\n                           (audio_frames/second). For example: 44100. 0,\n                           22254. 54.; number\naudiotracknumber           kMDItemAudioTrackNumber;\n                           com.apple.metadata:kMDItemAudioTrackNumber; The\n                           track number of a song or composition when it is\n                           part of an album.; number\nauthoraddresses            kMDItemAuthorAddresses;\n                           com.apple.metadata:kMDItemAuthorAddresses; This\n                           attribute indicates the author addresses of the\n                           document.; list of strings\nauthoremailaddresses       kMDItemAuthorEmailAddresses;\n                           com.apple.metadata:kMDItemAuthorEmailAddresses;\n                           This attribute indicates the author of the emails\n                           message addresses. (This is always the email\n                           address, and not the human readable version).; list\n                           of strings\nauthors                    kMDItemAuthors; com.apple.metadata:kMDItemAuthors;\n                           The author, or authors, of the contents of the\n                           file.; list of strings\nbitspersample              kMDItemBitsPerSample;\n                           com.apple.metadata:kMDItemBitsPerSample; The number\n                           of bits per sample. For example, the bit depth of\n                           an image (8-bit, 16-bit etc. . . ) or the bit depth\n                           per audio sample of uncompressed audio data (8, 16,\n                           24, 32, 64, etc. . ).; number\ncfbundleidentifier         kMDItemCFBundleIdentifier;\n                           com.apple.metadata:kMDItemCFBundleIdentifier; If\n                           this item is a bundle, then this is the\n                           CFBundleIdentifier.; string\ncity                       kMDItemCity; com.apple.metadata:kMDItemCity;\n                           Identifies city of origin according to guidelines\n                           established by the provider.; string\ncodecs                     kMDItemCodecs; com.apple.metadata:kMDItemCodecs;\n                           The codecs used to encode/decode the media.; list\n                           of strings\ncolorspace                 kMDItemColorSpace;\n                           com.apple.metadata:kMDItemColorSpace; The color\n                           space model used by the document contents. For\n                           example, "RGB", "CMYK", "YUV", or "YCbCr".; string\ncomment                    kMDItemComment; com.apple.metadata:kMDItemComment;\n                           A comment related to the file. This differs from\n                           the Finder comment, kMDItemFinderComment.; string\ncomposer                   kMDItemComposer;\n                           com.apple.metadata:kMDItemComposer; The composer of\n                           the music contained in the audio file.; string\ncontactkeywords            kMDItemContactKeywords;\n                           com.apple.metadata:kMDItemContactKeywords; A list\n                           of contacts that are associated with this document,\n                           not including the authors.; list of strings\ncontentcreationdate        kMDItemContentCreationDate;\n                           com.apple.metadata:kMDItemContentCreationDate; The\n                           creation date of an edited or optimized version of\n                           the song or composition.; date/time\ncontentmodificationdate    kMDItemContentModificationDate;\n                           com.apple.metadata:kMDItemContentModificationDate;\n                           The date and time that the contents of the file\n                           were last modified.; date/time\ncontenttype                kMDItemContentType;\n                           com.apple.metadata:kMDItemContentType; The UTI\n                           pedigree of a file.; string\ncontributors               kMDItemContributors;\n                           com.apple.metadata:kMDItemContributors; The\n                           entities responsible for making contributions to\n                           the content of the resource.; list of strings\ncopyright                  kMDItemCopyright;\n                           com.apple.metadata:kMDItemCopyright; The copyright\n                           owner of the file contents.; string\ncountry                    kMDItemCountry; com.apple.metadata:kMDItemCountry;\n                           The full, publishable name of the country or region\n                           where the intellectual property of the item was\n                           created, according to guidelines of the provider.;\n                           string\ncoverage                   kMDItemCoverage;\n                           com.apple.metadata:kMDItemCoverage; The extent or\n                           scope of the content of the resource.; string\ncreator                    kMDItemCreator; com.apple.metadata:kMDItemCreator;\n                           Application used to create the document content\n                           (for example "Word", "Pages", and so on).; string\ndeliverytype               kMDItemDeliveryType;\n                           com.apple.metadata:kMDItemDeliveryType; The\n                           delivery type. Values are "Fast start" or "RTSP".;\n                           string\ndescription                kMDItemDescription;\n                           com.apple.metadata:kMDItemDescription; A\n                           description of the content of the resource. The\n                           description may include an abstract, table of\n                           contents, reference to a graphical representation\n                           of content or a free-text account of the content.;\n                           string\ndirector                   kMDItemDirector;\n                           com.apple.metadata:kMDItemDirector; Directory of\n                           the movie.; string\ndisplayname                kMDItemDisplayName;\n                           com.apple.metadata:kMDItemDisplayName; The\n                           localized version of the file name.; string\ndownloadeddate             kMDItemDownloadedDate;\n                           com.apple.metadata:kMDItemDownloadedDate; Date the\n                           item was downloaded.; list of date/time\nduedate                    kMDItemDueDate; com.apple.metadata:kMDItemDueDate;\n                           Date this item is due.; date/time\ndurationseconds            kMDItemDurationSeconds;\n                           com.apple.metadata:kMDItemDurationSeconds; The\n                           duration, in seconds, of the content of file. A\n                           value of 10. 5 represents media that is 10 and 1/2\n                           seconds long.; number\nexifgpsversion             kMDItemEXIFGPSVersion;\n                           com.apple.metadata:kMDItemEXIFGPSVersion; The\n                           version of GPSInfoIFD in EXIF used to generate the\n                           metadata.; string\nexifversion                kMDItemEXIFVersion;\n                           com.apple.metadata:kMDItemEXIFVersion; The version\n                           of the EXIF header used to generate the metadata.;\n                           string\nemailaddresses             kMDItemEmailAddresses;\n                           com.apple.metadata:kMDItemEmailAddresses; Email\n                           addresses related to this item.; list of strings\nencodingapplications       kMDItemEncodingApplications;\n                           com.apple.metadata:kMDItemEncodingApplications;\n                           Application used to convert the original content\n                           into it\'s current form. For example, a PDF file\n                           might have an encoding application set to\n                           "Distiller".; list of strings\nexposuremode               kMDItemExposureMode;\n                           com.apple.metadata:kMDItemExposureMode; The\n                           exposure mode used to acquire the document\n                           contents.; number\nexposureprogram            kMDItemExposureProgram;\n                           com.apple.metadata:kMDItemExposureProgram; The\n                           class of the exposure program used by the camera to\n                           set exposure when the image is taken. Possible\n                           values include: Manual, Normal, and Aperture\n                           priority.; string\nexposuretimeseconds        kMDItemExposureTimeSeconds;\n                           com.apple.metadata:kMDItemExposureTimeSeconds; The\n                           exposure time, in seconds, used to acquire the\n                           document contents.; number\nexposuretimestring         kMDItemExposureTimeString;\n                           com.apple.metadata:kMDItemExposureTimeString; The\n                           time of the exposure.; string\nfnumber                    kMDItemFNumber; com.apple.metadata:kMDItemFNumber;\n                           The diameter of the diaphragm aperture in terms of\n                           the effective focal length of the lens.; number\nfscontentchangedate        kMDItemFSContentChangeDate;\n                           com.apple.metadata:kMDItemFSContentChangeDate; The\n                           date the file contents last changed.; date/time\nfscreationdate             kMDItemFSCreationDate;\n                           com.apple.metadata:kMDItemFSCreationDate; The date\n                           and time that the file was created.; date/time\nfshascustomicon            kMDItemFSHasCustomIcon;\n                           com.apple.metadata:kMDItemFSHasCustomIcon; Boolean\n                           indicating if this file has a custom icon.; boolean\nfsinvisible                kMDItemFSInvisible;\n                           com.apple.metadata:kMDItemFSInvisible; Indicates\n                           whether the file is invisible.; boolean\nfsisextensionhidden        kMDItemFSIsExtensionHidden;\n                           com.apple.metadata:kMDItemFSIsExtensionHidden;\n                           Indicates whether the file extension of the file is\n                           hidden.; boolean\nfsisstationery             kMDItemFSIsStationery;\n                           com.apple.metadata:kMDItemFSIsStationery; Boolean\n                           indicating if this file is stationery.; boolean\nfslabel                    kMDItemFSLabel; com.apple.metadata:kMDItemFSLabel;\n                           Index of the Finder label of the file. Possible\n                           values are 0 through 7.; number\nfsname                     kMDItemFSName; com.apple.metadata:kMDItemFSName;\n                           The file name of the item.; string\nfsnodecount                kMDItemFSNodeCount;\n                           com.apple.metadata:kMDItemFSNodeCount; Number of\n                           files in a directory.; number\nfsownergroupid             kMDItemFSOwnerGroupID;\n                           com.apple.metadata:kMDItemFSOwnerGroupID; The group\n                           ID of the owner of the file.; number\nfsowneruserid              kMDItemFSOwnerUserID;\n                           com.apple.metadata:kMDItemFSOwnerUserID; The user\n                           ID of the owner of the file.; number\nfssize                     kMDItemFSSize; com.apple.metadata:kMDItemFSSize;\n                           The size, in bytes, of the file on disk.; number\nfindercomment              kMDItemFinderComment;\n                           com.apple.metadata:kMDItemFinderComment; Finder\n                           comments for this file.; string\nflashonoff                 kMDItemFlashOnOff;\n                           com.apple.metadata:kMDItemFlashOnOff; Indicates if\n                           a camera flash was used.; number\nfocallength                kMDItemFocalLength;\n                           com.apple.metadata:kMDItemFocalLength; The actual\n                           focal length of the lens, in millimeters.; number\nfonts                      kMDItemFonts; com.apple.metadata:kMDItemFonts;\n                           Fonts used in this item. You should store the\n                           font\'s full name, the postscript name, or the font\n                           family name, based on the available information.;\n                           list of strings\ngpstrack                   kMDItemGPSTrack;\n                           com.apple.metadata:kMDItemGPSTrack; The direction\n                           of travel of the item, in degrees from true north.;\n                           string\ngenre                      kMDItemGenre; com.apple.metadata:kMDItemGenre;\n                           Genre of the movie.; string\nhasalphachannel            kMDItemHasAlphaChannel;\n                           com.apple.metadata:kMDItemHasAlphaChannel;\n                           Indicates if this image file has an alpha channel.;\n                           boolean\nheadline                   kMDItemHeadline;\n                           com.apple.metadata:kMDItemHeadline; A publishable\n                           entry providing a synopsis of the contents of the\n                           file. For example, "Apple Introduces the iPod\n                           Photo".; string\nisospeed                   kMDItemISOSpeed;\n                           com.apple.metadata:kMDItemISOSpeed; The ISO speed\n                           used to acquire the document contents.; number\nidentifier                 kMDItemIdentifier;\n                           com.apple.metadata:kMDItemIdentifier; A formal\n                           identifier used to reference the resource within a\n                           given context.; string\nimagedirection             kMDItemImageDirection;\n                           com.apple.metadata:kMDItemImageDirection; The\n                           direction of the item\'s image, in degrees from true\n                           north.; string\ninformation                kMDItemInformation;\n                           com.apple.metadata:kMDItemInformation; Information\n                           about the item.; string\ninstantmessageaddresses    kMDItemInstantMessageAddresses;\n                           com.apple.metadata:kMDItemInstantMessageAddresses;\n                           Instant message addresses related to this item.;\n                           list of strings\ninstructions               kMDItemInstructions;\n                           com.apple.metadata:kMDItemInstructions; Editorial\n                           instructions concerning the use of the item, such\n                           as embargoes and warnings. For example, "Second of\n                           four stories".; string\nisgeneralmidisequence      kMDItemIsGeneralMIDISequence;\n                           com.apple.metadata:kMDItemIsGeneralMIDISequence;\n                           Indicates whether the MIDI sequence contained in\n                           the file is setup for use with a General MIDI\n                           device.; boolean\nkeysignature               kMDItemKeySignature;\n                           com.apple.metadata:kMDItemKeySignature; The key of\n                           the music contained in the audio file. For example:\n                           C, Dm, F#m, Bb.; string\nkeywords                   kMDItemKeywords;\n                           com.apple.metadata:kMDItemKeywords; Keywords\n                           associated with this file. For example, "Birthday",\n                           "Important", etc.; list of strings\nkind                       kMDItemKind; com.apple.metadata:kMDItemKind; A\n                           description of the kind of item this file\n                           represents.; string\nlanguages                  kMDItemLanguages;\n                           com.apple.metadata:kMDItemLanguages; Indicates the\n                           languages of the intellectual content of the\n                           resource. Recommended best practice for the values\n                           of the Language element is defined by RFC 3066.;\n                           list of strings\nlastuseddate               kMDItemLastUsedDate;\n                           com.apple.metadata:kMDItemLastUsedDate; The date\n                           and time that the file was last used. This value is\n                           updated automatically by LaunchServices everytime a\n                           file is opened by double clicking, or by asking\n                           LaunchServices to open a file.; date/time\nlatitude                   kMDItemLatitude;\n                           com.apple.metadata:kMDItemLatitude; The latitude of\n                           the item in degrees north of the equator, expressed\n                           using the WGS84 datum. Negative values lie south of\n                           the equator.; string\nlayernames                 kMDItemLayerNames;\n                           com.apple.metadata:kMDItemLayerNames; The names of\n                           the layers in the file.; list of strings\nlongitude                  kMDItemLongitude;\n                           com.apple.metadata:kMDItemLongitude; The longitude\n                           of the item in degrees east of the prime meridian,\n                           expressed using the WGS84 datum. Negative values\n                           lie west of the prime meridian.; string\nlyricist                   kMDItemLyricist;\n                           com.apple.metadata:kMDItemLyricist; The lyricist,\n                           or text writer, of the music contained in the audio\n                           file.; string\nmaxaperture                kMDItemMaxAperture;\n                           com.apple.metadata:kMDItemMaxAperture; The smallest\n                           f-number of the lens. Ordinarily it is given in the\n                           range of 00. 00 to 99. 99.; number\nmediatypes                 kMDItemMediaTypes;\n                           com.apple.metadata:kMDItemMediaTypes; The media\n                           types present in the content.; list of strings\nmeteringmode               kMDItemMeteringMode;\n                           com.apple.metadata:kMDItemMeteringMode; The\n                           metering mode used to take the image.; string\nmusicalgenre               kMDItemMusicalGenre;\n                           com.apple.metadata:kMDItemMusicalGenre; The musical\n                           genre of the song or composition contained in the\n                           audio file. For example: Jazz, Pop, Rock,\n                           Classical.; string\nmusicalinstrumentcategory  kMDItemMusicalInstrumentCategory; com.apple.metadat\n                           a:kMDItemMusicalInstrumentCategory; Specifies the\n                           category of an instrument.; string\nmusicalinstrumentname      kMDItemMusicalInstrumentName;\n                           com.apple.metadata:kMDItemMusicalInstrumentName;\n                           Specifies the name of instrument relative to the\n                           instrument category.; string\nnamedlocation              kMDItemNamedLocation;\n                           com.apple.metadata:kMDItemNamedLocation; The name\n                           of the location or point of interest associated\n                           with the item. The name may be user provided.;\n                           string\nnumberofpages              kMDItemNumberOfPages;\n                           com.apple.metadata:kMDItemNumberOfPages; Number of\n                           pages in the document.; number\norganizations              kMDItemOrganizations;\n                           com.apple.metadata:kMDItemOrganizations; The\n                           company or organization that created the document.;\n                           list of strings\norientation                kMDItemOrientation;\n                           com.apple.metadata:kMDItemOrientation; The\n                           orientation of the document contents. Possible\n                           values are 0 (landscape) and 1 (portrait).; number\noriginalformat             kMDItemOriginalFormat;\n                           com.apple.metadata:kMDItemOriginalFormat; Original\n                           format of the movie.; string\noriginalsource             kMDItemOriginalSource;\n                           com.apple.metadata:kMDItemOriginalSource; Original\n                           source of the movie.; string\npageheight                 kMDItemPageHeight;\n                           com.apple.metadata:kMDItemPageHeight; Height of the\n                           document page, in points (72 points per inch). For\n                           PDF files this indicates the height of the first\n                           page only.; number\npagewidth                  kMDItemPageWidth;\n                           com.apple.metadata:kMDItemPageWidth; Width of the\n                           document page, in points (72 points per inch). For\n                           PDF files this indicates the width of the first\n                           page only.; number\nparticipants               kMDItemParticipants;\n                           com.apple.metadata:kMDItemParticipants; The list of\n                           people who are visible in an image or movie or\n                           written about in a document.; list of strings\npath                       kMDItemPath; com.apple.metadata:kMDItemPath; The\n                           complete path to the file.; string\nperformers                 kMDItemPerformers;\n                           com.apple.metadata:kMDItemPerformers; Performers in\n                           the movie.; list of strings\nphonenumbers               kMDItemPhoneNumbers;\n                           com.apple.metadata:kMDItemPhoneNumbers; Phone\n                           numbers related to this item.; list of strings\npixelcount                 kMDItemPixelCount;\n                           com.apple.metadata:kMDItemPixelCount; The total\n                           number of pixels in the contents. Same as\n                           kMDItemPixelWidth x kMDItemPixelHeight.; number\npixelheight                kMDItemPixelHeight;\n                           com.apple.metadata:kMDItemPixelHeight; The height,\n                           in pixels, of the contents. For example, the image\n                           height or the video frame height.; number\npixelwidth                 kMDItemPixelWidth;\n                           com.apple.metadata:kMDItemPixelWidth; The width, in\n                           pixels, of the contents. For example, the image\n                           width or the video frame width.; number\nproducer                   kMDItemProducer;\n                           com.apple.metadata:kMDItemProducer; Producer of the\n                           content.; string\nprofilename                kMDItemProfileName;\n                           com.apple.metadata:kMDItemProfileName; The name of\n                           the color profile used by the document contents.;\n                           string\nprojects                   kMDItemProjects;\n                           com.apple.metadata:kMDItemProjects; The list of\n                           projects that this file is part of. For example, if\n                           you were working on a movie all of the files could\n                           be marked as belonging to the project "My Movie".;\n                           list of strings\npublishers                 kMDItemPublishers;\n                           com.apple.metadata:kMDItemPublishers; The entity\n                           responsible for making the resource available. For\n                           example, a person, an organization, or a service.\n                           Typically, the name of a publisher should be used\n                           to indicate the entity.; list of strings\nrecipientaddresses         kMDItemRecipientAddresses;\n                           com.apple.metadata:kMDItemRecipientAddresses; This\n                           attribute indicates the recipient addresses of the\n                           document.; list of strings\nrecipientemailaddresses    kMDItemRecipientEmailAddresses;\n                           com.apple.metadata:kMDItemRecipientEmailAddresses;\n                           This attribute indicates the recipients email\n                           addresses. (This is always the email address, and\n                           not the human readable version).; list of strings\nrecipients                 kMDItemRecipients;\n                           com.apple.metadata:kMDItemRecipients; Recipients of\n                           this item.; list of strings\nrecordingdate              kMDItemRecordingDate;\n                           com.apple.metadata:kMDItemRecordingDate; The\n                           recording date of the song or composition.;\n                           date/time\nrecordingyear              kMDItemRecordingYear;\n                           com.apple.metadata:kMDItemRecordingYear; Indicates\n                           the year the item was recorded. For example, 1964,\n                           2003, etc.; number\nredeyeonoff                kMDItemRedEyeOnOff;\n                           com.apple.metadata:kMDItemRedEyeOnOff; Indicates if\n                           red-eye reduction was used to take the picture.;\n                           boolean\nresolutionheightdpi        kMDItemResolutionHeightDPI;\n                           com.apple.metadata:kMDItemResolutionHeightDPI;\n                           Resolution height, in DPI, of this image.; number\nresolutionwidthdpi         kMDItemResolutionWidthDPI;\n                           com.apple.metadata:kMDItemResolutionWidthDPI;\n                           Resolution width, in DPI, of this image.; number\nrights                     kMDItemRights; com.apple.metadata:kMDItemRights;\n                           Provides a link to information about rights held in\n                           and over the resource.; string\nsecuritymethod             kMDItemSecurityMethod;\n                           com.apple.metadata:kMDItemSecurityMethod; The\n                           security or encryption method used for the file.;\n                           string\nspeed                      kMDItemSpeed; com.apple.metadata:kMDItemSpeed; The\n                           speed of the item, in kilometers per hour.; string\nstarrating                 kMDItemStarRating;\n                           com.apple.metadata:kMDItemStarRating; User rating\n                           of this item. For example, the stars rating of an\n                           iTunes track.; number\nstateorprovince            kMDItemStateOrProvince;\n                           com.apple.metadata:kMDItemStateOrProvince;\n                           Identifies the province or state of origin\n                           according to guidelines established by the\n                           provider. For example, "CA", "Ontario", or\n                           "Sussex".; string\nstreamable                 kMDItemStreamable;\n                           com.apple.metadata:kMDItemStreamable; Whether the\n                           content is prepared for streaming.; boolean\nsubject                    kMDItemSubject; com.apple.metadata:kMDItemSubject;\n                           Subject of the this item.; string\ntempo                      kMDItemTempo; com.apple.metadata:kMDItemTempo; A\n                           float value that specifies the beats per minute of\n                           the music contained in the audio file.; number\ntextcontent                kMDItemTextContent;\n                           com.apple.metadata:kMDItemTextContent; Contains a\n                           text representation of the content of the document.\n                           Data in multiple fields should be combined using a\n                           whitespace character as a separator.; string\ntheme                      kMDItemTheme; com.apple.metadata:kMDItemTheme;\n                           Theme of the this item.; string\ntimesignature              kMDItemTimeSignature;\n                           com.apple.metadata:kMDItemTimeSignature; The time\n                           signature of the musical composition contained in\n                           the audio/MIDI file. For example: "4/4", "7/8".;\n                           string\ntimestamp                  kMDItemTimestamp;\n                           com.apple.metadata:kMDItemTimestamp; The timestamp\n                           on the item. This generally is used to indicate the\n                           time at which the event captured by the item took\n                           place.; string\ntitle                      kMDItemTitle; com.apple.metadata:kMDItemTitle; The\n                           title of the file. For example, this could be the\n                           title of a document, the name of a song, or the\n                           subject of an email message.; string\ntotalbitrate               kMDItemTotalBitRate;\n                           com.apple.metadata:kMDItemTotalBitRate; The total\n                           bit rate, audio and video combined, of the media.;\n                           number\nurl                        kMDItemURL; com.apple.metadata:kMDItemURL; Url of\n                           the item.; string\nversion                    kMDItemVersion; com.apple.metadata:kMDItemVersion;\n                           The version number of this file.; string\nvideobitrate               kMDItemVideoBitRate;\n                           com.apple.metadata:kMDItemVideoBitRate; The video\n                           bit rate.; number\nwherefroms                 kMDItemWhereFroms;\n                           com.apple.metadata:kMDItemWhereFroms; Describes\n                           where the file was obtained from.; list of strings\nwhitebalance               kMDItemWhiteBalance;\n                           com.apple.metadata:kMDItemWhiteBalance; The white\n                           balance setting used to acquire the document\n                           contents. Possible values are 0 (auto white\n                           balance) and 1 (manual).; number\n\n\n```\n<!-- [[[end]]] -->\n\n## Notes on backup/restore\n\nWhen run with `--backup`, osxmetadata backs up the metadata of each file in a file called `.osxmetadata.json`. A backup file is created in every directory that includes files being backup up. The format is plain JSON text with a record for each file that was backed up. If you delete a file then run the `--backup` again, the deleted file\'s record is not deleted from the `.osxmetadata.json` backup file. The backup file is kept in each directory/sub-directory and only the filename is used for `--restore` which means you can move/rename the directory (along with the `.osxmetadata.json` file) and the restore will still work correctly.\n\n**Note**: Prior to version 0.99.38, the backup file was not well-formed JSON which meant that some apps/viewers could not process the JSON file.  Version 0.99.38 fixes this and will silently update any `.osxmetadata.json` file encountered during `--backup` to be well-formed JSON but this breaks backwards compatibility with older versions of osxmetadata. If you use osxmetadata to sync data across multiple Macs, you must ensure all Macs are running the updated version.  For additional details, see [issue #57](https://github.com/RhetTbull/osxmetadata/issues/57).\n\n## Usage Notes\n\nThis will only work on file systems that support Mac OS X extended attributes.\n\n## Related Projects\n\n- [tag](https://github.com/jdberry/tag) A command line tool to manipulate tags on Mac OS X files, and to query for files with those tags.\n- [osx-tags](https://github.com/scooby/osx-tags) Python module to manipulate Finder tags in OS X.\n\n## Acknowledgements\n\nThis module was inspired by [osx-tags](https://github.com/scooby/osx-tags) by "Ben S / scooby".  I leveraged osx-tags to bootstrap the design of this module.  I wanted a more general OS X metadata library so I rolled my own.  This module is published under the same MIT license as osx-tags.\n\n## License\n\nMIT License\n\nCopyright (c) 2020 Rhet Turnbull\n\n## Contributing\n\nContributions of all kinds are welcome.  Please submit a pull request or open an issue.\n\n## Contributors ✨\n\nThanks goes to these wonderful people ([emoji key](https://allcontributors.org/docs/en/emoji-key)):\n\n<!-- ALL-CONTRIBUTORS-LIST:START - Do not remove or modify this section -->\n<!-- prettier-ignore-start -->\n<!-- markdownlint-disable -->\n<table>\n  <tbody>\n    <tr>\n      <td align="center"><a href="http://www.borja.glezseoane.es"><img src="https://avatars.githubusercontent.com/u/24481419?v=4?s=75" width="75px;" alt="Borja González Seoane"/><br /><sub><b>Borja González Seoane</b></sub></a><br /><a href="https://github.com/RhetTbull/osxmetadata/commits?author=bglezseoane" title="Code">💻</a></td>\n      <td align="center"><a href="https://github.com/porg"><img src="https://avatars.githubusercontent.com/u/737143?v=4?s=75" width="75px;" alt="porg"/><br /><sub><b>porg</b></sub></a><br /><a href="https://github.com/RhetTbull/osxmetadata/issues?q=author%3Aporg" title="Bug reports">🐛</a> <a href="#ideas-porg" title="Ideas, Planning, & Feedback">🤔</a></td>\n      <td align="center"><a href="https://github.com/nk9"><img src="https://avatars.githubusercontent.com/u/3646730?v=4?s=75" width="75px;" alt="Nick Kocharhook"/><br /><sub><b>Nick Kocharhook</b></sub></a><br /><a href="https://github.com/RhetTbull/osxmetadata/issues?q=author%3Ank9" title="Bug reports">🐛</a></td>\n      <td align="center"><a href="https://jakewilliami.github.io/"><img src="https://avatars.githubusercontent.com/u/54291317?v=4?s=75" width="75px;" alt="Jake Ireland"/><br /><sub><b>Jake Ireland</b></sub></a><br /><a href="#ideas-jakewilliami" title="Ideas, Planning, & Feedback">🤔</a></td>\n    </tr>\n  </tbody>\n</table>\n\n<!-- markdownlint-restore -->\n<!-- prettier-ignore-end -->\n\n<!-- ALL-CONTRIBUTORS-LIST:END -->\n\nThis project follows the [all-contributors](https://github.com/all-contributors/all-contributors) specification. Contributions of any kind welcome!\n',
    'author': 'Rhet Turnbull',
    'author_email': 'rturnbull+git@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/RhetTbull/osxmetadata',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<3.12',
}


setup(**setup_kwargs)
