"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Apisix = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const aws_ecs_patterns_1 = require("@aws-cdk/aws-ecs-patterns");
const efs = require("@aws-cdk/aws-efs");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const iam = require("@aws-cdk/aws-iam");
const log = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const DEFAULTS = {
    apisixContainer: 'public.ecr.aws/pahudnet/apisix-docker:v2.4',
    etcdContainer: 'public.ecr.aws/eks-distro/etcd-io/etcd:v3.4.14-eks-1-19-1',
    dashboardContainer: 'public.ecr.aws/pahudnet/apisix-dashboard:v2.4',
};
/**
 * The Apisix construct.
 *
 * @stability stable
 */
class Apisix extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b, _c, _d, _e, _f, _g;
        super(scope, id);
        const stack = cdk.Stack.of(this);
        const vpc = (_b = props.vpc) !== null && _b !== void 0 ? _b : getOrCreateVpc(this);
        this.vpc = vpc;
        const cluster = (_c = props.cluster) !== null && _c !== void 0 ? _c : new ecs.Cluster(this, 'Cluster', { vpc });
        this.cluster = cluster;
        const requiredContextVariables = [
            'ADMIN_KEY_ADMIN',
            'ADMIN_KEY_VIEWER',
            'DASHBOARD_ADMIN_PASSWORD',
            'DASHBOARD_USER_PASSWORD',
        ];
        requiredContextVariables.map(v => throwIfNotAvailable(this, v));
        throwIfNotAvailable(this, 'ADMIN_KEY_ADMIN');
        this.envVar = {
            ADMIN_KEY_ADMIN: stack.node.tryGetContext('ADMIN_KEY_ADMIN'),
            ADMIN_KEY_VIEWER: stack.node.tryGetContext('ADMIN_KEY_VIEWER'),
            ETCD_HOST: stack.node.tryGetContext('ETCD_HOST') || '0.0.0.0',
            ETCD_PORT: stack.node.tryGetContext('ETCD_PORT') || '2379',
            DASHBOARD_ADMIN_PASSWORD: stack.node.tryGetContext('DASHBOARD_ADMIN_PASSWORD'),
            DASHBOARD_USER_PASSWORD: stack.node.tryGetContext('DASHBOARD_USER_PASSWORD'),
        };
        /**
         * Amazon EFS filesystem for etcd
         */
        const fs = (_d = props.efsFilesystem) !== null && _d !== void 0 ? _d : this._createEfsFilesystem();
        /**
         * ApiSix service
         */
        const taskDefinition = new ecs.FargateTaskDefinition(this, 'TaskApiSix', {
            memoryLimitMiB: 512,
            cpu: 256,
        });
        const apisix = taskDefinition
            .addContainer('apisix', {
            image: (_e = props.apisixContainer) !== null && _e !== void 0 ? _e : ecs.ContainerImage.fromRegistry(DEFAULTS.apisixContainer),
            logging: new ecs.AwsLogDriver({
                streamPrefix: 'apisix',
                logRetention: log.RetentionDays.ONE_DAY,
            }),
            environment: {
                ADMIN_KEY_ADMIN: this.envVar.ADMIN_KEY_ADMIN,
                ADMIN_KEY_VIEWER: this.envVar.ADMIN_KEY_VIEWER,
            },
            portMappings: [{ containerPort: 9080 }],
        });
        taskDefinition.addVolume({
            name: 'etcd-data',
            efsVolumeConfiguration: {
                fileSystemId: fs.fileSystemId,
            },
        });
        taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
            actions: [
                'elasticfilesystem:ClientMount',
                'elasticfilesystem:ClientWrite',
            ],
            resources: [
                stack.formatArn({
                    service: 'elasticfilesystem',
                    resource: 'file-system',
                    sep: '/',
                    resourceName: fs.fileSystemId,
                }),
            ],
        }));
        const etcdContainer = taskDefinition
            .addContainer('etcd', {
            image: (_f = props.etcdContainer) !== null && _f !== void 0 ? _f : ecs.ContainerImage.fromRegistry(DEFAULTS.etcdContainer),
            environment: {
                ETCD_DATA_DIR: '/etcd_data',
                ETCD_ENABLE_V2: 'true',
                ALLOW_NONE_AUTHENTICATION: 'yes',
                ETCD_ADVERTISE_CLIENT_URLS: 'http://0.0.0.0:2379',
                ETCD_LISTEN_CLIENT_URLS: 'http://0.0.0.0:2379',
            },
            logging: new ecs.AwsLogDriver({
                streamPrefix: 'etcd',
                logRetention: log.RetentionDays.ONE_DAY,
            }),
        });
        etcdContainer.addMountPoints({
            containerPath: '/etcd_data',
            sourceVolume: 'etcd-data',
            readOnly: false,
        });
        etcdContainer.addPortMappings({
            containerPort: 2379,
        });
        apisix.addContainerDependencies({
            container: etcdContainer,
            condition: ecs.ContainerDependencyCondition.START,
        });
        // add dashboard container
        const dashboard = taskDefinition.addContainer('dashboard', {
            image: (_g = props.dashboardContainer) !== null && _g !== void 0 ? _g : ecs.ContainerImage.fromRegistry(DEFAULTS.dashboardContainer),
            logging: new ecs.AwsLogDriver({
                streamPrefix: 'dashboard',
                logRetention: log.RetentionDays.ONE_DAY,
            }),
            environment: {
                ETCD_HOST: this.envVar.ETCD_HOST,
                ETCD_PORT: this.envVar.ETCD_PORT,
                ADMIN_PASSWORD: this.envVar.DASHBOARD_ADMIN_PASSWORD,
                USER_PASSWORD: this.envVar.DASHBOARD_USER_PASSWORD,
            },
        });
        dashboard.addPortMappings({
            containerPort: 9000,
        });
        dashboard.addContainerDependencies({
            container: etcdContainer,
            condition: ecs.ContainerDependencyCondition.START,
        });
        const apisixService = new ecs.FargateService(this, 'APISIXService', {
            cluster,
            taskDefinition,
            platformVersion: ecs.FargatePlatformVersion.VERSION1_4,
        });
        /**
         * create ALB
         */
        const alb = new elbv2.ApplicationLoadBalancer(this, 'ALB', { vpc, internetFacing: true });
        // APISIX listener on 80
        const apisixListener = new elbv2.ApplicationListener(this, 'APISIXListener', {
            loadBalancer: alb,
            // defaultTargetGroups: [apisixTG],
            port: 80,
        });
        apisixListener.addTargets('ApiSixTargets', {
            port: 80,
            targets: [
                apisixService.loadBalancerTarget({
                    containerName: 'apisix',
                    containerPort: 9080,
                }),
            ],
            healthCheck: {
                healthyHttpCodes: '200-499',
            },
        });
        // dashboard listener on 9000
        const dashboardListener = new elbv2.ApplicationListener(this, 'DashboardListener', {
            loadBalancer: alb,
            protocol: elbv2.ApplicationProtocol.HTTP,
            port: 9000,
        });
        dashboardListener.addTargets('DashboardTargets', {
            port: 9000,
            protocol: elbv2.ApplicationProtocol.HTTP,
            targets: [
                apisixService.loadBalancerTarget({
                    containerName: 'dashboard',
                    containerPort: 9000,
                }),
            ],
            healthCheck: {
                healthyHttpCodes: '200-499',
            },
        });
        // allow all traffic from ALB to service
        apisixService.connections.allowFrom(alb, aws_ec2_1.Port.allTraffic());
        // allow connection between efs filesystem
        apisixService.connections.allowFrom(fs, aws_ec2_1.Port.tcp(2049));
        apisixService.connections.allowTo(fs, aws_ec2_1.Port.tcp(2049));
        new cdk.CfnOutput(this, 'ApiSixURL', {
            value: `http://${alb.loadBalancerDnsName}`,
        });
    }
    _createEfsFilesystem() {
        return new efs.FileSystem(this, 'filesystem', {
            vpc: this.vpc,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
    }
    /**
     * Create a basic web service on AWS Fargate.
     *
     * @stability stable
     */
    createWebService(id, options) {
        var _b, _c;
        // flask service
        const DEFAULT_SERVICE_IMAGE = 'public.ecr.aws/pahudnet/flask-docker-sample';
        const task = new ecs.FargateTaskDefinition(this, `task${id}`, {
            cpu: 256,
            memoryLimitMiB: 512,
        });
        task
            .addContainer(`container${id}`, {
            image: (_b = options.image) !== null && _b !== void 0 ? _b : ecs.ContainerImage.fromRegistry(DEFAULT_SERVICE_IMAGE),
            environment: options.environment,
            logging: new ecs.AwsLogDriver({
                streamPrefix: id,
                logRetention: log.RetentionDays.ONE_DAY,
            }),
        })
            .addPortMappings({
            containerPort: (_c = options.port) !== null && _c !== void 0 ? _c : 80,
        });
        const service = new aws_ecs_patterns_1.NetworkLoadBalancedFargateService(this, `service${id}`, {
            cluster: this.cluster,
            taskDefinition: task,
            assignPublicIp: true,
        });
        // allow Fargate task behind NLB to accept all traffic
        service.service.connections.allowFromAnyIpv4(aws_ec2_1.Port.tcp(80));
        service.targetGroup.setAttribute('deregistration_delay.timeout_seconds', '30');
        service.loadBalancer.setAttribute('load_balancing.cross_zone.enabled', 'true');
        return service;
    }
}
exports.Apisix = Apisix;
_a = JSII_RTTI_SYMBOL_1;
Apisix[_a] = { fqn: "cdk-apisix.Apisix", version: "0.0.278" };
function getOrCreateVpc(scope) {
    // use an existing vpc or create a new one
    return scope.node.tryGetContext('use_default_vpc') === '1' ?
        aws_ec2_1.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
        scope.node.tryGetContext('use_vpc_id') ?
            aws_ec2_1.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
            new aws_ec2_1.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
}
function isContextAvailable(scope, key) {
    return cdk.Stack.of(scope).node.tryGetContext(key);
}
/**
 * Throws if the context is not available
 */
function throwIfNotAvailable(scope, key) {
    if (!isContextAvailable(scope, key)) {
        throw new Error(`${key} is required in the context variable`);
    }
}
//# sourceMappingURL=data:application/json;base64,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