"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SelfManagedKafkaEventSource = exports.ManagedKafkaEventSource = exports.AuthenticationMethod = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const stream_1 = require("./stream");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("@aws-cdk/core");
/**
 * The authentication method to use with SelfManagedKafkaEventSource.
 *
 * @stability stable
 */
var AuthenticationMethod;
(function (AuthenticationMethod) {
    AuthenticationMethod["SASL_SCRAM_512_AUTH"] = "SASL_SCRAM_512_AUTH";
    AuthenticationMethod["SASL_SCRAM_256_AUTH"] = "SASL_SCRAM_256_AUTH";
    AuthenticationMethod["BASIC_AUTH"] = "BASIC_AUTH";
})(AuthenticationMethod = exports.AuthenticationMethod || (exports.AuthenticationMethod = {}));
/**
 * Use a MSK cluster as a streaming source for AWS Lambda.
 *
 * @stability stable
 */
class ManagedKafkaEventSource extends stream_1.StreamEventSource {
    /**
     * @stability stable
     */
    constructor(props) {
        super(props);
        this.innerProps = props;
    }
    /**
     * Called by `lambda.addEventSource` to allow the event source to bind to this function.
     *
     * @stability stable
     */
    bind(target) {
        var _c;
        target.addEventSourceMapping(`KafkaEventSource:${this.innerProps.clusterArn}${this.innerProps.topic}`, this.enrichMappingOptions({
            eventSourceArn: this.innerProps.clusterArn,
            startingPosition: this.innerProps.startingPosition,
            sourceAccessConfigurations: this.sourceAccessConfigurations(),
            kafkaTopic: this.innerProps.topic,
        }));
        if (this.innerProps.secret !== undefined) {
            this.innerProps.secret.grantRead(target);
        }
        target.addToRolePolicy(new iam.PolicyStatement({
            actions: ['kafka:DescribeCluster', 'kafka:GetBootstrapBrokers', 'kafka:ListScramSecrets'],
            resources: [this.innerProps.clusterArn],
        }));
        (_c = target.role) === null || _c === void 0 ? void 0 : _c.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaMSKExecutionRole'));
    }
    sourceAccessConfigurations() {
        const sourceAccessConfigurations = [];
        if (this.innerProps.secret !== undefined) {
            // "Amazon MSK only supports SCRAM-SHA-512 authentication." from https://docs.aws.amazon.com/msk/latest/developerguide/msk-password.html#msk-password-limitations
            sourceAccessConfigurations.push({
                type: lambda.SourceAccessConfigurationType.SASL_SCRAM_512_AUTH,
                uri: this.innerProps.secret.secretArn,
            });
        }
        return sourceAccessConfigurations.length === 0
            ? undefined
            : sourceAccessConfigurations;
    }
}
exports.ManagedKafkaEventSource = ManagedKafkaEventSource;
_a = JSII_RTTI_SYMBOL_1;
ManagedKafkaEventSource[_a] = { fqn: "@aws-cdk/aws-lambda-event-sources.ManagedKafkaEventSource", version: "1.130.0" };
/**
 * Use a self hosted Kafka installation as a streaming source for AWS Lambda.
 *
 * @stability stable
 */
class SelfManagedKafkaEventSource extends stream_1.StreamEventSource {
    /**
     * @stability stable
     */
    constructor(props) {
        super(props);
        if (props.vpc) {
            if (!props.securityGroup) {
                throw new Error('securityGroup must be set when providing vpc');
            }
            if (!props.vpcSubnets) {
                throw new Error('vpcSubnets must be set when providing vpc');
            }
        }
        else if (!props.secret) {
            throw new Error('secret must be set if Kafka brokers accessed over Internet');
        }
        this.innerProps = props;
    }
    /**
     * Called by `lambda.addEventSource` to allow the event source to bind to this function.
     *
     * @stability stable
     */
    bind(target) {
        if (!core_2.Construct.isConstruct(target)) {
            throw new Error('Function is not a construct. Unexpected error.');
        }
        target.addEventSourceMapping(this.mappingId(target), this.enrichMappingOptions({
            kafkaBootstrapServers: this.innerProps.bootstrapServers,
            kafkaTopic: this.innerProps.topic,
            startingPosition: this.innerProps.startingPosition,
            sourceAccessConfigurations: this.sourceAccessConfigurations(),
        }));
        if (this.innerProps.secret !== undefined) {
            this.innerProps.secret.grantRead(target);
        }
    }
    mappingId(target) {
        let hash = crypto.createHash('md5');
        hash.update(JSON.stringify(core_1.Stack.of(target).resolve(this.innerProps.bootstrapServers)));
        const idHash = hash.digest('hex');
        return `KafkaEventSource:${idHash}:${this.innerProps.topic}`;
    }
    sourceAccessConfigurations() {
        var _c;
        let authType;
        switch (this.innerProps.authenticationMethod) {
            case AuthenticationMethod.BASIC_AUTH:
                authType = lambda.SourceAccessConfigurationType.BASIC_AUTH;
                break;
            case AuthenticationMethod.SASL_SCRAM_256_AUTH:
                authType = lambda.SourceAccessConfigurationType.SASL_SCRAM_256_AUTH;
                break;
            case AuthenticationMethod.SASL_SCRAM_512_AUTH:
            default:
                authType = lambda.SourceAccessConfigurationType.SASL_SCRAM_512_AUTH;
                break;
        }
        const sourceAccessConfigurations = [];
        if (this.innerProps.secret !== undefined) {
            sourceAccessConfigurations.push({ type: authType, uri: this.innerProps.secret.secretArn });
        }
        if (this.innerProps.vpcSubnets !== undefined && this.innerProps.securityGroup !== undefined) {
            sourceAccessConfigurations.push({
                type: lambda.SourceAccessConfigurationType.VPC_SECURITY_GROUP,
                uri: this.innerProps.securityGroup.securityGroupId,
            });
            (_c = this.innerProps.vpc) === null || _c === void 0 ? void 0 : _c.selectSubnets(this.innerProps.vpcSubnets).subnetIds.forEach((id) => {
                sourceAccessConfigurations.push({ type: lambda.SourceAccessConfigurationType.VPC_SUBNET, uri: id });
            });
        }
        return sourceAccessConfigurations.length === 0
            ? undefined
            : sourceAccessConfigurations;
    }
}
exports.SelfManagedKafkaEventSource = SelfManagedKafkaEventSource;
_b = JSII_RTTI_SYMBOL_1;
SelfManagedKafkaEventSource[_b] = { fqn: "@aws-cdk/aws-lambda-event-sources.SelfManagedKafkaEventSource", version: "1.130.0" };
//# sourceMappingURL=data:application/json;base64,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