"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscaling = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const runner_configuration_1 = require("../runner-configuration");
const cache_1 = require("./cache");
const network_1 = require("./network");
/**
 * The Gitlab Runner autoscaling on EC2 by Docker Machine.
 *
 * @stability stable
 * @example
 *
 * <caption>Provisioning a basic Runner</caption>
 * const app = new cdk.App();
 * const stack = new cdk.Stack(app, "RunnerStack", {
 * env: {
 * account: "000000000000",
 * region: "us-east-1",
 * }
 * });
 *
 * new GitlabRunnerAutoscaling(scope, "GitlabRunner", {
 * gitlabToken: "xxxxxxxxxxxxxxxxxxxx",
 * });
 */
class GitlabRunnerAutoscaling extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d;
        super(scope, id);
        const { manager, cache, runners, network } = props;
        /**
         * S3 Bucket for Runners' cache
         */
        this.cacheBucket = (cache === null || cache === void 0 ? void 0 : cache.bucket) || new cache_1.Cache(scope, "Cache", cache === null || cache === void 0 ? void 0 : cache.options).bucket;
        /**
         * Network
         */
        this.network = new network_1.Network(scope, "Network", network);
        /**
         * IAM
         */
        const ec2ServicePrincipal = new aws_iam_1.ServicePrincipal("ec2.amazonaws.com", {});
        const ec2ManagedPolicyForSSM = aws_iam_1.ManagedPolicy.fromManagedPolicyArn(scope, "AmazonEC2RoleforSSM", "arn:aws:iam::aws:policy/service-role/AmazonEC2RoleforSSM");
        /**
         * GitLab Runners
         */
        const runnersSecurityGroupName = `${scope.stackName}-RunnersSecurityGroup`;
        const runnersSecurityGroup = new aws_ec2_1.SecurityGroup(scope, "RunnersSecurityGroup", {
            securityGroupName: runnersSecurityGroupName,
            description: "Security group for GitLab Runners.",
            vpc: this.network.vpc,
        });
        const runnersRole = (runners === null || runners === void 0 ? void 0 : runners.role) ||
            new aws_iam_1.Role(scope, "RunnersRole", {
                assumedBy: ec2ServicePrincipal,
                managedPolicies: [ec2ManagedPolicyForSSM],
            });
        const runnersInstanceProfile = new aws_iam_1.CfnInstanceProfile(scope, "RunnersInstanceProfile", {
            roles: [runnersRole.roleName],
        });
        const runnersInstanceType = (runners === null || runners === void 0 ? void 0 : runners.instanceType) || aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.MICRO);
        const runnersLookupMachineImage = new aws_ec2_1.LookupMachineImage({
            name: "ubuntu/images/hvm-ssd/ubuntu-focal-20.04-amd64-server-*",
            owners: ["099720109477"],
            filters: {
                architecture: ["x86_64"],
                "image-type": ["machine"],
                state: ["available"],
                "root-device-type": ["ebs"],
                "virtualization-type": ["hvm"],
            },
        }).getImage(this);
        const runnersMachineImage = (runners === null || runners === void 0 ? void 0 : runners.machineImage) ||
            aws_ec2_1.MachineImage.genericLinux({
                [scope.region]: runnersLookupMachineImage.imageId,
            });
        /**
         * GitLab Manager
         */
        const managerSecurityGroup = new aws_ec2_1.SecurityGroup(scope, "ManagerSecurityGroup", {
            vpc: this.network.vpc,
            description: "Security group for GitLab Runners Manager.",
        });
        managerSecurityGroup.connections.allowTo(runnersSecurityGroup, aws_ec2_1.Port.tcp(22), "SSH traffic from Manager");
        managerSecurityGroup.connections.allowTo(runnersSecurityGroup, aws_ec2_1.Port.tcp(2376), "SSH traffic from Docker");
        const managerInstanceType = (runners === null || runners === void 0 ? void 0 : runners.instanceType) || aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.NANO);
        const managerMachineImage = (runners === null || runners === void 0 ? void 0 : runners.machineImage) ||
            aws_ec2_1.MachineImage.latestAmazonLinux({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
                edition: aws_ec2_1.AmazonLinuxEdition.STANDARD,
                virtualization: aws_ec2_1.AmazonLinuxVirt.HVM,
                storage: aws_ec2_1.AmazonLinuxStorage.EBS,
                cpuType: aws_ec2_1.AmazonLinuxCpuType.X86_64,
            });
        const managerRole = new aws_iam_1.Role(scope, "ManagerRole", {
            assumedBy: ec2ServicePrincipal,
            managedPolicies: [ec2ManagedPolicyForSSM],
            inlinePolicies: {
                Cache: aws_iam_1.PolicyDocument.fromJson({
                    Version: "2012-10-17",
                    Statement: [
                        {
                            Effect: "Allow",
                            Action: ["s3:ListObjects*", "s3:GetObject*", "s3:DeleteObject*", "s3:PutObject*"],
                            Resource: [`${this.cacheBucket.bucketArn}/*`],
                        },
                        {
                            Effect: "Allow",
                            Action: ["s3:ListBucket"],
                            Resource: [`${this.cacheBucket.bucketArn}`],
                        },
                    ],
                }),
                Runners: aws_iam_1.PolicyDocument.fromJson({
                    Version: "2012-10-17",
                    Statement: [
                        {
                            Effect: "Allow",
                            Action: ["ec2:CreateKeyPair", "ec2:DeleteKeyPair", "ec2:ImportKeyPair", "ec2:Describe*"],
                            Resource: ["*"],
                        },
                        {
                            Effect: "Allow",
                            Action: ["ec2:CreateTags", "ssm:UpdateInstanceInformation"],
                            Resource: ["*"],
                            Condition: {
                                StringLike: {
                                    "aws:RequestTag/Name": "*gitlab-runner-*",
                                },
                                "ForAllValues:StringEquals": {
                                    "aws:TagKeys": ["Name"],
                                },
                            },
                        },
                        {
                            Effect: "Allow",
                            Action: ["ec2:RequestSpotInstances", "ec2:CancelSpotInstanceRequests"],
                            Resource: ["*"],
                            Condition: {
                                StringEqualsIfExists: {
                                    "ec2:Region": `${scope.region}`,
                                },
                                ArnEqualsIfExists: {
                                    "ec2:Vpc": `${this.network.vpc.vpcArn}`,
                                },
                            },
                        },
                        {
                            Effect: "Allow",
                            Action: ["ec2:RunInstances"],
                            Resource: ["*"],
                            Condition: {
                                StringEquals: {
                                    "ec2:InstanceType": [`${runnersInstanceType.toString()}`],
                                    "ec2:InstanceProfile": `${runnersInstanceProfile.ref}`,
                                },
                            },
                        },
                        {
                            Effect: "Allow",
                            Action: ["ec2:TerminateInstances", "ec2:StopInstances", "ec2:StartInstances", "ec2:RebootInstances"],
                            Resource: ["*"],
                            Condition: {
                                StringLike: {
                                    "ec2:ResourceTag/Name": "*gitlab-runner-*",
                                },
                            },
                        },
                        {
                            Effect: "Allow",
                            Action: ["iam:PassRole"],
                            Resource: [`${runnersRole.roleArn}`],
                        },
                    ],
                }),
            },
        });
        const userData = aws_ec2_1.UserData.forLinux({});
        userData.addCommands(`yum update -y aws-cfn-bootstrap` // !/bin/bash -xe
        );
        const gitlabRunnerConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        gitlabRunnerConfigRestartHandle._addFile("/etc/gitlab-runner/config.toml");
        const rsyslogConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        rsyslogConfigRestartHandle._addFile("/etc/rsyslog.d/25-gitlab-runner.conf");
        /**
         * Config set keys
         * @see {@link https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-init.html#aws-resource-init-configsets}
         */
        const REPOSITORIES = "repositories";
        const PACKAGES = "packages";
        const CONFIG = "config";
        const RESTART = "restart";
        const initConfig = aws_ec2_1.CloudFormationInit.fromConfigSets({
            configSets: {
                default: [REPOSITORIES, PACKAGES, CONFIG, RESTART],
            },
            configs: {
                [REPOSITORIES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://packages.gitlab.com/install/repositories/runner/gitlab-runner/script.rpm.sh | bash", { key: "10-gitlab-runner" }),
                ]),
                [PACKAGES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitPackage.yum("docker"),
                    aws_ec2_1.InitPackage.yum("gitlab-runner"),
                    aws_ec2_1.InitPackage.yum("tzdata"),
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://gitlab-docker-machine-downloads.s3.amazonaws.com/v0.16.2-gitlab.12/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine", 
                    //"curl -L https://github.com/docker/machine/releases/download/v0.16.2/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine",
                    { key: "10-docker-machine" }),
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner start", {
                        key: "20-gitlab-runner-start",
                    }),
                ]),
                [CONFIG]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitFile.fromString("/etc/gitlab-runner/config.toml", runner_configuration_1.ConfigurationMapper.withDefaults({
                        globalConfiguration: {
                            concurrent: props === null || props === void 0 ? void 0 : props.concurrent,
                            checkInterval: props === null || props === void 0 ? void 0 : props.checkInterval,
                            logFormat: props === null || props === void 0 ? void 0 : props.logFormat,
                            logLevel: props === null || props === void 0 ? void 0 : props.logLevel,
                        },
                        runnerConfiguration: {
                            token: props.gitlabToken,
                            url: props.gitlabUrl,
                            limit: runners === null || runners === void 0 ? void 0 : runners.limit,
                            outputLimit: runners === null || runners === void 0 ? void 0 : runners.outputLimit,
                            environment: runners === null || runners === void 0 ? void 0 : runners.environment,
                        },
                        dockerConfiguration: {
                            ...runners === null || runners === void 0 ? void 0 : runners.docker,
                        },
                        machineConfiguration: {
                            ...runners === null || runners === void 0 ? void 0 : runners.machine,
                            machineOptions: {
                                ...(_b = runners === null || runners === void 0 ? void 0 : runners.machine) === null || _b === void 0 ? void 0 : _b.machineOptions,
                                instanceType: (_c = runners === null || runners === void 0 ? void 0 : runners.instanceType) === null || _c === void 0 ? void 0 : _c.toString(),
                                ami: (_d = runners === null || runners === void 0 ? void 0 : runners.machineImage) === null || _d === void 0 ? void 0 : _d.getImage(scope).imageId,
                                region: scope.region,
                                vpcId: this.network.vpc.vpcId,
                                zone: this.network.availabilityZone.slice(-1),
                                subnetId: this.network.subnet.subnetId,
                                securityGroup: `${runnersSecurityGroupName}`,
                                usePrivateAddress: true,
                                iamInstanceProfile: `${runnersInstanceProfile.ref}`,
                            },
                        },
                        autoscalingConfigurations: (runners === null || runners === void 0 ? void 0 : runners.autoscaling) || [],
                        cacheConfiguration: {
                            s3: {
                                serverAddress: `s3.${scope.urlSuffix}`,
                                bucketName: `${this.cacheBucket.bucketName}`,
                                bucketLocation: `${scope.region}`,
                            },
                        },
                    }).toToml(), {
                        owner: "gitlab-runner",
                        group: "gitlab-runner",
                        mode: "000600",
                    }),
                    aws_ec2_1.InitFile.fromString("/etc/rsyslog.d/25-gitlab-runner.conf", `:programname, isequal, "gitlab-runner" /var/log/gitlab-runner.log`, {
                        owner: "root",
                        group: "root",
                        mode: "000644",
                    }),
                    aws_ec2_1.InitService.enable("gitlab-runner", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: gitlabRunnerConfigRestartHandle,
                    }),
                    aws_ec2_1.InitService.enable("rsyslog", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: rsyslogConfigRestartHandle,
                    }),
                ]),
                [RESTART]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner restart", {
                        key: "10-gitlab-runner-restart",
                    }),
                ]),
            },
        });
        const managerAutoScalingGroup = new aws_autoscaling_1.AutoScalingGroup(scope, "ManagerAutoscalingGroup", {
            vpc: this.network.vpc,
            vpcSubnets: {
                subnets: [this.network.subnet],
            },
            instanceType: managerInstanceType,
            machineImage: managerMachineImage,
            keyName: manager === null || manager === void 0 ? void 0 : manager.keyPairName,
            securityGroup: managerSecurityGroup,
            role: managerRole,
            userData: userData,
            init: initConfig,
            initOptions: {
                ignoreFailures: false,
            },
            maxCapacity: 1,
            minCapacity: 1,
            desiredCapacity: 1,
            signals: aws_autoscaling_1.Signals.waitForCount(1, { timeout: core_1.Duration.minutes(15) }),
        });
        this.runners = {
            securityGroupName: runnersSecurityGroupName,
            securityGroup: runnersSecurityGroup,
            role: runnersRole,
            instanceProfile: runnersInstanceProfile,
            instanceType: runnersInstanceType,
            machineImage: runnersMachineImage,
        };
        this.manager = {
            securityGroup: managerSecurityGroup,
            instanceType: managerInstanceType,
            machineImage: managerMachineImage,
            autoScalingGroup: managerAutoScalingGroup,
            role: managerRole,
        };
    }
}
exports.GitlabRunnerAutoscaling = GitlabRunnerAutoscaling;
_a = JSII_RTTI_SYMBOL_1;
GitlabRunnerAutoscaling[_a] = { fqn: "@pepperize-testing/cdk-autoscaling-gitlab-runner.GitlabRunnerAutoscaling", version: "0.0.70" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicnVubmVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL3J1bm5lci9ydW5uZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSw4REFBd0Y7QUFDeEYsOENBdUIwQjtBQUMxQiw4Q0FBb0g7QUFFcEgsd0NBQTJEO0FBQzNELGtFQU1pQztBQUNqQyxtQ0FBNEM7QUFDNUMsdUNBQWtEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7OztBQXVFbEQsTUFBYSx1QkFBd0IsU0FBUSxnQkFBUzs7OztJQVNwRCxZQUFZLEtBQVksRUFBRSxFQUFVLEVBQUUsS0FBbUM7O1FBQ3ZFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDakIsTUFBTSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsT0FBTyxFQUFFLE9BQU8sRUFBRSxHQUFpQyxLQUFLLENBQUM7UUFFakY7O1dBRUc7UUFDSCxJQUFJLENBQUMsV0FBVyxHQUFHLENBQUEsS0FBSyxhQUFMLEtBQUssdUJBQUwsS0FBSyxDQUFFLE1BQU0sS0FBSSxJQUFJLGFBQUssQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFLEtBQUssYUFBTCxLQUFLLHVCQUFMLEtBQUssQ0FBRSxPQUFPLENBQUMsQ0FBQyxNQUFNLENBQUM7UUFFckY7O1dBRUc7UUFDSCxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksaUJBQU8sQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRXREOztXQUVHO1FBQ0gsTUFBTSxtQkFBbUIsR0FBRyxJQUFJLDBCQUFnQixDQUFDLG1CQUFtQixFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQzFFLE1BQU0sc0JBQXNCLEdBQUcsdUJBQWEsQ0FBQyxvQkFBb0IsQ0FDL0QsS0FBSyxFQUNMLHFCQUFxQixFQUNyQiwwREFBMEQsQ0FDM0QsQ0FBQztRQUVGOztXQUVHO1FBQ0gsTUFBTSx3QkFBd0IsR0FBRyxHQUFHLEtBQUssQ0FBQyxTQUFTLHVCQUF1QixDQUFDO1FBQzNFLE1BQU0sb0JBQW9CLEdBQUcsSUFBSSx1QkFBYSxDQUFDLEtBQUssRUFBRSxzQkFBc0IsRUFBRTtZQUM1RSxpQkFBaUIsRUFBRSx3QkFBd0I7WUFDM0MsV0FBVyxFQUFFLG9DQUFvQztZQUNqRCxHQUFHLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHO1NBQ3RCLENBQUMsQ0FBQztRQUVILE1BQU0sV0FBVyxHQUNmLENBQUEsT0FBTyxhQUFQLE9BQU8sdUJBQVAsT0FBTyxDQUFFLElBQUk7WUFDYixJQUFJLGNBQUksQ0FBQyxLQUFLLEVBQUUsYUFBYSxFQUFFO2dCQUM3QixTQUFTLEVBQUUsbUJBQW1CO2dCQUM5QixlQUFlLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQzthQUMxQyxDQUFDLENBQUM7UUFFTCxNQUFNLHNCQUFzQixHQUFHLElBQUksNEJBQWtCLENBQUMsS0FBSyxFQUFFLHdCQUF3QixFQUFFO1lBQ3JGLEtBQUssRUFBRSxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUM7U0FDOUIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxtQkFBbUIsR0FBRyxDQUFBLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxZQUFZLEtBQUksc0JBQVksQ0FBQyxFQUFFLENBQUMsdUJBQWEsQ0FBQyxFQUFFLEVBQUUsc0JBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUUzRyxNQUFNLHlCQUF5QixHQUFHLElBQUksNEJBQWtCLENBQUM7WUFDdkQsSUFBSSxFQUFFLHlEQUF5RDtZQUMvRCxNQUFNLEVBQUUsQ0FBQyxjQUFjLENBQUM7WUFDeEIsT0FBTyxFQUFFO2dCQUNQLFlBQVksRUFBRSxDQUFDLFFBQVEsQ0FBQztnQkFDeEIsWUFBWSxFQUFFLENBQUMsU0FBUyxDQUFDO2dCQUN6QixLQUFLLEVBQUUsQ0FBQyxXQUFXLENBQUM7Z0JBQ3BCLGtCQUFrQixFQUFFLENBQUMsS0FBSyxDQUFDO2dCQUMzQixxQkFBcUIsRUFBRSxDQUFDLEtBQUssQ0FBQzthQUMvQjtTQUNGLENBQUMsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFbEIsTUFBTSxtQkFBbUIsR0FDdkIsQ0FBQSxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsWUFBWTtZQUNyQixzQkFBWSxDQUFDLFlBQVksQ0FBQztnQkFDeEIsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEVBQUUseUJBQXlCLENBQUMsT0FBTzthQUNsRCxDQUFDLENBQUM7UUFFTDs7V0FFRztRQUNILE1BQU0sb0JBQW9CLEdBQUcsSUFBSSx1QkFBYSxDQUFDLEtBQUssRUFBRSxzQkFBc0IsRUFBRTtZQUM1RSxHQUFHLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHO1lBQ3JCLFdBQVcsRUFBRSw0Q0FBNEM7U0FDMUQsQ0FBQyxDQUFDO1FBQ0gsb0JBQW9CLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsRUFBRSxjQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFLDBCQUEwQixDQUFDLENBQUM7UUFDekcsb0JBQW9CLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsRUFBRSxjQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLHlCQUF5QixDQUFDLENBQUM7UUFFMUcsTUFBTSxtQkFBbUIsR0FBRyxDQUFBLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxZQUFZLEtBQUksc0JBQVksQ0FBQyxFQUFFLENBQUMsdUJBQWEsQ0FBQyxFQUFFLEVBQUUsc0JBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUUxRyxNQUFNLG1CQUFtQixHQUN2QixDQUFBLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxZQUFZO1lBQ3JCLHNCQUFZLENBQUMsaUJBQWlCLENBQUM7Z0JBQzdCLFVBQVUsRUFBRSwrQkFBcUIsQ0FBQyxjQUFjO2dCQUNoRCxPQUFPLEVBQUUsNEJBQWtCLENBQUMsUUFBUTtnQkFDcEMsY0FBYyxFQUFFLHlCQUFlLENBQUMsR0FBRztnQkFDbkMsT0FBTyxFQUFFLDRCQUFrQixDQUFDLEdBQUc7Z0JBQy9CLE9BQU8sRUFBRSw0QkFBa0IsQ0FBQyxNQUFNO2FBQ25DLENBQUMsQ0FBQztRQUVMLE1BQU0sV0FBVyxHQUFHLElBQUksY0FBSSxDQUFDLEtBQUssRUFBRSxhQUFhLEVBQUU7WUFDakQsU0FBUyxFQUFFLG1CQUFtQjtZQUM5QixlQUFlLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQztZQUN6QyxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLHdCQUFjLENBQUMsUUFBUSxDQUFDO29CQUM3QixPQUFPLEVBQUUsWUFBWTtvQkFDckIsU0FBUyxFQUFFO3dCQUNUOzRCQUNFLE1BQU0sRUFBRSxPQUFPOzRCQUNmLE1BQU0sRUFBRSxDQUFDLGlCQUFpQixFQUFFLGVBQWUsRUFBRSxrQkFBa0IsRUFBRSxlQUFlLENBQUM7NEJBQ2pGLFFBQVEsRUFBRSxDQUFDLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxTQUFTLElBQUksQ0FBQzt5QkFDOUM7d0JBQ0Q7NEJBQ0UsTUFBTSxFQUFFLE9BQU87NEJBQ2YsTUFBTSxFQUFFLENBQUMsZUFBZSxDQUFDOzRCQUN6QixRQUFRLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsU0FBUyxFQUFFLENBQUM7eUJBQzVDO3FCQUNGO2lCQUNGLENBQUM7Z0JBQ0YsT0FBTyxFQUFFLHdCQUFjLENBQUMsUUFBUSxDQUFDO29CQUMvQixPQUFPLEVBQUUsWUFBWTtvQkFDckIsU0FBUyxFQUFFO3dCQUNUOzRCQUNFLE1BQU0sRUFBRSxPQUFPOzRCQUNmLE1BQU0sRUFBRSxDQUFDLG1CQUFtQixFQUFFLG1CQUFtQixFQUFFLG1CQUFtQixFQUFFLGVBQWUsQ0FBQzs0QkFDeEYsUUFBUSxFQUFFLENBQUMsR0FBRyxDQUFDO3lCQUNoQjt3QkFDRDs0QkFDRSxNQUFNLEVBQUUsT0FBTzs0QkFDZixNQUFNLEVBQUUsQ0FBQyxnQkFBZ0IsRUFBRSwrQkFBK0IsQ0FBQzs0QkFDM0QsUUFBUSxFQUFFLENBQUMsR0FBRyxDQUFDOzRCQUNmLFNBQVMsRUFBRTtnQ0FDVCxVQUFVLEVBQUU7b0NBQ1YscUJBQXFCLEVBQUUsa0JBQWtCO2lDQUMxQztnQ0FDRCwyQkFBMkIsRUFBRTtvQ0FDM0IsYUFBYSxFQUFFLENBQUMsTUFBTSxDQUFDO2lDQUN4Qjs2QkFDRjt5QkFDRjt3QkFDRDs0QkFDRSxNQUFNLEVBQUUsT0FBTzs0QkFDZixNQUFNLEVBQUUsQ0FBQywwQkFBMEIsRUFBRSxnQ0FBZ0MsQ0FBQzs0QkFDdEUsUUFBUSxFQUFFLENBQUMsR0FBRyxDQUFDOzRCQUNmLFNBQVMsRUFBRTtnQ0FDVCxvQkFBb0IsRUFBRTtvQ0FDcEIsWUFBWSxFQUFFLEdBQUcsS0FBSyxDQUFDLE1BQU0sRUFBRTtpQ0FDaEM7Z0NBQ0QsaUJBQWlCLEVBQUU7b0NBQ2pCLFNBQVMsRUFBRSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLE1BQU0sRUFBRTtpQ0FDeEM7NkJBQ0Y7eUJBQ0Y7d0JBQ0Q7NEJBQ0UsTUFBTSxFQUFFLE9BQU87NEJBQ2YsTUFBTSxFQUFFLENBQUMsa0JBQWtCLENBQUM7NEJBQzVCLFFBQVEsRUFBRSxDQUFDLEdBQUcsQ0FBQzs0QkFDZixTQUFTLEVBQUU7Z0NBQ1QsWUFBWSxFQUFFO29DQUNaLGtCQUFrQixFQUFFLENBQUMsR0FBRyxtQkFBbUIsQ0FBQyxRQUFRLEVBQUUsRUFBRSxDQUFDO29DQUN6RCxxQkFBcUIsRUFBRSxHQUFHLHNCQUFzQixDQUFDLEdBQUcsRUFBRTtpQ0FDdkQ7NkJBQ0Y7eUJBQ0Y7d0JBQ0Q7NEJBQ0UsTUFBTSxFQUFFLE9BQU87NEJBQ2YsTUFBTSxFQUFFLENBQUMsd0JBQXdCLEVBQUUsbUJBQW1CLEVBQUUsb0JBQW9CLEVBQUUscUJBQXFCLENBQUM7NEJBQ3BHLFFBQVEsRUFBRSxDQUFDLEdBQUcsQ0FBQzs0QkFDZixTQUFTLEVBQUU7Z0NBQ1QsVUFBVSxFQUFFO29DQUNWLHNCQUFzQixFQUFFLGtCQUFrQjtpQ0FDM0M7NkJBQ0Y7eUJBQ0Y7d0JBQ0Q7NEJBQ0UsTUFBTSxFQUFFLE9BQU87NEJBQ2YsTUFBTSxFQUFFLENBQUMsY0FBYyxDQUFDOzRCQUN4QixRQUFRLEVBQUUsQ0FBQyxHQUFHLFdBQVcsQ0FBQyxPQUFPLEVBQUUsQ0FBQzt5QkFDckM7cUJBQ0Y7aUJBQ0YsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxRQUFRLEdBQUcsa0JBQVEsQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDdkMsUUFBUSxDQUFDLFdBQVcsQ0FDbEIsaUNBQWlDLENBQUMsaUJBQWlCO1NBQ3BELENBQUM7UUFFRixNQUFNLCtCQUErQixHQUFHLElBQUksa0NBQXdCLEVBQUUsQ0FBQztRQUN2RSwrQkFBK0IsQ0FBQyxRQUFRLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUUzRSxNQUFNLDBCQUEwQixHQUFHLElBQUksa0NBQXdCLEVBQUUsQ0FBQztRQUNsRSwwQkFBMEIsQ0FBQyxRQUFRLENBQUMsc0NBQXNDLENBQUMsQ0FBQztRQUU1RTs7O1dBR0c7UUFDSCxNQUFNLFlBQVksR0FBRyxjQUFjLENBQUM7UUFDcEMsTUFBTSxRQUFRLEdBQUcsVUFBVSxDQUFDO1FBQzVCLE1BQU0sTUFBTSxHQUFHLFFBQVEsQ0FBQztRQUN4QixNQUFNLE9BQU8sR0FBRyxTQUFTLENBQUM7UUFFMUIsTUFBTSxVQUFVLEdBQUcsNEJBQWtCLENBQUMsY0FBYyxDQUFDO1lBQ25ELFVBQVUsRUFBRTtnQkFDVixPQUFPLEVBQUUsQ0FBQyxZQUFZLEVBQUUsUUFBUSxFQUFFLE1BQU0sRUFBRSxPQUFPLENBQUM7YUFDbkQ7WUFDRCxPQUFPLEVBQUU7Z0JBQ1AsQ0FBQyxZQUFZLENBQUMsRUFBRSxJQUFJLG9CQUFVLENBQUM7b0JBQzdCLHFCQUFXLENBQUMsWUFBWSxDQUN0QixvR0FBb0csRUFDcEcsRUFBRSxHQUFHLEVBQUUsa0JBQWtCLEVBQUUsQ0FDNUI7aUJBQ0YsQ0FBQztnQkFDRixDQUFDLFFBQVEsQ0FBQyxFQUFFLElBQUksb0JBQVUsQ0FBQztvQkFDekIscUJBQVcsQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDO29CQUN6QixxQkFBVyxDQUFDLEdBQUcsQ0FBQyxlQUFlLENBQUM7b0JBQ2hDLHFCQUFXLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQztvQkFDekIscUJBQVcsQ0FBQyxZQUFZLENBQ3RCLHNNQUFzTTtvQkFDdE0sMExBQTBMO29CQUMxTCxFQUFFLEdBQUcsRUFBRSxtQkFBbUIsRUFBRSxDQUM3QjtvQkFDRCxxQkFBVyxDQUFDLFlBQVksQ0FBQyxxQkFBcUIsRUFBRTt3QkFDOUMsR0FBRyxFQUFFLHdCQUF3QjtxQkFDOUIsQ0FBQztpQkFDSCxDQUFDO2dCQUNGLENBQUMsTUFBTSxDQUFDLEVBQUUsSUFBSSxvQkFBVSxDQUFDO29CQUN2QixrQkFBUSxDQUFDLFVBQVUsQ0FDakIsZ0NBQWdDLEVBQ2hDLDBDQUFtQixDQUFDLFlBQVksQ0FBQzt3QkFDL0IsbUJBQW1CLEVBQUU7NEJBQ25CLFVBQVUsRUFBRSxLQUFLLGFBQUwsS0FBSyx1QkFBTCxLQUFLLENBQUUsVUFBVTs0QkFDN0IsYUFBYSxFQUFFLEtBQUssYUFBTCxLQUFLLHVCQUFMLEtBQUssQ0FBRSxhQUFhOzRCQUNuQyxTQUFTLEVBQUUsS0FBSyxhQUFMLEtBQUssdUJBQUwsS0FBSyxDQUFFLFNBQVM7NEJBQzNCLFFBQVEsRUFBRSxLQUFLLGFBQUwsS0FBSyx1QkFBTCxLQUFLLENBQUUsUUFBUTt5QkFDMUI7d0JBQ0QsbUJBQW1CLEVBQUU7NEJBQ25CLEtBQUssRUFBRSxLQUFLLENBQUMsV0FBVzs0QkFDeEIsR0FBRyxFQUFFLEtBQUssQ0FBQyxTQUFTOzRCQUNwQixLQUFLLEVBQUUsT0FBTyxhQUFQLE9BQU8sdUJBQVAsT0FBTyxDQUFFLEtBQUs7NEJBQ3JCLFdBQVcsRUFBRSxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsV0FBVzs0QkFDakMsV0FBVyxFQUFFLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxXQUFXO3lCQUNsQzt3QkFDRCxtQkFBbUIsRUFBRTs0QkFDbkIsR0FBRyxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsTUFBTTt5QkFDbkI7d0JBQ0Qsb0JBQW9CLEVBQUU7NEJBQ3BCLEdBQUcsT0FBTyxhQUFQLE9BQU8sdUJBQVAsT0FBTyxDQUFFLE9BQU87NEJBQ25CLGNBQWMsRUFBRTtnQ0FDZCxTQUFHLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxPQUFPLDBDQUFFLGNBQWM7Z0NBQ25DLFlBQVksUUFBRSxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsWUFBWSwwQ0FBRSxRQUFRLEVBQUU7Z0NBQy9DLEdBQUcsUUFBRSxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsWUFBWSwwQ0FBRSxRQUFRLENBQUMsS0FBSyxFQUFFLE9BQU87Z0NBQ25ELE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTTtnQ0FDcEIsS0FBSyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEtBQUs7Z0NBQzdCLElBQUksRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztnQ0FDN0MsUUFBUSxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLFFBQVE7Z0NBQ3RDLGFBQWEsRUFBRSxHQUFHLHdCQUF3QixFQUFFO2dDQUM1QyxpQkFBaUIsRUFBRSxJQUFJO2dDQUN2QixrQkFBa0IsRUFBRSxHQUFHLHNCQUFzQixDQUFDLEdBQUcsRUFBRTs2QkFDcEQ7eUJBQ0Y7d0JBQ0QseUJBQXlCLEVBQUUsQ0FBQSxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsV0FBVyxLQUFJLEVBQUU7d0JBQ3JELGtCQUFrQixFQUFFOzRCQUNsQixFQUFFLEVBQUU7Z0NBQ0YsYUFBYSxFQUFFLE1BQU0sS0FBSyxDQUFDLFNBQVMsRUFBRTtnQ0FDdEMsVUFBVSxFQUFFLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxVQUFVLEVBQUU7Z0NBQzVDLGNBQWMsRUFBRSxHQUFHLEtBQUssQ0FBQyxNQUFNLEVBQUU7NkJBQ2xDO3lCQUNGO3FCQUNGLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFDWDt3QkFDRSxLQUFLLEVBQUUsZUFBZTt3QkFDdEIsS0FBSyxFQUFFLGVBQWU7d0JBQ3RCLElBQUksRUFBRSxRQUFRO3FCQUNmLENBQ0Y7b0JBQ0Qsa0JBQVEsQ0FBQyxVQUFVLENBQ2pCLHNDQUFzQyxFQUN0QyxtRUFBbUUsRUFDbkU7d0JBQ0UsS0FBSyxFQUFFLE1BQU07d0JBQ2IsS0FBSyxFQUFFLE1BQU07d0JBQ2IsSUFBSSxFQUFFLFFBQVE7cUJBQ2YsQ0FDRjtvQkFDRCxxQkFBVyxDQUFDLE1BQU0sQ0FBQyxlQUFlLEVBQUU7d0JBQ2xDLGFBQWEsRUFBRSxJQUFJO3dCQUNuQixPQUFPLEVBQUUsSUFBSTt3QkFDYixvQkFBb0IsRUFBRSwrQkFBK0I7cUJBQ3RELENBQUM7b0JBQ0YscUJBQVcsQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFO3dCQUM1QixhQUFhLEVBQUUsSUFBSTt3QkFDbkIsT0FBTyxFQUFFLElBQUk7d0JBQ2Isb0JBQW9CLEVBQUUsMEJBQTBCO3FCQUNqRCxDQUFDO2lCQUNILENBQUM7Z0JBQ0YsQ0FBQyxPQUFPLENBQUMsRUFBRSxJQUFJLG9CQUFVLENBQUM7b0JBQ3hCLHFCQUFXLENBQUMsWUFBWSxDQUFDLHVCQUF1QixFQUFFO3dCQUNoRCxHQUFHLEVBQUUsMEJBQTBCO3FCQUNoQyxDQUFDO2lCQUNILENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUVILE1BQU0sdUJBQXVCLEdBQUcsSUFBSSxrQ0FBZ0IsQ0FBQyxLQUFLLEVBQUUseUJBQXlCLEVBQUU7WUFDckYsR0FBRyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRztZQUNyQixVQUFVLEVBQUU7Z0JBQ1YsT0FBTyxFQUFFLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUM7YUFDL0I7WUFDRCxZQUFZLEVBQUUsbUJBQW1CO1lBQ2pDLFlBQVksRUFBRSxtQkFBbUI7WUFDakMsT0FBTyxFQUFFLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxXQUFXO1lBQzdCLGFBQWEsRUFBRSxvQkFBb0I7WUFDbkMsSUFBSSxFQUFFLFdBQVc7WUFDakIsUUFBUSxFQUFFLFFBQVE7WUFDbEIsSUFBSSxFQUFFLFVBQVU7WUFDaEIsV0FBVyxFQUFFO2dCQUNYLGNBQWMsRUFBRSxLQUFLO2FBQ3RCO1lBQ0QsV0FBVyxFQUFFLENBQUM7WUFDZCxXQUFXLEVBQUUsQ0FBQztZQUNkLGVBQWUsRUFBRSxDQUFDO1lBQ2xCLE9BQU8sRUFBRSx5QkFBTyxDQUFDLFlBQVksQ0FBQyxDQUFDLEVBQUUsRUFBRSxPQUFPLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDO1NBQ3BFLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxPQUFPLEdBQUc7WUFDYixpQkFBaUIsRUFBRSx3QkFBd0I7WUFDM0MsYUFBYSxFQUFFLG9CQUFvQjtZQUNuQyxJQUFJLEVBQUUsV0FBVztZQUNqQixlQUFlLEVBQUUsc0JBQXNCO1lBQ3ZDLFlBQVksRUFBRSxtQkFBbUI7WUFDakMsWUFBWSxFQUFFLG1CQUFtQjtTQUNsQyxDQUFDO1FBRUYsSUFBSSxDQUFDLE9BQU8sR0FBRztZQUNiLGFBQWEsRUFBRSxvQkFBb0I7WUFDbkMsWUFBWSxFQUFFLG1CQUFtQjtZQUNqQyxZQUFZLEVBQUUsbUJBQW1CO1lBQ2pDLGdCQUFnQixFQUFFLHVCQUF1QjtZQUN6QyxJQUFJLEVBQUUsV0FBVztTQUNsQixDQUFDO0lBQ0osQ0FBQzs7QUFuVkgsMERBb1ZDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQXV0b1NjYWxpbmdHcm91cCwgSUF1dG9TY2FsaW5nR3JvdXAsIFNpZ25hbHMgfSBmcm9tIFwiQGF3cy1jZGsvYXdzLWF1dG9zY2FsaW5nXCI7XG5pbXBvcnQge1xuICBBbWF6b25MaW51eENwdVR5cGUsXG4gIEFtYXpvbkxpbnV4RWRpdGlvbixcbiAgQW1hem9uTGludXhHZW5lcmF0aW9uLFxuICBBbWF6b25MaW51eFN0b3JhZ2UsXG4gIEFtYXpvbkxpbnV4VmlydCxcbiAgQ2xvdWRGb3JtYXRpb25Jbml0LFxuICBJTWFjaGluZUltYWdlLFxuICBJbml0Q29tbWFuZCxcbiAgSW5pdENvbmZpZyxcbiAgSW5pdEZpbGUsXG4gIEluaXRQYWNrYWdlLFxuICBJbml0U2VydmljZSxcbiAgSW5pdFNlcnZpY2VSZXN0YXJ0SGFuZGxlLFxuICBJbnN0YW5jZUNsYXNzLFxuICBJbnN0YW5jZVNpemUsXG4gIEluc3RhbmNlVHlwZSxcbiAgSVNlY3VyaXR5R3JvdXAsXG4gIExvb2t1cE1hY2hpbmVJbWFnZSxcbiAgTWFjaGluZUltYWdlLFxuICBQb3J0LFxuICBTZWN1cml0eUdyb3VwLFxuICBVc2VyRGF0YSxcbn0gZnJvbSBcIkBhd3MtY2RrL2F3cy1lYzJcIjtcbmltcG9ydCB7IENmbkluc3RhbmNlUHJvZmlsZSwgSVJvbGUsIE1hbmFnZWRQb2xpY3ksIFBvbGljeURvY3VtZW50LCBSb2xlLCBTZXJ2aWNlUHJpbmNpcGFsIH0gZnJvbSBcIkBhd3MtY2RrL2F3cy1pYW1cIjtcbmltcG9ydCB7IElCdWNrZXQgfSBmcm9tIFwiQGF3cy1jZGsvYXdzLXMzXCI7XG5pbXBvcnQgeyBDb25zdHJ1Y3QsIER1cmF0aW9uLCBTdGFjayB9IGZyb20gXCJAYXdzLWNkay9jb3JlXCI7XG5pbXBvcnQge1xuICBBdXRvc2NhbGluZ0NvbmZpZ3VyYXRpb24sXG4gIENvbmZpZ3VyYXRpb25NYXBwZXIsXG4gIERvY2tlckNvbmZpZ3VyYXRpb24sXG4gIEdsb2JhbENvbmZpZ3VyYXRpb24sXG4gIE1hY2hpbmVDb25maWd1cmF0aW9uLFxufSBmcm9tIFwiLi4vcnVubmVyLWNvbmZpZ3VyYXRpb25cIjtcbmltcG9ydCB7IENhY2hlLCBDYWNoZVByb3BzIH0gZnJvbSBcIi4vY2FjaGVcIjtcbmltcG9ydCB7IE5ldHdvcmssIE5ldHdvcmtQcm9wcyB9IGZyb20gXCIuL25ldHdvcmtcIjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgR2l0bGFiUnVubmVyQXV0b3NjYWxpbmdQcm9wcyBleHRlbmRzIEdsb2JhbENvbmZpZ3VyYXRpb24ge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBnaXRsYWJUb2tlbjogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZ2l0bGFiVXJsPzogc3RyaW5nO1xuXG4gIHJlYWRvbmx5IGNhY2hlPzogR2l0bGFiUnVubmVyQXV0b3NjYWxpbmdDYWNoZVByb3BzO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG5ldHdvcms/OiBOZXR3b3JrUHJvcHM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbWFuYWdlcj86IEdpdGxhYlJ1bm5lckF1dG9zY2FsaW5nTWFuYWdlclByb3BzO1xuXG4gIHJlYWRvbmx5IHJ1bm5lcnM/OiBHaXRsYWJSdW5uZXJBdXRvc2NhbGluZ1J1bm5lclByb3BzO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEdpdGxhYlJ1bm5lckF1dG9zY2FsaW5nQ2FjaGVQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBidWNrZXQ/OiBJQnVja2V0O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBvcHRpb25zPzogQ2FjaGVQcm9wcztcbn1cblxuZXhwb3J0IGludGVyZmFjZSBHaXRsYWJSdW5uZXJBdXRvc2NhbGluZ01hbmFnZXJQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG1hY2hpbmVJbWFnZT86IElNYWNoaW5lSW1hZ2U7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaW5zdGFuY2VUeXBlPzogSW5zdGFuY2VUeXBlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkga2V5UGFpck5hbWU/OiBzdHJpbmc7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgR2l0bGFiUnVubmVyQXV0b3NjYWxpbmdSdW5uZXJQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpbnN0YW5jZVR5cGU/OiBJbnN0YW5jZVR5cGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtYWNoaW5lSW1hZ2U/OiBJTWFjaGluZUltYWdlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJvbGU/OiBJUm9sZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGxpbWl0PzogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG91dHB1dExpbWl0PzogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW52aXJvbm1lbnQ/OiBzdHJpbmdbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkb2NrZXI/OiBEb2NrZXJDb25maWd1cmF0aW9uO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG1hY2hpbmU/OiBNYWNoaW5lQ29uZmlndXJhdGlvbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhdXRvc2NhbGluZz86IEF1dG9zY2FsaW5nQ29uZmlndXJhdGlvbltdO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBHaXRsYWJSdW5uZXJBdXRvc2NhbGluZyBleHRlbmRzIENvbnN0cnVjdCB7XG4gIHJlYWRvbmx5IG5ldHdvcms6IE5ldHdvcms7XG5cbiAgcmVhZG9ubHkgY2FjaGVCdWNrZXQ6IElCdWNrZXQ7XG5cbiAgcmVhZG9ubHkgcnVubmVyczogR2l0bGFiUnVubmVyQXV0b3NjYWxpbmdSdW5uZXJzO1xuXG4gIHJlYWRvbmx5IG1hbmFnZXI6IEdpdGxhYlJ1bm5lckF1dG9zY2FsaW5nTWFuYWdlcjtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogU3RhY2ssIGlkOiBzdHJpbmcsIHByb3BzOiBHaXRsYWJSdW5uZXJBdXRvc2NhbGluZ1Byb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICBjb25zdCB7IG1hbmFnZXIsIGNhY2hlLCBydW5uZXJzLCBuZXR3b3JrIH06IEdpdGxhYlJ1bm5lckF1dG9zY2FsaW5nUHJvcHMgPSBwcm9wcztcblxuICAgIC8qKlxuICAgICAqIFMzIEJ1Y2tldCBmb3IgUnVubmVycycgY2FjaGVcbiAgICAgKi9cbiAgICB0aGlzLmNhY2hlQnVja2V0ID0gY2FjaGU/LmJ1Y2tldCB8fCBuZXcgQ2FjaGUoc2NvcGUsIFwiQ2FjaGVcIiwgY2FjaGU/Lm9wdGlvbnMpLmJ1Y2tldDtcblxuICAgIC8qKlxuICAgICAqIE5ldHdvcmtcbiAgICAgKi9cbiAgICB0aGlzLm5ldHdvcmsgPSBuZXcgTmV0d29yayhzY29wZSwgXCJOZXR3b3JrXCIsIG5ldHdvcmspO1xuXG4gICAgLyoqXG4gICAgICogSUFNXG4gICAgICovXG4gICAgY29uc3QgZWMyU2VydmljZVByaW5jaXBhbCA9IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKFwiZWMyLmFtYXpvbmF3cy5jb21cIiwge30pO1xuICAgIGNvbnN0IGVjMk1hbmFnZWRQb2xpY3lGb3JTU00gPSBNYW5hZ2VkUG9saWN5LmZyb21NYW5hZ2VkUG9saWN5QXJuKFxuICAgICAgc2NvcGUsXG4gICAgICBcIkFtYXpvbkVDMlJvbGVmb3JTU01cIixcbiAgICAgIFwiYXJuOmF3czppYW06OmF3czpwb2xpY3kvc2VydmljZS1yb2xlL0FtYXpvbkVDMlJvbGVmb3JTU01cIlxuICAgICk7XG5cbiAgICAvKipcbiAgICAgKiBHaXRMYWIgUnVubmVyc1xuICAgICAqL1xuICAgIGNvbnN0IHJ1bm5lcnNTZWN1cml0eUdyb3VwTmFtZSA9IGAke3Njb3BlLnN0YWNrTmFtZX0tUnVubmVyc1NlY3VyaXR5R3JvdXBgO1xuICAgIGNvbnN0IHJ1bm5lcnNTZWN1cml0eUdyb3VwID0gbmV3IFNlY3VyaXR5R3JvdXAoc2NvcGUsIFwiUnVubmVyc1NlY3VyaXR5R3JvdXBcIiwge1xuICAgICAgc2VjdXJpdHlHcm91cE5hbWU6IHJ1bm5lcnNTZWN1cml0eUdyb3VwTmFtZSxcbiAgICAgIGRlc2NyaXB0aW9uOiBcIlNlY3VyaXR5IGdyb3VwIGZvciBHaXRMYWIgUnVubmVycy5cIixcbiAgICAgIHZwYzogdGhpcy5uZXR3b3JrLnZwYyxcbiAgICB9KTtcblxuICAgIGNvbnN0IHJ1bm5lcnNSb2xlID1cbiAgICAgIHJ1bm5lcnM/LnJvbGUgfHxcbiAgICAgIG5ldyBSb2xlKHNjb3BlLCBcIlJ1bm5lcnNSb2xlXCIsIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBlYzJTZXJ2aWNlUHJpbmNpcGFsLFxuICAgICAgICBtYW5hZ2VkUG9saWNpZXM6IFtlYzJNYW5hZ2VkUG9saWN5Rm9yU1NNXSxcbiAgICAgIH0pO1xuXG4gICAgY29uc3QgcnVubmVyc0luc3RhbmNlUHJvZmlsZSA9IG5ldyBDZm5JbnN0YW5jZVByb2ZpbGUoc2NvcGUsIFwiUnVubmVyc0luc3RhbmNlUHJvZmlsZVwiLCB7XG4gICAgICByb2xlczogW3J1bm5lcnNSb2xlLnJvbGVOYW1lXSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHJ1bm5lcnNJbnN0YW5jZVR5cGUgPSBydW5uZXJzPy5pbnN0YW5jZVR5cGUgfHwgSW5zdGFuY2VUeXBlLm9mKEluc3RhbmNlQ2xhc3MuVDMsIEluc3RhbmNlU2l6ZS5NSUNSTyk7XG5cbiAgICBjb25zdCBydW5uZXJzTG9va3VwTWFjaGluZUltYWdlID0gbmV3IExvb2t1cE1hY2hpbmVJbWFnZSh7XG4gICAgICBuYW1lOiBcInVidW50dS9pbWFnZXMvaHZtLXNzZC91YnVudHUtZm9jYWwtMjAuMDQtYW1kNjQtc2VydmVyLSpcIixcbiAgICAgIG93bmVyczogW1wiMDk5NzIwMTA5NDc3XCJdLFxuICAgICAgZmlsdGVyczoge1xuICAgICAgICBhcmNoaXRlY3R1cmU6IFtcIng4Nl82NFwiXSxcbiAgICAgICAgXCJpbWFnZS10eXBlXCI6IFtcIm1hY2hpbmVcIl0sXG4gICAgICAgIHN0YXRlOiBbXCJhdmFpbGFibGVcIl0sXG4gICAgICAgIFwicm9vdC1kZXZpY2UtdHlwZVwiOiBbXCJlYnNcIl0sXG4gICAgICAgIFwidmlydHVhbGl6YXRpb24tdHlwZVwiOiBbXCJodm1cIl0sXG4gICAgICB9LFxuICAgIH0pLmdldEltYWdlKHRoaXMpO1xuXG4gICAgY29uc3QgcnVubmVyc01hY2hpbmVJbWFnZTogSU1hY2hpbmVJbWFnZSA9XG4gICAgICBydW5uZXJzPy5tYWNoaW5lSW1hZ2UgfHxcbiAgICAgIE1hY2hpbmVJbWFnZS5nZW5lcmljTGludXgoe1xuICAgICAgICBbc2NvcGUucmVnaW9uXTogcnVubmVyc0xvb2t1cE1hY2hpbmVJbWFnZS5pbWFnZUlkLFxuICAgICAgfSk7XG5cbiAgICAvKipcbiAgICAgKiBHaXRMYWIgTWFuYWdlclxuICAgICAqL1xuICAgIGNvbnN0IG1hbmFnZXJTZWN1cml0eUdyb3VwID0gbmV3IFNlY3VyaXR5R3JvdXAoc2NvcGUsIFwiTWFuYWdlclNlY3VyaXR5R3JvdXBcIiwge1xuICAgICAgdnBjOiB0aGlzLm5ldHdvcmsudnBjLFxuICAgICAgZGVzY3JpcHRpb246IFwiU2VjdXJpdHkgZ3JvdXAgZm9yIEdpdExhYiBSdW5uZXJzIE1hbmFnZXIuXCIsXG4gICAgfSk7XG4gICAgbWFuYWdlclNlY3VyaXR5R3JvdXAuY29ubmVjdGlvbnMuYWxsb3dUbyhydW5uZXJzU2VjdXJpdHlHcm91cCwgUG9ydC50Y3AoMjIpLCBcIlNTSCB0cmFmZmljIGZyb20gTWFuYWdlclwiKTtcbiAgICBtYW5hZ2VyU2VjdXJpdHlHcm91cC5jb25uZWN0aW9ucy5hbGxvd1RvKHJ1bm5lcnNTZWN1cml0eUdyb3VwLCBQb3J0LnRjcCgyMzc2KSwgXCJTU0ggdHJhZmZpYyBmcm9tIERvY2tlclwiKTtcblxuICAgIGNvbnN0IG1hbmFnZXJJbnN0YW5jZVR5cGUgPSBydW5uZXJzPy5pbnN0YW5jZVR5cGUgfHwgSW5zdGFuY2VUeXBlLm9mKEluc3RhbmNlQ2xhc3MuVDMsIEluc3RhbmNlU2l6ZS5OQU5PKTtcblxuICAgIGNvbnN0IG1hbmFnZXJNYWNoaW5lSW1hZ2UgPVxuICAgICAgcnVubmVycz8ubWFjaGluZUltYWdlIHx8XG4gICAgICBNYWNoaW5lSW1hZ2UubGF0ZXN0QW1hem9uTGludXgoe1xuICAgICAgICBnZW5lcmF0aW9uOiBBbWF6b25MaW51eEdlbmVyYXRpb24uQU1BWk9OX0xJTlVYXzIsXG4gICAgICAgIGVkaXRpb246IEFtYXpvbkxpbnV4RWRpdGlvbi5TVEFOREFSRCxcbiAgICAgICAgdmlydHVhbGl6YXRpb246IEFtYXpvbkxpbnV4VmlydC5IVk0sXG4gICAgICAgIHN0b3JhZ2U6IEFtYXpvbkxpbnV4U3RvcmFnZS5FQlMsXG4gICAgICAgIGNwdVR5cGU6IEFtYXpvbkxpbnV4Q3B1VHlwZS5YODZfNjQsXG4gICAgICB9KTtcblxuICAgIGNvbnN0IG1hbmFnZXJSb2xlID0gbmV3IFJvbGUoc2NvcGUsIFwiTWFuYWdlclJvbGVcIiwge1xuICAgICAgYXNzdW1lZEJ5OiBlYzJTZXJ2aWNlUHJpbmNpcGFsLFxuICAgICAgbWFuYWdlZFBvbGljaWVzOiBbZWMyTWFuYWdlZFBvbGljeUZvclNTTV0sXG4gICAgICBpbmxpbmVQb2xpY2llczoge1xuICAgICAgICBDYWNoZTogUG9saWN5RG9jdW1lbnQuZnJvbUpzb24oe1xuICAgICAgICAgIFZlcnNpb246IFwiMjAxMi0xMC0xN1wiLFxuICAgICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICAgICAgQWN0aW9uOiBbXCJzMzpMaXN0T2JqZWN0cypcIiwgXCJzMzpHZXRPYmplY3QqXCIsIFwiczM6RGVsZXRlT2JqZWN0KlwiLCBcInMzOlB1dE9iamVjdCpcIl0sXG4gICAgICAgICAgICAgIFJlc291cmNlOiBbYCR7dGhpcy5jYWNoZUJ1Y2tldC5idWNrZXRBcm59LypgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgICAgICBBY3Rpb246IFtcInMzOkxpc3RCdWNrZXRcIl0sXG4gICAgICAgICAgICAgIFJlc291cmNlOiBbYCR7dGhpcy5jYWNoZUJ1Y2tldC5idWNrZXRBcm59YF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0pLFxuICAgICAgICBSdW5uZXJzOiBQb2xpY3lEb2N1bWVudC5mcm9tSnNvbih7XG4gICAgICAgICAgVmVyc2lvbjogXCIyMDEyLTEwLTE3XCIsXG4gICAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgICAgICBBY3Rpb246IFtcImVjMjpDcmVhdGVLZXlQYWlyXCIsIFwiZWMyOkRlbGV0ZUtleVBhaXJcIiwgXCJlYzI6SW1wb3J0S2V5UGFpclwiLCBcImVjMjpEZXNjcmliZSpcIl0sXG4gICAgICAgICAgICAgIFJlc291cmNlOiBbXCIqXCJdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgICAgIEFjdGlvbjogW1wiZWMyOkNyZWF0ZVRhZ3NcIiwgXCJzc206VXBkYXRlSW5zdGFuY2VJbmZvcm1hdGlvblwiXSxcbiAgICAgICAgICAgICAgUmVzb3VyY2U6IFtcIipcIl0sXG4gICAgICAgICAgICAgIENvbmRpdGlvbjoge1xuICAgICAgICAgICAgICAgIFN0cmluZ0xpa2U6IHtcbiAgICAgICAgICAgICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvTmFtZVwiOiBcIipnaXRsYWItcnVubmVyLSpcIixcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIFwiRm9yQWxsVmFsdWVzOlN0cmluZ0VxdWFsc1wiOiB7XG4gICAgICAgICAgICAgICAgICBcImF3czpUYWdLZXlzXCI6IFtcIk5hbWVcIl0sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgICAgICBBY3Rpb246IFtcImVjMjpSZXF1ZXN0U3BvdEluc3RhbmNlc1wiLCBcImVjMjpDYW5jZWxTcG90SW5zdGFuY2VSZXF1ZXN0c1wiXSxcbiAgICAgICAgICAgICAgUmVzb3VyY2U6IFtcIipcIl0sXG4gICAgICAgICAgICAgIENvbmRpdGlvbjoge1xuICAgICAgICAgICAgICAgIFN0cmluZ0VxdWFsc0lmRXhpc3RzOiB7XG4gICAgICAgICAgICAgICAgICBcImVjMjpSZWdpb25cIjogYCR7c2NvcGUucmVnaW9ufWAsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBBcm5FcXVhbHNJZkV4aXN0czoge1xuICAgICAgICAgICAgICAgICAgXCJlYzI6VnBjXCI6IGAke3RoaXMubmV0d29yay52cGMudnBjQXJufWAsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgICAgICBBY3Rpb246IFtcImVjMjpSdW5JbnN0YW5jZXNcIl0sXG4gICAgICAgICAgICAgIFJlc291cmNlOiBbXCIqXCJdLFxuICAgICAgICAgICAgICBDb25kaXRpb246IHtcbiAgICAgICAgICAgICAgICBTdHJpbmdFcXVhbHM6IHtcbiAgICAgICAgICAgICAgICAgIFwiZWMyOkluc3RhbmNlVHlwZVwiOiBbYCR7cnVubmVyc0luc3RhbmNlVHlwZS50b1N0cmluZygpfWBdLFxuICAgICAgICAgICAgICAgICAgXCJlYzI6SW5zdGFuY2VQcm9maWxlXCI6IGAke3J1bm5lcnNJbnN0YW5jZVByb2ZpbGUucmVmfWAsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgICAgICBBY3Rpb246IFtcImVjMjpUZXJtaW5hdGVJbnN0YW5jZXNcIiwgXCJlYzI6U3RvcEluc3RhbmNlc1wiLCBcImVjMjpTdGFydEluc3RhbmNlc1wiLCBcImVjMjpSZWJvb3RJbnN0YW5jZXNcIl0sXG4gICAgICAgICAgICAgIFJlc291cmNlOiBbXCIqXCJdLFxuICAgICAgICAgICAgICBDb25kaXRpb246IHtcbiAgICAgICAgICAgICAgICBTdHJpbmdMaWtlOiB7XG4gICAgICAgICAgICAgICAgICBcImVjMjpSZXNvdXJjZVRhZy9OYW1lXCI6IFwiKmdpdGxhYi1ydW5uZXItKlwiLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICAgICAgQWN0aW9uOiBbXCJpYW06UGFzc1JvbGVcIl0sXG4gICAgICAgICAgICAgIFJlc291cmNlOiBbYCR7cnVubmVyc1JvbGUucm9sZUFybn1gXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgY29uc3QgdXNlckRhdGEgPSBVc2VyRGF0YS5mb3JMaW51eCh7fSk7XG4gICAgdXNlckRhdGEuYWRkQ29tbWFuZHMoXG4gICAgICBgeXVtIHVwZGF0ZSAteSBhd3MtY2ZuLWJvb3RzdHJhcGAgLy8gIS9iaW4vYmFzaCAteGVcbiAgICApO1xuXG4gICAgY29uc3QgZ2l0bGFiUnVubmVyQ29uZmlnUmVzdGFydEhhbmRsZSA9IG5ldyBJbml0U2VydmljZVJlc3RhcnRIYW5kbGUoKTtcbiAgICBnaXRsYWJSdW5uZXJDb25maWdSZXN0YXJ0SGFuZGxlLl9hZGRGaWxlKFwiL2V0Yy9naXRsYWItcnVubmVyL2NvbmZpZy50b21sXCIpO1xuXG4gICAgY29uc3QgcnN5c2xvZ0NvbmZpZ1Jlc3RhcnRIYW5kbGUgPSBuZXcgSW5pdFNlcnZpY2VSZXN0YXJ0SGFuZGxlKCk7XG4gICAgcnN5c2xvZ0NvbmZpZ1Jlc3RhcnRIYW5kbGUuX2FkZEZpbGUoXCIvZXRjL3JzeXNsb2cuZC8yNS1naXRsYWItcnVubmVyLmNvbmZcIik7XG5cbiAgICAvKipcbiAgICAgKiBDb25maWcgc2V0IGtleXNcbiAgICAgKiBAc2VlIHtAbGluayBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS9hd3MtcmVzb3VyY2UtaW5pdC5odG1sI2F3cy1yZXNvdXJjZS1pbml0LWNvbmZpZ3NldHN9XG4gICAgICovXG4gICAgY29uc3QgUkVQT1NJVE9SSUVTID0gXCJyZXBvc2l0b3JpZXNcIjtcbiAgICBjb25zdCBQQUNLQUdFUyA9IFwicGFja2FnZXNcIjtcbiAgICBjb25zdCBDT05GSUcgPSBcImNvbmZpZ1wiO1xuICAgIGNvbnN0IFJFU1RBUlQgPSBcInJlc3RhcnRcIjtcblxuICAgIGNvbnN0IGluaXRDb25maWcgPSBDbG91ZEZvcm1hdGlvbkluaXQuZnJvbUNvbmZpZ1NldHMoe1xuICAgICAgY29uZmlnU2V0czoge1xuICAgICAgICBkZWZhdWx0OiBbUkVQT1NJVE9SSUVTLCBQQUNLQUdFUywgQ09ORklHLCBSRVNUQVJUXSxcbiAgICAgIH0sXG4gICAgICBjb25maWdzOiB7XG4gICAgICAgIFtSRVBPU0lUT1JJRVNdOiBuZXcgSW5pdENvbmZpZyhbXG4gICAgICAgICAgSW5pdENvbW1hbmQuc2hlbGxDb21tYW5kKFxuICAgICAgICAgICAgXCJjdXJsIC1MIGh0dHBzOi8vcGFja2FnZXMuZ2l0bGFiLmNvbS9pbnN0YWxsL3JlcG9zaXRvcmllcy9ydW5uZXIvZ2l0bGFiLXJ1bm5lci9zY3JpcHQucnBtLnNoIHwgYmFzaFwiLFxuICAgICAgICAgICAgeyBrZXk6IFwiMTAtZ2l0bGFiLXJ1bm5lclwiIH1cbiAgICAgICAgICApLFxuICAgICAgICBdKSxcbiAgICAgICAgW1BBQ0tBR0VTXTogbmV3IEluaXRDb25maWcoW1xuICAgICAgICAgIEluaXRQYWNrYWdlLnl1bShcImRvY2tlclwiKSxcbiAgICAgICAgICBJbml0UGFja2FnZS55dW0oXCJnaXRsYWItcnVubmVyXCIpLFxuICAgICAgICAgIEluaXRQYWNrYWdlLnl1bShcInR6ZGF0YVwiKSxcbiAgICAgICAgICBJbml0Q29tbWFuZC5zaGVsbENvbW1hbmQoXG4gICAgICAgICAgICBcImN1cmwgLUwgaHR0cHM6Ly9naXRsYWItZG9ja2VyLW1hY2hpbmUtZG93bmxvYWRzLnMzLmFtYXpvbmF3cy5jb20vdjAuMTYuMi1naXRsYWIuMTIvZG9ja2VyLW1hY2hpbmUtYHVuYW1lIC1zYC1gdW5hbWUgLW1gID4gL3RtcC9kb2NrZXItbWFjaGluZSAmJiBpbnN0YWxsIC90bXAvZG9ja2VyLW1hY2hpbmUgL3Vzci9iaW4vZG9ja2VyLW1hY2hpbmVcIixcbiAgICAgICAgICAgIC8vXCJjdXJsIC1MIGh0dHBzOi8vZ2l0aHViLmNvbS9kb2NrZXIvbWFjaGluZS9yZWxlYXNlcy9kb3dubG9hZC92MC4xNi4yL2RvY2tlci1tYWNoaW5lLWB1bmFtZSAtc2AtYHVuYW1lIC1tYCA+IC90bXAvZG9ja2VyLW1hY2hpbmUgJiYgaW5zdGFsbCAvdG1wL2RvY2tlci1tYWNoaW5lIC91c3IvYmluL2RvY2tlci1tYWNoaW5lXCIsXG4gICAgICAgICAgICB7IGtleTogXCIxMC1kb2NrZXItbWFjaGluZVwiIH1cbiAgICAgICAgICApLFxuICAgICAgICAgIEluaXRDb21tYW5kLnNoZWxsQ29tbWFuZChcImdpdGxhYi1ydW5uZXIgc3RhcnRcIiwge1xuICAgICAgICAgICAga2V5OiBcIjIwLWdpdGxhYi1ydW5uZXItc3RhcnRcIixcbiAgICAgICAgICB9KSxcbiAgICAgICAgXSksXG4gICAgICAgIFtDT05GSUddOiBuZXcgSW5pdENvbmZpZyhbXG4gICAgICAgICAgSW5pdEZpbGUuZnJvbVN0cmluZyhcbiAgICAgICAgICAgIFwiL2V0Yy9naXRsYWItcnVubmVyL2NvbmZpZy50b21sXCIsXG4gICAgICAgICAgICBDb25maWd1cmF0aW9uTWFwcGVyLndpdGhEZWZhdWx0cyh7XG4gICAgICAgICAgICAgIGdsb2JhbENvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICAgICAgICBjb25jdXJyZW50OiBwcm9wcz8uY29uY3VycmVudCxcbiAgICAgICAgICAgICAgICBjaGVja0ludGVydmFsOiBwcm9wcz8uY2hlY2tJbnRlcnZhbCxcbiAgICAgICAgICAgICAgICBsb2dGb3JtYXQ6IHByb3BzPy5sb2dGb3JtYXQsXG4gICAgICAgICAgICAgICAgbG9nTGV2ZWw6IHByb3BzPy5sb2dMZXZlbCxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgcnVubmVyQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgICAgICAgIHRva2VuOiBwcm9wcy5naXRsYWJUb2tlbixcbiAgICAgICAgICAgICAgICB1cmw6IHByb3BzLmdpdGxhYlVybCxcbiAgICAgICAgICAgICAgICBsaW1pdDogcnVubmVycz8ubGltaXQsXG4gICAgICAgICAgICAgICAgb3V0cHV0TGltaXQ6IHJ1bm5lcnM/Lm91dHB1dExpbWl0LFxuICAgICAgICAgICAgICAgIGVudmlyb25tZW50OiBydW5uZXJzPy5lbnZpcm9ubWVudCxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgZG9ja2VyQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgICAgICAgIC4uLnJ1bm5lcnM/LmRvY2tlcixcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgbWFjaGluZUNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICAgICAgICAuLi5ydW5uZXJzPy5tYWNoaW5lLFxuICAgICAgICAgICAgICAgIG1hY2hpbmVPcHRpb25zOiB7XG4gICAgICAgICAgICAgICAgICAuLi5ydW5uZXJzPy5tYWNoaW5lPy5tYWNoaW5lT3B0aW9ucyxcbiAgICAgICAgICAgICAgICAgIGluc3RhbmNlVHlwZTogcnVubmVycz8uaW5zdGFuY2VUeXBlPy50b1N0cmluZygpLFxuICAgICAgICAgICAgICAgICAgYW1pOiBydW5uZXJzPy5tYWNoaW5lSW1hZ2U/LmdldEltYWdlKHNjb3BlKS5pbWFnZUlkLFxuICAgICAgICAgICAgICAgICAgcmVnaW9uOiBzY29wZS5yZWdpb24sXG4gICAgICAgICAgICAgICAgICB2cGNJZDogdGhpcy5uZXR3b3JrLnZwYy52cGNJZCxcbiAgICAgICAgICAgICAgICAgIHpvbmU6IHRoaXMubmV0d29yay5hdmFpbGFiaWxpdHlab25lLnNsaWNlKC0xKSxcbiAgICAgICAgICAgICAgICAgIHN1Ym5ldElkOiB0aGlzLm5ldHdvcmsuc3VibmV0LnN1Ym5ldElkLFxuICAgICAgICAgICAgICAgICAgc2VjdXJpdHlHcm91cDogYCR7cnVubmVyc1NlY3VyaXR5R3JvdXBOYW1lfWAsXG4gICAgICAgICAgICAgICAgICB1c2VQcml2YXRlQWRkcmVzczogdHJ1ZSxcbiAgICAgICAgICAgICAgICAgIGlhbUluc3RhbmNlUHJvZmlsZTogYCR7cnVubmVyc0luc3RhbmNlUHJvZmlsZS5yZWZ9YCxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBhdXRvc2NhbGluZ0NvbmZpZ3VyYXRpb25zOiBydW5uZXJzPy5hdXRvc2NhbGluZyB8fCBbXSxcbiAgICAgICAgICAgICAgY2FjaGVDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgICAgICAgczM6IHtcbiAgICAgICAgICAgICAgICAgIHNlcnZlckFkZHJlc3M6IGBzMy4ke3Njb3BlLnVybFN1ZmZpeH1gLFxuICAgICAgICAgICAgICAgICAgYnVja2V0TmFtZTogYCR7dGhpcy5jYWNoZUJ1Y2tldC5idWNrZXROYW1lfWAsXG4gICAgICAgICAgICAgICAgICBidWNrZXRMb2NhdGlvbjogYCR7c2NvcGUucmVnaW9ufWAsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0pLnRvVG9tbCgpLFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBvd25lcjogXCJnaXRsYWItcnVubmVyXCIsXG4gICAgICAgICAgICAgIGdyb3VwOiBcImdpdGxhYi1ydW5uZXJcIixcbiAgICAgICAgICAgICAgbW9kZTogXCIwMDA2MDBcIixcbiAgICAgICAgICAgIH1cbiAgICAgICAgICApLFxuICAgICAgICAgIEluaXRGaWxlLmZyb21TdHJpbmcoXG4gICAgICAgICAgICBcIi9ldGMvcnN5c2xvZy5kLzI1LWdpdGxhYi1ydW5uZXIuY29uZlwiLFxuICAgICAgICAgICAgYDpwcm9ncmFtbmFtZSwgaXNlcXVhbCwgXCJnaXRsYWItcnVubmVyXCIgL3Zhci9sb2cvZ2l0bGFiLXJ1bm5lci5sb2dgLFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBvd25lcjogXCJyb290XCIsXG4gICAgICAgICAgICAgIGdyb3VwOiBcInJvb3RcIixcbiAgICAgICAgICAgICAgbW9kZTogXCIwMDA2NDRcIixcbiAgICAgICAgICAgIH1cbiAgICAgICAgICApLFxuICAgICAgICAgIEluaXRTZXJ2aWNlLmVuYWJsZShcImdpdGxhYi1ydW5uZXJcIiwge1xuICAgICAgICAgICAgZW5zdXJlUnVubmluZzogdHJ1ZSxcbiAgICAgICAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgICBzZXJ2aWNlUmVzdGFydEhhbmRsZTogZ2l0bGFiUnVubmVyQ29uZmlnUmVzdGFydEhhbmRsZSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgICBJbml0U2VydmljZS5lbmFibGUoXCJyc3lzbG9nXCIsIHtcbiAgICAgICAgICAgIGVuc3VyZVJ1bm5pbmc6IHRydWUsXG4gICAgICAgICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgICAgICAgc2VydmljZVJlc3RhcnRIYW5kbGU6IHJzeXNsb2dDb25maWdSZXN0YXJ0SGFuZGxlLFxuICAgICAgICAgIH0pLFxuICAgICAgICBdKSxcbiAgICAgICAgW1JFU1RBUlRdOiBuZXcgSW5pdENvbmZpZyhbXG4gICAgICAgICAgSW5pdENvbW1hbmQuc2hlbGxDb21tYW5kKFwiZ2l0bGFiLXJ1bm5lciByZXN0YXJ0XCIsIHtcbiAgICAgICAgICAgIGtleTogXCIxMC1naXRsYWItcnVubmVyLXJlc3RhcnRcIixcbiAgICAgICAgICB9KSxcbiAgICAgICAgXSksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgY29uc3QgbWFuYWdlckF1dG9TY2FsaW5nR3JvdXAgPSBuZXcgQXV0b1NjYWxpbmdHcm91cChzY29wZSwgXCJNYW5hZ2VyQXV0b3NjYWxpbmdHcm91cFwiLCB7XG4gICAgICB2cGM6IHRoaXMubmV0d29yay52cGMsXG4gICAgICB2cGNTdWJuZXRzOiB7XG4gICAgICAgIHN1Ym5ldHM6IFt0aGlzLm5ldHdvcmsuc3VibmV0XSxcbiAgICAgIH0sXG4gICAgICBpbnN0YW5jZVR5cGU6IG1hbmFnZXJJbnN0YW5jZVR5cGUsXG4gICAgICBtYWNoaW5lSW1hZ2U6IG1hbmFnZXJNYWNoaW5lSW1hZ2UsXG4gICAgICBrZXlOYW1lOiBtYW5hZ2VyPy5rZXlQYWlyTmFtZSxcbiAgICAgIHNlY3VyaXR5R3JvdXA6IG1hbmFnZXJTZWN1cml0eUdyb3VwLFxuICAgICAgcm9sZTogbWFuYWdlclJvbGUsXG4gICAgICB1c2VyRGF0YTogdXNlckRhdGEsXG4gICAgICBpbml0OiBpbml0Q29uZmlnLFxuICAgICAgaW5pdE9wdGlvbnM6IHtcbiAgICAgICAgaWdub3JlRmFpbHVyZXM6IGZhbHNlLFxuICAgICAgfSxcbiAgICAgIG1heENhcGFjaXR5OiAxLFxuICAgICAgbWluQ2FwYWNpdHk6IDEsXG4gICAgICBkZXNpcmVkQ2FwYWNpdHk6IDEsXG4gICAgICBzaWduYWxzOiBTaWduYWxzLndhaXRGb3JDb3VudCgxLCB7IHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMTUpIH0pLFxuICAgIH0pO1xuXG4gICAgdGhpcy5ydW5uZXJzID0ge1xuICAgICAgc2VjdXJpdHlHcm91cE5hbWU6IHJ1bm5lcnNTZWN1cml0eUdyb3VwTmFtZSxcbiAgICAgIHNlY3VyaXR5R3JvdXA6IHJ1bm5lcnNTZWN1cml0eUdyb3VwLFxuICAgICAgcm9sZTogcnVubmVyc1JvbGUsXG4gICAgICBpbnN0YW5jZVByb2ZpbGU6IHJ1bm5lcnNJbnN0YW5jZVByb2ZpbGUsXG4gICAgICBpbnN0YW5jZVR5cGU6IHJ1bm5lcnNJbnN0YW5jZVR5cGUsXG4gICAgICBtYWNoaW5lSW1hZ2U6IHJ1bm5lcnNNYWNoaW5lSW1hZ2UsXG4gICAgfTtcblxuICAgIHRoaXMubWFuYWdlciA9IHtcbiAgICAgIHNlY3VyaXR5R3JvdXA6IG1hbmFnZXJTZWN1cml0eUdyb3VwLFxuICAgICAgaW5zdGFuY2VUeXBlOiBtYW5hZ2VySW5zdGFuY2VUeXBlLFxuICAgICAgbWFjaGluZUltYWdlOiBtYW5hZ2VyTWFjaGluZUltYWdlLFxuICAgICAgYXV0b1NjYWxpbmdHcm91cDogbWFuYWdlckF1dG9TY2FsaW5nR3JvdXAsXG4gICAgICByb2xlOiBtYW5hZ2VyUm9sZSxcbiAgICB9O1xuICB9XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgR2l0bGFiUnVubmVyQXV0b3NjYWxpbmdSdW5uZXJzIHtcbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cE5hbWU6IHN0cmluZztcbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cDogSVNlY3VyaXR5R3JvdXA7XG4gIHJlYWRvbmx5IHJvbGU6IElSb2xlO1xuICByZWFkb25seSBpbnN0YW5jZVByb2ZpbGU6IENmbkluc3RhbmNlUHJvZmlsZTtcbiAgcmVhZG9ubHkgaW5zdGFuY2VUeXBlOiBJbnN0YW5jZVR5cGU7XG4gIHJlYWRvbmx5IG1hY2hpbmVJbWFnZTogSU1hY2hpbmVJbWFnZTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBHaXRsYWJSdW5uZXJBdXRvc2NhbGluZ01hbmFnZXIge1xuICByZWFkb25seSBzZWN1cml0eUdyb3VwOiBJU2VjdXJpdHlHcm91cDtcbiAgcmVhZG9ubHkgaW5zdGFuY2VUeXBlOiBJbnN0YW5jZVR5cGU7XG4gIHJlYWRvbmx5IG1hY2hpbmVJbWFnZTogSU1hY2hpbmVJbWFnZTtcbiAgcmVhZG9ubHkgYXV0b1NjYWxpbmdHcm91cDogSUF1dG9TY2FsaW5nR3JvdXA7XG4gIHJlYWRvbmx5IHJvbGU6IElSb2xlO1xufVxuIl19