"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsSolutionsChecks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const common_1 = require("../common");
const analytics_1 = require("./rules/analytics");
const application_integration_1 = require("./rules/application_integration");
const index_1 = require("./rules/compute/index");
const index_2 = require("./rules/databases/index");
const developer_tools_1 = require("./rules/developer_tools");
const index_3 = require("./rules/machine_learning/index");
const management_and_governance_1 = require("./rules/management_and_governance");
const media_services_1 = require("./rules/media_services");
const index_4 = require("./rules/network_and_delivery/index");
const security_and_compliance_1 = require("./rules/security_and_compliance");
const serverless_1 = require("./rules/serverless");
const index_5 = require("./rules/storage/index");
/**
 * Check Best practices based on AWS Solutions Security Matrix.
 *
 * @stability stable
 */
class AwsSolutionsChecks extends common_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        var _b;
        if (node instanceof monocdk_1.CfnResource) {
            // Get ignores metadata if it exists
            const ignores = (_b = node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
            this.checkCompute(node, ignores);
            this.checkStorage(node, ignores);
            this.checkDatabases(node, ignores);
            this.checkNetworkDelivery(node, ignores);
            this.checkManagementGovernance(node, ignores);
            this.checkMachineLearning(node, ignores);
            this.checkAnalytics(node, ignores);
            this.checkSecurityCompliance(node, ignores);
            this.checkServerless(node, ignores);
            this.checkApplicationIntegration(node, ignores);
            this.checkMediaServices(node, ignores);
            this.checkDeveloperTools(node, ignores);
        }
    }
    /**
     * Check Compute Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCompute(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-EC23') &&
            !index_1.awsSolutionsEc23(node)) {
            const ruleId = 'AwsSolutions-EC23';
            const info = 'The Security Group allows for 0.0.0.0/0 or ::/0 inbound access.';
            const explanation = 'Large port ranges, when open, expose instances to unwanted attacks. More than that, they make traceability of vulnerabilities very difficult. For instance, your web servers may only require 80 and 443 ports to be open, but not all. One of the most common mistakes observed is when  all ports for 0.0.0.0/0 range are open in a rush to access the instance. EC2 instances must expose only to those ports enabled on the corresponding security group level.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-EC26') &&
            !index_1.awsSolutionsEc26(node)) {
            const ruleId = 'AwsSolutions-EC26';
            const info = 'The EBS volume has encryption disabled.';
            const explanation = "With EBS encryption, you aren't required to build, maintain, and secure your own key management infrastructure. EBS encryption uses KMS keys when creating encrypted volumes and snapshots. This helps protect data at rest.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-EC27') &&
            !index_1.awsSolutionsEc27(node)) {
            const ruleId = 'AwsSolutions-EC27';
            const info = 'The Security Group does not have a description.';
            const explanation = 'Descriptions help simplify operations and remove any opportunities for operator errors.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-EC28') &&
            !index_1.awsSolutionsEc28(node)) {
            const ruleId = 'AwsSolutions-EC28';
            const info = 'The EC2 instance/AutoScaling launch configuration does not have detailed monitoring enabled.';
            const explanation = 'Monitoring data helps make better decisions on architecting and managing compute resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-EC29') &&
            !index_1.awsSolutionsEc29(node)) {
            const ruleId = 'AwsSolutions-EC29';
            const info = 'The EC2 instance is not part of an ASG and has Termination Protection disabled.';
            const explanation = 'Termination Protection safety feature enabled in order to protect the instances from being accidentally terminated.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ECR1') &&
            !index_1.awsSolutionsEcr1(node)) {
            const ruleId = 'AwsSolutions-ECR1';
            const info = 'The ECR Repository allows open access.';
            const explanation = 'Removing * principals in an ECR Repository helps protect against unauthorized access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ECS4') &&
            !index_1.awsSolutionsEcs4(node)) {
            const ruleId = 'AwsSolutions-ECS4';
            const info = 'The ECS Cluster has CloudWatch Container Insights disabled.';
            const explanation = 'CloudWatch Container Insights allow operators to gain a better perspective on how the cluster’s applications and microservices are performing.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ECS7') &&
            !index_1.awsSolutionsEcs7(node)) {
            const ruleId = 'AwsSolutions-ECS7';
            const info = 'The ECS Task Definition does not have awslogs logging enabled at the minimum.';
            const explanation = 'Container logging allows operators to view and aggregate the logs from the container.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ELB1') &&
            !index_1.awsSolutionsElb1(node)) {
            const ruleId = 'AwsSolutions-ELB1';
            const info = 'The CLB is used for incoming HTTP/HTTPS traffic. Use ALBs instead.';
            const explanation = 'HTTP/HTTPS applications (monolithic or containerized) should use the ALB instead of The CLB for enhanced incoming traffic distribution, better performance and lower costs.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ELB2a') &&
            !index_1.awsSolutionsElb2a(node)) {
            const ruleId = 'AwsSolutions-ELB2a';
            const info = 'The ALB does not have access logs enabled.';
            const explanation = 'Access logs allow operators to to analyze traffic patterns and identify and troubleshoot security issues.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ELB2e') &&
            !index_1.awsSolutionsElb2e(node)) {
            const ruleId = 'AwsSolutions-ELB2e';
            const info = 'The CLB does not have access logs enabled.';
            const explanation = 'Access logs allow operators to to analyze traffic patterns and identify and troubleshoot security issues.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ELB3') &&
            !index_1.awsSolutionsElb3(node)) {
            const ruleId = 'AwsSolutions-ELB3';
            const info = 'The CLB does not have connection draining enabled.';
            const explanation = 'With Connection Draining feature enabled, if an EC2 backend instance fails health checks The CLB will not send any new requests to the unhealthy instance. However, it will still allow existing (in-flight) requests to complete for the duration of the configured timeout.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ELB4') &&
            !index_1.awsSolutionsElb4(node)) {
            const ruleId = 'AwsSolutions-ELB4';
            const info = 'The CLB does not use at least two AZs with the Cross-Zone Load Balancing feature enabled.';
            const explanation = 'CLBs can distribute the traffic evenly across all backend instances. To use Cross-Zone Load Balancing at optimal level, the system should maintain an equal EC2 capacity distribution in each of the AZs registered with the load balancer.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ELB5') &&
            !index_1.awsSolutionsElb5(node)) {
            const ruleId = 'AwsSolutions-ELB5';
            const info = 'The CLB listener is not configured for secure (HTTPs or SSL) protocols for client communication.';
            const explanation = 'The HTTPs or SSL protocols enable secure communication by encrypting the communication between the client and the load balancer.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Storage Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkStorage(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-S1') && !index_5.awsSolutionsS1(node)) {
            const ruleId = 'AwsSolutions-S1';
            const info = 'The S3 Bucket has server access logs disabled.';
            const explanation = 'The bucket should have server access logging enabled to provide detailed records for the requests that are made to the bucket.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-S2') && !index_5.awsSolutionsS2(node)) {
            const ruleId = 'AwsSolutions-S2';
            const info = 'The S3 Bucket does not have public access restricted and blocked.';
            const explanation = 'The bucket should have public access restricted and blocked to prevent unauthorized access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-S3') && !index_5.awsSolutionsS3(node)) {
            const ruleId = 'AwsSolutions-S3';
            const info = 'The S3 Bucket does not default encryption enabled.';
            const explanation = 'The bucket should minimally have SSE enabled to help protect data-at-rest.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-EFS1') &&
            !index_5.awsSolutionsEfs1(node)) {
            const ruleId = 'AwsSolutions-EFS1';
            const info = 'The EFS is not configured for encryption at rest.';
            const explanation = 'By using an encrypted file system, data and metadata are automatically encrypted before being written to the file system. Similarly, as data and metadata are read, they are automatically decrypted before being presented to the application. These processes are handled transparently by EFS without requiring modification of applications.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Database Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkDatabases(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-RDS2') &&
            !index_2.awsSolutionsRds2(node)) {
            const ruleId = 'AwsSolutions-RDS2';
            const info = 'The RDS instance or Aurora cluster does not have storage encryption enabled.';
            const explanation = 'Storage encryption helps protect data-at-rest by encrypting the underlying storage, automated backups, read replicas, and snapshots for the database.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RDS6') &&
            !index_2.awsSolutionsRds6(node)) {
            const ruleId = 'AwsSolutions-RDS6';
            const info = 'The RDS Aurora MySQL/PostgresSQL cluster does not have IAM Database Authentication enabled.';
            const explanation = "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the MySQL/PostgreSQL database instances, instead it uses an authentication token.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RDS10') &&
            !index_2.awsSolutionsRds10(node)) {
            const ruleId = 'AwsSolutions-RDS10';
            const info = 'The RDS instance or Aurora cluster does not have deletion protection enabled.';
            const explanation = 'The deletion protection feature helps protect the database from being accidentally deleted.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RDS11') &&
            !index_2.awsSolutionsRds11(node)) {
            const ruleId = 'AwsSolutions-RDS11';
            const info = 'The RDS instance or Aurora cluster uses the default endpoint port.';
            const explanation = 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MySQL/Aurora port 3306, SQL Server port 1433, PostgreSQL port 5432, etc).';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RDS13') &&
            !index_2.awsSolutionsRds13(node)) {
            const ruleId = 'AwsSolutions-RDS13';
            const info = 'The RDS instance is not configured for automated backups.';
            const explanation = 'Automated backups allow for point-in-time recovery.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RDS14') &&
            !index_2.awsSolutionsRds14(node)) {
            const ruleId = 'AwsSolutions-RDS14';
            const info = 'The RDS Aurora MySQL cluster does not have Backtrack enabled.';
            const explanation = 'Backtrack helps order to rewind cluster tables to a specific time, without using backups.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RDS16') &&
            !index_2.awsSolutionsRds16(node)) {
            const ruleId = 'AwsSolutions-RDS16';
            const info = 'The RDS Aurora MySQL serverless cluster does not have audit, error, general, and slowquery Log Exports enabled.';
            const explanation = 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-DDB3') &&
            !index_2.awsSolutionsDdb3(node)) {
            const ruleId = 'AwsSolutions-DDB3';
            const info = 'The DynamoDB table does not have Point-in-time Recovery enabled.';
            const explanation = 'DynamoDB continuous backups represent an additional layer of insurance against accidental loss of data on top of on-demand backups. The DynamoDB service can back up the data with per-second granularity and restore it to any single second from the time PITR was enabled up to the prior 35 days.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-DDB4') &&
            !index_2.awsSolutionsDdb4(node)) {
            const ruleId = 'AwsSolutions-DDB4';
            const info = 'The DAX cluster does not have server-side encryption enabled.';
            const explanation = 'Data in cache, configuration data and log files should be encrypted using Server-Side Encryption in order to protect from unauthorized access to the underlying storage.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-AEC1') &&
            !index_2.awsSolutionsAec1(node)) {
            const ruleId = 'AwsSolutions-AEC1';
            const info = 'The ElastiCache cluster is not provisioned in a VPC.';
            const explanation = 'Provisioning the cluster within a VPC allows for better flexibility and control over the cache clusters security, availability, traffic routing and more.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-AEC3') &&
            !index_2.awsSolutionsAec3(node)) {
            const ruleId = 'AwsSolutions-AEC3';
            const info = 'The ElastiCache Redis cluster does not have both encryption in transit and at rest enabled.';
            const explanation = 'Encryption in transit helps secure communications to the cluster. Encryption at rest helps protect data at rest from unauthorized access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-AEC4') &&
            !index_2.awsSolutionsAec4(node)) {
            const ruleId = 'AwsSolutions-AEC4';
            const info = 'The ElastiCache Redis cluster is not deployed in a Multi-AZ configuration.';
            const explanation = 'The cluster should use a Multi-AZ deployment configuration for high availability.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-AEC5') &&
            !index_2.awsSolutionsAec5(node)) {
            const ruleId = 'AwsSolutions-AEC5';
            const info = 'The ElastiCache cluster uses the default endpoint port.';
            const explanation = 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redis port 6379 and Memcached port 11211).';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-AEC6') &&
            !index_2.awsSolutionsAec6(node)) {
            const ruleId = 'AwsSolutions-AEC6';
            const info = 'The ElastiCache Redis cluster does not use Redis AUTH for user authentication.';
            const explanation = 'Redis authentication tokens enable Redis to require a token (password) before allowing clients to execute commands, thereby improving data security.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-N1') && !index_2.awsSolutionsN1(node)) {
            const ruleId = 'AwsSolutions-N1';
            const info = 'The Neptune DB cluster is not deployed in a Multi-AZ configuration.';
            const explanation = 'The cluster should use a Multi-AZ deployment configuration for high availability.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-N2') && !index_2.awsSolutionsN2(node)) {
            const ruleId = 'AwsSolutions-N2';
            const info = 'The Neptune DB instance does have Auto Minor Version Upgrade enabled.';
            const explanation = 'The Neptune service regularly releases engine updates. Enabling Auto Minor Version Upgrade will allow the service to automatically apply these upgrades to DB Instances.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-N3') && !index_2.awsSolutionsN3(node)) {
            const ruleId = 'AwsSolutions-N3';
            const info = 'The Neptune DB cluster does not have a reasonable minimum backup retention period configured.';
            const explanation = 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-N4') && !index_2.awsSolutionsN4(node)) {
            const ruleId = 'AwsSolutions-N4';
            const info = 'The Neptune DB cluster does not have encryption at rest enabled.';
            const explanation = 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-N5') && !index_2.awsSolutionsN5(node)) {
            const ruleId = 'AwsSolutions-N5';
            const info = 'The Neptune DB cluster does not have IAM Database Authentication enabled.';
            const explanation = "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the cluster.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS1') &&
            !index_2.awsSolutionsRs1(node)) {
            const ruleId = 'AwsSolutions-RS1';
            const info = 'The Redshift cluster parameter group must have the "require_ssl" parameter enabled.';
            const explanation = 'Enabling the "require_ssl" parameter secures data-in-transit by encrypting the connection between the clients and the Redshift clusters.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS2') &&
            !index_2.awsSolutionsRs2(node)) {
            const ruleId = 'AwsSolutions-RS2';
            const info = 'The Redshift cluster is not provisioned in a VPC.';
            const explanation = 'Provisioning the cluster within a VPC allows for better flexibility and control over the Redshift clusters security, availability, traffic routing and more.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS3') &&
            !index_2.awsSolutionsRs3(node)) {
            const ruleId = 'AwsSolutions-RS3';
            const info = 'The Redshift cluster uses the default "awsuser" username.';
            const explanation = 'Using a custom master user name instead of the default master user name (i.e. "awsuser") provides an additional layer of defense against non-targeted attacks.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS4') &&
            !index_2.awsSolutionsRs4(node)) {
            const ruleId = 'AwsSolutions-RS4';
            const info = 'The Redshift cluster uses the default endpoint port.';
            const explanation = 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redshift port 5439).';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS5') &&
            !index_2.awsSolutionsRs5(node)) {
            const ruleId = 'AwsSolutions-RS5';
            const info = 'The Redshift cluster does not have audit logging enabled.';
            const explanation = 'Audit logging helps operators troubleshoot issues and ensure security.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS6') &&
            !index_2.awsSolutionsRs6(node)) {
            const ruleId = 'AwsSolutions-RS6';
            const info = 'The Redshift cluster does not have encryption at rest enabled.';
            const explanation = 'Encrypting data-at-rest protects data confidentiality.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS8') &&
            !index_2.awsSolutionsRs8(node)) {
            const ruleId = 'AwsSolutions-RS8';
            const info = 'The Redshift cluster is publicly accessible.';
            const explanation = 'Disabling public accessibility helps minimize security risks.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS9') &&
            !index_2.awsSolutionsRs9(node)) {
            const ruleId = 'AwsSolutions-RS9';
            const info = 'The Redshift cluster does not have version upgrade enabled.';
            const explanation = 'Version Upgrade must enabled to enable the cluster to automatically receive upgrades during the maintenance window.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS10') &&
            !index_2.awsSolutionsRs10(node)) {
            const ruleId = 'AwsSolutions-RS10';
            const info = 'The Redshift cluster does not have a retention period for automated snapshots configured.';
            const explanation = 'The retention period represents the number of days to retain automated snapshots. A positive retention period should be set to configure this feature.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-DOC1') &&
            !index_2.awsSolutionsDoc1(node)) {
            const ruleId = 'AwsSolutions-DOC1';
            const info = 'The Document DB cluster does not have encryption at rest enabled.';
            const explanation = 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-DOC2') &&
            !index_2.awsSolutionsDoc2(node)) {
            const ruleId = 'AwsSolutions-DOC2';
            const info = 'The Document DB cluster uses the default endpoint port.';
            const explanation = 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MongoDB port 27017).';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-DOC3') &&
            !index_2.awsSolutionsDoc3(node)) {
            const ruleId = 'AwsSolutions-DOC3';
            const info = 'The Document DB cluster does not have the username and password stored in Secrets Manager.';
            const explanation = "Secrets Manager enables operators to replace hardcoded credentials in your code, including passwords, with an API call to Secrets Manager to retrieve the secret programmatically. This helps ensure the secret can't be compromised by someone examining system code, because the secret no longer exists in the code. Also, operators can configure Secrets Manager to automatically rotate the secret for you according to a specified schedule. This enables you to replace long-term secrets with short-term ones, significantly reducing the risk of compromise.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-DOC4') &&
            !index_2.awsSolutionsDoc4(node)) {
            const ruleId = 'AwsSolutions-DOC4';
            const info = 'The Document DB cluster does not have a reasonable minimum backup retention period configured.';
            const explanation = 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-DOC5') &&
            !index_2.awsSolutionsDoc5(node)) {
            const ruleId = 'AwsSolutions-DOC5';
            const info = 'The Document DB cluster does not have authenticate, createIndex, and dropCollection Log Exports enabled.';
            const explanation = 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database. The events recorded by the AWS DocumentDB audit logs include successful and failed authentication attempts, creating indexes or dropping a collection in a database within the DocumentDB cluster.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Network and Delivery Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkNetworkDelivery(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-VPC3') &&
            !index_4.awsSolutionsVpc3(node)) {
            const ruleId = 'AwsSolutions-VPC3';
            const info = 'A Network ACL or Network ACL entry has been implemented.';
            const explanation = 'Network ACLs should be used sparingly for the following reasons: they can be complex to manage, they are stateless, every IP address must be explicitly opened in each (inbound/outbound) direction, and they affect a complete subnet. Use security groups when possible as they are stateful and easier to manage.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-CFR1') &&
            !index_4.awsSolutionsCfr1(node)) {
            const ruleId = 'AwsSolutions-CFR1';
            const info = 'The CloudFront distribution may require Geo restrictions.';
            const explanation = 'Geo restriction may need to be enabled for the distribution in order to allow or deny a country in order to allow or restrict users in specific locations from accessing content.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-CFR2') &&
            !index_4.awsSolutionsCfr2(node)) {
            const ruleId = 'AwsSolutions-CFR2';
            const info = 'The CloudFront distribution may require integration with AWS WAF.';
            const explanation = 'The Web Application Firewall can help protect against application-layer attacks that can compromise the security of the system or place unnecessary load on them.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-CFR3') &&
            !index_4.awsSolutionsCfr3(node)) {
            const ruleId = 'AwsSolutions-CFR3';
            const info = 'The CloudFront distributions does not have access logging enabled.';
            const explanation = 'Enabling access logs helps operators track all viewer requests for the content delivered through the Content Delivery Network.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-CFR5') &&
            !index_4.awsSolutionsCfr5(node)) {
            const ruleId = 'AwsSolutions-CFR5';
            const info = 'The CloudFront distributions uses SSLv3 or TLSv1 for communication to the origin.';
            const explanation = 'Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Using a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS helps protect viewer connections.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-CFR6') &&
            !index_4.awsSolutionsCfr6(node)) {
            const ruleId = 'AwsSolutions-CFR6';
            const info = 'The CloudFront distribution does not use an origin access identity an S3 origin.';
            const explanation = 'Origin access identities help with security by restricting any direct access to objects through S3 URLs.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-APIG1') &&
            !index_4.awsSolutionsApig1(node)) {
            const ruleId = 'AwsSolutions-APIG1';
            const info = 'The API does not have access logging enabled.';
            const explanation = 'Enabling access logs helps operators view who accessed an API and how the caller accessed the API.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-APIG4') &&
            !index_4.awsSolutionsApig4(node)) {
            const ruleId = 'AwsSolutions-APIG4';
            const info = 'The API does not implement authorization.';
            const explanation = 'In most cases an API needs to have an authentication and authorization implementation strategy. This includes using such approaches as IAM, Cognito User Pools, Custom authorizer, etc.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-APIG6') &&
            !index_4.awsSolutionsApig6(node)) {
            const ruleId = 'AwsSolutions-APIG6';
            const info = 'The REST API Stage does not have CloudWatch logging enabled for all methods.';
            const explanation = 'Enabling CloudWatch logs at the stage level helps operators to track and analyze execution behavior at the API stage level.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Management and Governance Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkManagementGovernance(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-AS1') &&
            !management_and_governance_1.awsSolutionsAs1(node)) {
            const ruleId = 'AwsSolutions-AS1';
            const info = 'The Auto Scaling Group does not have a cooldown period.';
            const explanation = 'A cooldown period temporarily suspends any scaling activities in order to allow the newly launched EC2 instance(s) some time to start handling the application traffic.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-AS2') &&
            !management_and_governance_1.awsSolutionsAs2(node)) {
            const ruleId = 'AwsSolutions-AS2';
            const info = 'The Auto Scaling Group does not have properly configured health checks.';
            const explanation = 'The health check feature enables the service to detect whether its registered EC2 instances are healthy or not.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-AS3') &&
            !management_and_governance_1.awsSolutionsAs3(node)) {
            const ruleId = 'AwsSolutions-AS3';
            const info = 'The Auto Scaling Group does not have notifications configured for all scaling events.';
            const explanation = 'Notifications on EC2 instance launch, launch error, termination, and termination errors allow operators to gain better insights into systems attributes such as activity and health.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Machine Learning Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkMachineLearning(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-SM1') &&
            !index_3.awsSolutionsSm1(node)) {
            const ruleId = 'AwsSolutions-SM1';
            const info = 'The SageMaker notebook instance is not provisioned inside a VPC.';
            const explanation = 'Provisioning the notebook instances inside a VPC enables the notebook to access VPC-only resources such as EFS file systems.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-SM2') &&
            !index_3.awsSolutionsSm2(node)) {
            const ruleId = 'AwsSolutions-SM2';
            const info = 'The SageMaker notebook instance does not have an encrypted storage volume.';
            const explanation = 'Encrypting storage volumes helps protect SageMaker data-at-rest.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-SM3') &&
            !index_3.awsSolutionsSm3(node)) {
            const ruleId = 'AwsSolutions-SM3';
            const info = 'The SageMaker notebook instance has direct internet access enabled.';
            const explanation = 'Disabling public accessibility helps minimize security risks.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Analytics Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkAnalytics(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-ATH1') &&
            !analytics_1.awsSolutionsAth1(node)) {
            const ruleId = 'AwsSolutions-ATH1';
            const info = 'The Athena workgroup does not encrypt query results.';
            const explanation = 'Encrypting query results stored in S3 helps secure data to meet compliance requirements for data-at-rest encryption.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-EMR2') &&
            !analytics_1.awsSolutionsEmr2(node)) {
            const ruleId = 'AwsSolutions-EMR2';
            const info = 'The EMR cluster does not have S3 logging enabled.';
            const explanation = 'Uploading logs to S3 enables the system to keep the logging data for historical purposes or to track and analyze the clusters behavior.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-EMR6') &&
            !analytics_1.awsSolutionsEmr6(node)) {
            const ruleId = 'AwsSolutions-EMR6';
            const info = 'The EMR cluster does not implement authentication via an EC2 Key Pair or Kerberos.';
            const explanation = 'SSH clients can use an EC2 key pair to authenticate to cluster instances. Alternatively, with EMR release version 5.10.0 or later, solutions can configure Kerberos to authenticate users and SSH connections to the master node.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ESH1') &&
            !analytics_1.awsSolutionsEsh1(node)) {
            const ruleId = 'AwsSolutions-ESH1';
            const info = 'The ES domain is not provisioned inside a VPC.';
            const explanation = 'Provisioning the domain within a VPC enables better flexibility and control over the clusters access and security as this feature keeps all traffic between the VPC and Elasticsearch domains within the AWS network instead of going over the public Internet.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ESH2') &&
            !analytics_1.awsSolutionsEsh2(node)) {
            const ruleId = 'AwsSolutions-ESH2';
            const info = 'The ES domain does not have node-to-node encryption enabled.';
            const explanation = 'Enabling the node-to-node encryption feature adds an extra layer of data protection on top of the existing ES security features such as HTTPS client to cluster encryption and data-at-rest encryption.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ESH3') &&
            !analytics_1.awsSolutionsEsh3(node)) {
            const ruleId = 'AwsSolutions-ESH3';
            const info = 'The ES domain does not only grant access via allowlisted IP addresses.';
            const explanation = 'Using allowlisted IP addresses helps protect the domain against unauthorized access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ESH4') &&
            !analytics_1.awsSolutionsEsh4(node)) {
            const ruleId = 'AwsSolutions-ESH4';
            const info = 'The ES domain does not use dedicated master nodes.';
            const explanation = 'Using dedicated master nodes helps improve environmental stability by offloading all the management tasks from the data nodes.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ESH5') &&
            !analytics_1.awsSolutionsEsh5(node)) {
            const ruleId = 'AwsSolutions-ESH5';
            const info = 'The ES domain does not allow for unsigned requests or anonymous access.';
            const explanation = 'Restricting public access helps prevent unauthorized access and prevents any unsigned requests to be made to the resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ESH7') &&
            !analytics_1.awsSolutionsEsh7(node)) {
            const ruleId = 'AwsSolutions-ESH7';
            const info = 'The ES domain does not have Zone Awareness enabled.';
            const explanation = 'Enabling cross-zone replication (Zone Awareness) increases the availability of the ES domain by allocating the nodes and replicate the data across two AZs in the same region in order to prevent data loss and minimize downtime in the event of node or AZ failure.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ESH8') &&
            !analytics_1.awsSolutionsEsh8(node)) {
            const ruleId = 'AwsSolutions-ESH8';
            const info = 'The ES domain does not have encryption at rest enabled.';
            const explanation = 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ESH9') &&
            !analytics_1.awsSolutionsEsh9(node)) {
            const ruleId = 'AwsSolutions-ESH9';
            const info = 'The ES domain does not minimally publish SEARCH_SLOW_LOGS and INDEX_SLOW_LOGS to CloudWatch Logs.';
            const explanation = 'These logs enable operators to gain full insight into the performance of these operations.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-KDA3') &&
            !analytics_1.awsSolutionsKda3(node)) {
            const ruleId = 'AwsSolutions-KDA3';
            const info = 'The Kinesis Data Analytics Flink Application does not have checkpointing enabled.';
            const explanation = 'Checkpoints are backups of application state that KDA automatically creates periodically and uses to restore from faults.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-KDS1') &&
            !analytics_1.awsSolutionsKds1(node)) {
            const ruleId = 'AwsSolutions-KDS1';
            const info = 'The Kinesis Data Stream does not has server-side encryption enabled.';
            const explanation = "Data is encrypted before it's written to the Kinesis stream storage layer, and decrypted after it’s retrieved from storage. This allows the system to meet strict regulatory requirements and enhance the security of system data.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-KDF1') &&
            !analytics_1.awsSolutionsKdf1(node)) {
            const ruleId = 'AwsSolutions-KDF1';
            const info = 'The Kinesis Data Firehose delivery stream does have server-side encryption enabled.';
            const explanation = 'This allows the system to meet strict regulatory requirements and enhance the security of system data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-MSK2') &&
            !analytics_1.awsSolutionsMsk2(node)) {
            const ruleId = 'AwsSolutions-MSK2';
            const info = 'The MSK cluster uses plaintext communication between clients and brokers.';
            const explanation = 'TLS only communication secures data-in-transit by encrypting the connection between the clients and brokers.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-MSK3') &&
            !analytics_1.awsSolutionsMsk3(node)) {
            const ruleId = 'AwsSolutions-MSK3';
            const info = 'The MSK cluster uses plaintext communication between brokers.';
            const explanation = 'TLS communication secures data-in-transit by encrypting the connection between brokers.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-MSK6') &&
            !analytics_1.awsSolutionsMsk6(node)) {
            const ruleId = 'AwsSolutions-MSK6';
            const info = 'The MSK cluster does not send broker logs to a supported destination.';
            const explanation = 'Broker logs enable operators to troubleshoot Apache Kafka applications and to analyze their communications with the MSK cluster. The cluster can deliver logs to the following resources: a CloudWatch log group, an S3 bucket, a Kinesis Data Firehose delivery stream.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-QS1') &&
            !analytics_1.awsSolutionsQs1(node)) {
            const ruleId = 'AwsSolutions-QS1';
            const info = 'The Quicksight data sources connection is not configured to use SSL.';
            const explanation = 'SSL secures communications to data sources, especially when using public networks. Using SSL with QuickSight requires the use of certificates signed by a publicly-recognized certificate authority.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Security and Compliance Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkSecurityCompliance(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-IAM4') &&
            !security_and_compliance_1.awsSolutionsIam4(node)) {
            const ruleId = 'AwsSolutions-IAM4';
            const info = 'The IAM user, role, or group uses AWS managed policies.';
            const explanation = 'An AWS managed policy is a standalone policy that is created and administered by AWS. Currently, many AWS managed policies do not restrict resource scope. Replace AWS managed policies with system specific (customer) managed policies.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-IAM5') &&
            !security_and_compliance_1.awsSolutionsIam5(node)) {
            const ruleId = 'AwsSolutions-IAM5';
            const info = 'The IAM entity contains wildcard permissions and does not have a cdk_nag rule suppression with evidence for those permission.';
            const explanation = 'Metadata explaining the evidence (e.g. via supporting links) for wildcard permissions allows for transparency to operators.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-COG1') &&
            !security_and_compliance_1.awsSolutionsCog1(node)) {
            const ruleId = 'AwsSolutions-COG1';
            const info = 'The Cognito user pool does not have a password policy that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters.';
            const explanation = 'Strong password policies increase system security by encouraging users to create reliable and secure passwords.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-COG2') &&
            !security_and_compliance_1.awsSolutionsCog2(node)) {
            const ruleId = 'AwsSolutions-COG2';
            const info = 'The Cognito user pool does not require MFA.';
            const explanation = 'Multi-factor authentication (MFA) increases security for the application by adding another authentication method, and not relying solely on user name and password.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-COG3') &&
            !security_and_compliance_1.awsSolutionsCog3(node)) {
            const ruleId = 'AwsSolutions-COG3';
            const info = 'The Cognito user pool does not have AdvancedSecurityMode set to ENFORCED.';
            const explanation = 'Advanced security features enable the system to detect and act upon malicious sign-in attempts.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-COG7') &&
            !security_and_compliance_1.awsSolutionsCog7(node)) {
            const ruleId = 'AwsSolutions-COG7';
            const info = 'The Cognito identity pool allows for unauthenticated logins and does not have a cdk_nag rule suppression with a reason.';
            const explanation = 'In many cases applications do not warrant unauthenticated guest access applications. Metadata explaining the use case allows for transparency to operators.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-KMS5') &&
            !security_and_compliance_1.awsSolutionsKms5(node)) {
            const ruleId = 'AwsSolutions-KMS5';
            const info = 'The KMS Symmetric CMK does not have Key Rotation enabled.';
            const explanation = 'KMS Key Rotation allow a system to set an yearly rotation schedule for a CMK so when a customer master key is required to encrypt new data, the KMS service can automatically use the latest version of the HSA backing key to perform the encryption.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Serverless Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkServerless(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-SF1') &&
            !serverless_1.awsSolutionsSf1(node)) {
            const ruleId = 'AwsSolutions-SF1';
            const info = 'The Step Function does not log "ALL" events to CloudWatch Logs.';
            const explanation = 'Logging "ALL" events to CloudWatch logs help operators troubleshoot and audit systems.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-SF2') &&
            !serverless_1.awsSolutionsSf2(node)) {
            const ruleId = 'AwsSolutions-SF2';
            const info = 'The Step Function does not have X-Ray tracing enabled.';
            const explanation = 'X-ray provides an end-to-end view of how an application is performing. This helps operators to discover performance issues, detect permission problems, and track requests made to and from other AWS services.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Application Integration Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkApplicationIntegration(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-SNS2') &&
            !application_integration_1.awsSolutionsSns2(node)) {
            const ruleId = 'AwsSolutions-SNS2';
            const info = 'The SNS Topic does not have server-side encryption enabled.';
            const explanation = 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-SQS2') &&
            !application_integration_1.awsSolutionsSqs2(node)) {
            const ruleId = 'AwsSolutions-SQS2';
            const info = 'The SQS Queue does not have server-side encryption enabled.';
            const explanation = 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-SQS3') &&
            !application_integration_1.awsSolutionsSqs3(node)) {
            const ruleId = 'AwsSolutions-SQS3';
            const info = 'The SQS queue does not have a dead-letter queue (DLQ) enabled or have a cdk_nag rule suppression indicating it is a DLQ.';
            const explanation = 'Using a DLQ helps maintain the queue flow and avoid losing data by detecting and mitigating failures and service disruptions on time.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Media Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkMediaServices(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-MS1') &&
            !media_services_1.awsSolutionsMs1(node)) {
            const ruleId = 'AwsSolutions-MS1';
            const info = 'The MediaStore container does not have container access logging enabled.';
            const explanation = 'The container should have access logging enabled to provide detailed records for the requests that are made to the container.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-MS4') &&
            !media_services_1.awsSolutionsMs4(node)) {
            const ruleId = 'AwsSolutions-MS4';
            const info = 'The MediaStore container does not define a metric policy to send metrics to CloudWatch.';
            const explanation = 'Using a combination of MediaStore metrics and CloudWatch alarms helps operators gain better insights into container operations.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-MS7') &&
            !media_services_1.awsSolutionsMs7(node)) {
            const ruleId = 'AwsSolutions-MS7';
            const info = 'The MediaStore container does not define a container policy.';
            const explanation = 'Using a container policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-MS8') &&
            !media_services_1.awsSolutionsMs8(node)) {
            const ruleId = 'AwsSolutions-MS8';
            const info = 'The MediaStore container does not define a CORS policy.';
            const explanation = 'Using a CORS policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-MS10') &&
            !media_services_1.awsSolutionsMs10(node)) {
            const ruleId = 'AwsSolutions-MS10';
            const info = 'The MediaStore container does not define a lifecycle policy.';
            const explanation = 'Many use cases warrant the usage of lifecycle configurations to manage container objects during their lifetime.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Developer Tools Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkDeveloperTools(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-CB3') &&
            !developer_tools_1.awsSolutionsCb3(node)) {
            const ruleId = 'AwsSolutions-CB3';
            const info = 'The CodeBuild project has privileged mode enabled.';
            const explanation = 'Privileged grants elevated rights to the system, which introduces additional risk. Privileged mode should only be set to true only if the build project is used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-CB4') &&
            !developer_tools_1.awsSolutionsCb4(node)) {
            const ruleId = 'AwsSolutions-CB4';
            const info = 'The CodeBuild project does not use a CMK for encryption.';
            const explanation = 'Using a CMK helps follow the standard security advice of granting least privilege to objects generated by the project.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-CB5') &&
            !developer_tools_1.awsSolutionsCb5(node)) {
            const ruleId = 'AwsSolutions-CB5';
            const info = 'The Codebuild project does not use images provided by the CodeBuild service or have a cdk_nag suppression rule explaining the need for a custom image.';
            const explanation = 'Explaining differences/edits to Docker images helps operators better understand system dependencies.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-C91') &&
            !developer_tools_1.awsSolutionsC91(node)) {
            const ruleId = 'AwsSolutions-C91';
            const info = 'The Cloud9 instance does not use a no-ingress EC2 instance with AWS Systems Manager.';
            const explanation = 'SSM adds an additional layer of protection as it allows operators to control access through IAM permissions and does not require opening inbound ports.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
}
exports.AwsSolutionsChecks = AwsSolutionsChecks;
_a = JSII_RTTI_SYMBOL_1;
AwsSolutionsChecks[_a] = { fqn: "monocdk-nag.AwsSolutionsChecks", version: "0.0.40" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzLXNvbHV0aW9ucy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9Bd3NTb2x1dGlvbnMvYXdzLXNvbHV0aW9ucy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBOzs7RUFHRTtBQUNGLHFDQUErRDtBQUMvRCxzQ0FBb0M7QUFDcEMsaURBbUIyQjtBQUMzQiw2RUFJeUM7QUFDekMsaURBZStCO0FBQy9CLG1EQWtDaUM7QUFDakMsNkRBS2lDO0FBQ2pDLDBEQUl3QztBQUN4QyxpRkFJMkM7QUFDM0MsMkRBTWdDO0FBQ2hDLDhEQVU0QztBQUM1Qyw2RUFReUM7QUFDekMsbURBQXNFO0FBQ3RFLGlEQUsrQjs7Ozs7O0FBRy9CLE1BQWEsa0JBQW1CLFNBQVEsZ0JBQU87Ozs7OztJQUN0QyxLQUFLLENBQUMsSUFBZ0I7O1FBQzNCLElBQUksSUFBSSxZQUFZLHFCQUFXLEVBQUU7WUFDL0Isb0NBQW9DO1lBQ3BDLE1BQU0sT0FBTyxTQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLDBDQUFFLGlCQUFpQixDQUFDO1lBQy9ELElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ2pDLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ2pDLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ25DLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDekMsSUFBSSxDQUFDLHlCQUF5QixDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztZQUM5QyxJQUFJLENBQUMsb0JBQW9CLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3pDLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ25DLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDNUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDcEMsSUFBSSxDQUFDLDJCQUEyQixDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNoRCxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3ZDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7U0FDekM7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLFlBQVksQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDbEQsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsd0JBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQ1IsaUVBQWlFLENBQUM7WUFDcEUsTUFBTSxXQUFXLEdBQ2YscWNBQXFjLENBQUM7WUFDeGMscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLHdCQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUFHLHlDQUF5QyxDQUFDO1lBQ3ZELE1BQU0sV0FBVyxHQUNmLDhOQUE4TixDQUFDO1lBQ2pPLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQyx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FBRyxpREFBaUQsQ0FBQztZQUMvRCxNQUFNLFdBQVcsR0FDZix5RkFBeUYsQ0FBQztZQUM1RixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsd0JBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQ1IsOEZBQThGLENBQUM7WUFDakcsTUFBTSxXQUFXLEdBQ2YsNkZBQTZGLENBQUM7WUFDaEcscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLHdCQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUNSLGlGQUFpRixDQUFDO1lBQ3BGLE1BQU0sV0FBVyxHQUNmLHFIQUFxSCxDQUFDO1lBQ3hILHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQyx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FBRyx3Q0FBd0MsQ0FBQztZQUN0RCxNQUFNLFdBQVcsR0FDZix1RkFBdUYsQ0FBQztZQUMxRixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsd0JBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQ1IsNkRBQTZELENBQUM7WUFDaEUsTUFBTSxXQUFXLEdBQ2YsZ0pBQWdKLENBQUM7WUFDbkoscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLHdCQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUNSLCtFQUErRSxDQUFDO1lBQ2xGLE1BQU0sV0FBVyxHQUNmLHVGQUF1RixDQUFDO1lBQzFGLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQyx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FDUixvRUFBb0UsQ0FBQztZQUN2RSxNQUFNLFdBQVcsR0FDZiw2S0FBNkssQ0FBQztZQUNoTCxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG9CQUFvQixDQUFDO1lBQy9DLENBQUMseUJBQWlCLENBQUMsSUFBSSxDQUFDLEVBQ3hCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsb0JBQW9CLENBQUM7WUFDcEMsTUFBTSxJQUFJLEdBQUcsNENBQTRDLENBQUM7WUFDMUQsTUFBTSxXQUFXLEdBQ2YsMkdBQTJHLENBQUM7WUFDOUcscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxvQkFBb0IsQ0FBQztZQUMvQyxDQUFDLHlCQUFpQixDQUFDLElBQUksQ0FBQyxFQUN4QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG9CQUFvQixDQUFDO1lBQ3BDLE1BQU0sSUFBSSxHQUFHLDRDQUE0QyxDQUFDO1lBQzFELE1BQU0sV0FBVyxHQUNmLDJHQUEyRyxDQUFDO1lBQzlHLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQyx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FBRyxvREFBb0QsQ0FBQztZQUNsRSxNQUFNLFdBQVcsR0FDZiwrUUFBK1EsQ0FBQztZQUNsUixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsd0JBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQ1IsMkZBQTJGLENBQUM7WUFDOUYsTUFBTSxXQUFXLEdBQ2YsNk9BQTZPLENBQUM7WUFDaFAscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLHdCQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUNSLGtHQUFrRyxDQUFDO1lBQ3JHLE1BQU0sV0FBVyxHQUNmLGtJQUFrSSxDQUFDO1lBQ3JJLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLFlBQVksQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDbEQsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLGlCQUFpQixDQUFDLElBQUksQ0FBQyxzQkFBYyxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQ3pFLE1BQU0sTUFBTSxHQUFHLGlCQUFpQixDQUFDO1lBQ2pDLE1BQU0sSUFBSSxHQUFHLGdEQUFnRCxDQUFDO1lBQzlELE1BQU0sV0FBVyxHQUNmLGdJQUFnSSxDQUFDO1lBQ25JLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsaUJBQWlCLENBQUMsSUFBSSxDQUFDLHNCQUFjLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDekUsTUFBTSxNQUFNLEdBQUcsaUJBQWlCLENBQUM7WUFDakMsTUFBTSxJQUFJLEdBQ1IsbUVBQW1FLENBQUM7WUFDdEUsTUFBTSxXQUFXLEdBQ2YsNkZBQTZGLENBQUM7WUFDaEcscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsc0JBQWMsQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUN6RSxNQUFNLE1BQU0sR0FBRyxpQkFBaUIsQ0FBQztZQUNqQyxNQUFNLElBQUksR0FBRyxvREFBb0QsQ0FBQztZQUNsRSxNQUFNLFdBQVcsR0FDZiw0RUFBNEUsQ0FBQztZQUMvRSxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsd0JBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQUcsbURBQW1ELENBQUM7WUFDakUsTUFBTSxXQUFXLEdBQ2Ysa1ZBQWtWLENBQUM7WUFDclYscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssY0FBYyxDQUFDLElBQWlCLEVBQUUsT0FBWTtRQUNwRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQyx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FDUiw4RUFBOEUsQ0FBQztZQUNqRixNQUFNLFdBQVcsR0FDZix1SkFBdUosQ0FBQztZQUMxSixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsd0JBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQ1IsNkZBQTZGLENBQUM7WUFDaEcsTUFBTSxXQUFXLEdBQ2YsMExBQTBMLENBQUM7WUFDN0wscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxvQkFBb0IsQ0FBQztZQUMvQyxDQUFDLHlCQUFpQixDQUFDLElBQUksQ0FBQyxFQUN4QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG9CQUFvQixDQUFDO1lBQ3BDLE1BQU0sSUFBSSxHQUNSLCtFQUErRSxDQUFDO1lBQ2xGLE1BQU0sV0FBVyxHQUNmLDZGQUE2RixDQUFDO1lBQ2hHLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsb0JBQW9CLENBQUM7WUFDL0MsQ0FBQyx5QkFBaUIsQ0FBQyxJQUFJLENBQUMsRUFDeEI7WUFDQSxNQUFNLE1BQU0sR0FBRyxvQkFBb0IsQ0FBQztZQUNwQyxNQUFNLElBQUksR0FDUixvRUFBb0UsQ0FBQztZQUN2RSxNQUFNLFdBQVcsR0FDZix1TUFBdU0sQ0FBQztZQUMxTSxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG9CQUFvQixDQUFDO1lBQy9DLENBQUMseUJBQWlCLENBQUMsSUFBSSxDQUFDLEVBQ3hCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsb0JBQW9CLENBQUM7WUFDcEMsTUFBTSxJQUFJLEdBQUcsMkRBQTJELENBQUM7WUFDekUsTUFBTSxXQUFXLEdBQUcscURBQXFELENBQUM7WUFDMUUscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxvQkFBb0IsQ0FBQztZQUMvQyxDQUFDLHlCQUFpQixDQUFDLElBQUksQ0FBQyxFQUN4QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG9CQUFvQixDQUFDO1lBQ3BDLE1BQU0sSUFBSSxHQUNSLCtEQUErRCxDQUFDO1lBQ2xFLE1BQU0sV0FBVyxHQUNmLDJGQUEyRixDQUFDO1lBQzlGLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsb0JBQW9CLENBQUM7WUFDL0MsQ0FBQyx5QkFBaUIsQ0FBQyxJQUFJLENBQUMsRUFDeEI7WUFDQSxNQUFNLE1BQU0sR0FBRyxvQkFBb0IsQ0FBQztZQUNwQyxNQUFNLElBQUksR0FDUixpSEFBaUgsQ0FBQztZQUNwSCxNQUFNLFdBQVcsR0FDZixpR0FBaUcsQ0FBQztZQUNwRyxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsd0JBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQ1Isa0VBQWtFLENBQUM7WUFDckUsTUFBTSxXQUFXLEdBQ2YsdVNBQXVTLENBQUM7WUFDMVMscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsVUFBVSxDQUM3QixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLHdCQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUNSLCtEQUErRCxDQUFDO1lBQ2xFLE1BQU0sV0FBVyxHQUNmLDBLQUEwSyxDQUFDO1lBQzdLLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQyx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FBRyxzREFBc0QsQ0FBQztZQUNwRSxNQUFNLFdBQVcsR0FDZiwySkFBMkosQ0FBQztZQUM5SixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsd0JBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQ1IsNkZBQTZGLENBQUM7WUFDaEcsTUFBTSxXQUFXLEdBQ2YsMklBQTJJLENBQUM7WUFDOUkscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLHdCQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUNSLDRFQUE0RSxDQUFDO1lBQy9FLE1BQU0sV0FBVyxHQUNmLG1GQUFtRixDQUFDO1lBQ3RGLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQyx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FBRyx5REFBeUQsQ0FBQztZQUN2RSxNQUFNLFdBQVcsR0FDZix3S0FBd0ssQ0FBQztZQUMzSyxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsd0JBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQ1IsZ0ZBQWdGLENBQUM7WUFDbkYsTUFBTSxXQUFXLEdBQ2Ysc0pBQXNKLENBQUM7WUFDekoscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsc0JBQWMsQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUN6RSxNQUFNLE1BQU0sR0FBRyxpQkFBaUIsQ0FBQztZQUNqQyxNQUFNLElBQUksR0FDUixxRUFBcUUsQ0FBQztZQUN4RSxNQUFNLFdBQVcsR0FDZixtRkFBbUYsQ0FBQztZQUN0RixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLGlCQUFpQixDQUFDLElBQUksQ0FBQyxzQkFBYyxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQ3pFLE1BQU0sTUFBTSxHQUFHLGlCQUFpQixDQUFDO1lBQ2pDLE1BQU0sSUFBSSxHQUNSLHVFQUF1RSxDQUFDO1lBQzFFLE1BQU0sV0FBVyxHQUNmLDBLQUEwSyxDQUFDO1lBQzdLLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsaUJBQWlCLENBQUMsSUFBSSxDQUFDLHNCQUFjLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDekUsTUFBTSxNQUFNLEdBQUcsaUJBQWlCLENBQUM7WUFDakMsTUFBTSxJQUFJLEdBQ1IsK0ZBQStGLENBQUM7WUFDbEcsTUFBTSxXQUFXLEdBQ2Ysc0xBQXNMLENBQUM7WUFDekwscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsc0JBQWMsQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUN6RSxNQUFNLE1BQU0sR0FBRyxpQkFBaUIsQ0FBQztZQUNqQyxNQUFNLElBQUksR0FDUixrRUFBa0UsQ0FBQztZQUNyRSxNQUFNLFdBQVcsR0FDZiw2SEFBNkgsQ0FBQztZQUNoSSxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLGlCQUFpQixDQUFDLElBQUksQ0FBQyxzQkFBYyxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQ3pFLE1BQU0sTUFBTSxHQUFHLGlCQUFpQixDQUFDO1lBQ2pDLE1BQU0sSUFBSSxHQUNSLDJFQUEyRSxDQUFDO1lBQzlFLE1BQU0sV0FBVyxHQUNmLHFIQUFxSCxDQUFDO1lBQ3hILHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQyx1QkFBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUNSLHFGQUFxRixDQUFDO1lBQ3hGLE1BQU0sV0FBVyxHQUNmLDBJQUEwSSxDQUFDO1lBQzdJLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQyx1QkFBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUFHLG1EQUFtRCxDQUFDO1lBQ2pFLE1BQU0sV0FBVyxHQUNmLDhKQUE4SixDQUFDO1lBQ2pLLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQyx1QkFBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUFHLDJEQUEyRCxDQUFDO1lBQ3pFLE1BQU0sV0FBVyxHQUNmLGdLQUFnSyxDQUFDO1lBQ25LLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQyx1QkFBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUFHLHNEQUFzRCxDQUFDO1lBQ3BFLE1BQU0sV0FBVyxHQUNmLGtKQUFrSixDQUFDO1lBQ3JKLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQyx1QkFBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUFHLDJEQUEyRCxDQUFDO1lBQ3pFLE1BQU0sV0FBVyxHQUNmLHdFQUF3RSxDQUFDO1lBQzNFLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQyx1QkFBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUNSLGdFQUFnRSxDQUFDO1lBQ25FLE1BQU0sV0FBVyxHQUNmLHdEQUF3RCxDQUFDO1lBQzNELHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQyx1QkFBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUFHLDhDQUE4QyxDQUFDO1lBQzVELE1BQU0sV0FBVyxHQUNmLCtEQUErRCxDQUFDO1lBQ2xFLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQyx1QkFBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUNSLDZEQUE2RCxDQUFDO1lBQ2hFLE1BQU0sV0FBVyxHQUNmLHFIQUFxSCxDQUFDO1lBQ3hILHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQyx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FDUiwyRkFBMkYsQ0FBQztZQUM5RixNQUFNLFdBQVcsR0FDZix3SkFBd0osQ0FBQztZQUMzSixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsd0JBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQ1IsbUVBQW1FLENBQUM7WUFDdEUsTUFBTSxXQUFXLEdBQ2YsNkhBQTZILENBQUM7WUFDaEkscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLHdCQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUFHLHlEQUF5RCxDQUFDO1lBQ3ZFLE1BQU0sV0FBVyxHQUNmLGtKQUFrSixDQUFDO1lBQ3JKLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQyx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FDUiw0RkFBNEYsQ0FBQztZQUMvRixNQUFNLFdBQVcsR0FDZix3aUJBQXdpQixDQUFDO1lBQzNpQixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsd0JBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQ1IsZ0dBQWdHLENBQUM7WUFDbkcsTUFBTSxXQUFXLEdBQ2Ysc0xBQXNMLENBQUM7WUFDekwscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLHdCQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUNSLDBHQUEwRyxDQUFDO1lBQzdHLE1BQU0sV0FBVyxHQUNmLG9TQUFvUyxDQUFDO1lBQ3ZTLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLG9CQUFvQixDQUFDLElBQWlCLEVBQUUsT0FBWTtRQUMxRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQyx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FBRywwREFBMEQsQ0FBQztZQUN4RSxNQUFNLFdBQVcsR0FDZixzVEFBc1QsQ0FBQztZQUN6VCxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxVQUFVLENBQzdCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsd0JBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQUcsMkRBQTJELENBQUM7WUFDekUsTUFBTSxXQUFXLEdBQ2YsbUxBQW1MLENBQUM7WUFDdEwscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsVUFBVSxDQUM3QixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLHdCQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUNSLG1FQUFtRSxDQUFDO1lBQ3RFLE1BQU0sV0FBVyxHQUNmLG1LQUFtSyxDQUFDO1lBQ3RLLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFVBQVUsQ0FDN0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQyx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FDUixvRUFBb0UsQ0FBQztZQUN2RSxNQUFNLFdBQVcsR0FDZixnSUFBZ0ksQ0FBQztZQUNuSSxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsd0JBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQ1IsbUZBQW1GLENBQUM7WUFDdEYsTUFBTSxXQUFXLEdBQ2YsdU9BQXVPLENBQUM7WUFDMU8scUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLHdCQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUNSLGtGQUFrRixDQUFDO1lBQ3JGLE1BQU0sV0FBVyxHQUNmLDBHQUEwRyxDQUFDO1lBQzdHLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsb0JBQW9CLENBQUM7WUFDL0MsQ0FBQyx5QkFBaUIsQ0FBQyxJQUFJLENBQUMsRUFDeEI7WUFDQSxNQUFNLE1BQU0sR0FBRyxvQkFBb0IsQ0FBQztZQUNwQyxNQUFNLElBQUksR0FBRywrQ0FBK0MsQ0FBQztZQUM3RCxNQUFNLFdBQVcsR0FDZixvR0FBb0csQ0FBQztZQUN2RyxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG9CQUFvQixDQUFDO1lBQy9DLENBQUMseUJBQWlCLENBQUMsSUFBSSxDQUFDLEVBQ3hCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsb0JBQW9CLENBQUM7WUFDcEMsTUFBTSxJQUFJLEdBQUcsMkNBQTJDLENBQUM7WUFDekQsTUFBTSxXQUFXLEdBQ2YseUxBQXlMLENBQUM7WUFDNUwscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxvQkFBb0IsQ0FBQztZQUMvQyxDQUFDLHlCQUFpQixDQUFDLElBQUksQ0FBQyxFQUN4QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG9CQUFvQixDQUFDO1lBQ3BDLE1BQU0sSUFBSSxHQUNSLDhFQUE4RSxDQUFDO1lBQ2pGLE1BQU0sV0FBVyxHQUNmLDZIQUE2SCxDQUFDO1lBQ2hJLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLHlCQUF5QixDQUFDLElBQWlCLEVBQUUsT0FBWTtRQUMvRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQywyQ0FBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUFHLHlEQUF5RCxDQUFDO1lBQ3ZFLE1BQU0sV0FBVyxHQUNmLHlLQUF5SyxDQUFDO1lBQzVLLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQywyQ0FBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUNSLHlFQUF5RSxDQUFDO1lBQzVFLE1BQU0sV0FBVyxHQUNmLGlIQUFpSCxDQUFDO1lBQ3BILHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQywyQ0FBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUNSLHVGQUF1RixDQUFDO1lBQzFGLE1BQU0sV0FBVyxHQUNmLHNMQUFzTCxDQUFDO1lBQ3pMLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLG9CQUFvQixDQUFDLElBQWlCLEVBQUUsT0FBWTtRQUMxRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQyx1QkFBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUNSLGtFQUFrRSxDQUFDO1lBQ3JFLE1BQU0sV0FBVyxHQUNmLDhIQUE4SCxDQUFDO1lBQ2pJLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQyx1QkFBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUNSLDRFQUE0RSxDQUFDO1lBQy9FLE1BQU0sV0FBVyxHQUNmLGtFQUFrRSxDQUFDO1lBQ3JFLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQyx1QkFBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUNSLHFFQUFxRSxDQUFDO1lBQ3hFLE1BQU0sV0FBVyxHQUNmLCtEQUErRCxDQUFDO1lBQ2xFLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLGNBQWMsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDcEQsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsNEJBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQUcsc0RBQXNELENBQUM7WUFDcEUsTUFBTSxXQUFXLEdBQ2Ysc0hBQXNILENBQUM7WUFDekgscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLDRCQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUFHLG1EQUFtRCxDQUFDO1lBQ2pFLE1BQU0sV0FBVyxHQUNmLHlJQUF5SSxDQUFDO1lBQzVJLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQyw0QkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FDUixvRkFBb0YsQ0FBQztZQUN2RixNQUFNLFdBQVcsR0FDZixtT0FBbU8sQ0FBQztZQUN0TyxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsNEJBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQUcsZ0RBQWdELENBQUM7WUFDOUQsTUFBTSxXQUFXLEdBQ2YsaVFBQWlRLENBQUM7WUFDcFEscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLDRCQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUNSLDhEQUE4RCxDQUFDO1lBQ2pFLE1BQU0sV0FBVyxHQUNmLHlNQUF5TSxDQUFDO1lBQzVNLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQyw0QkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FDUix3RUFBd0UsQ0FBQztZQUMzRSxNQUFNLFdBQVcsR0FDZixzRkFBc0YsQ0FBQztZQUN6RixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsNEJBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQUcsb0RBQW9ELENBQUM7WUFDbEUsTUFBTSxXQUFXLEdBQ2YsZ0lBQWdJLENBQUM7WUFDbkkscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLDRCQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUNSLHlFQUF5RSxDQUFDO1lBQzVFLE1BQU0sV0FBVyxHQUNmLDZIQUE2SCxDQUFDO1lBQ2hJLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQyw0QkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FBRyxxREFBcUQsQ0FBQztZQUNuRSxNQUFNLFdBQVcsR0FDZix1UUFBdVEsQ0FBQztZQUMxUSxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsNEJBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQUcseURBQXlELENBQUM7WUFDdkUsTUFBTSxXQUFXLEdBQ2YsNkhBQTZILENBQUM7WUFDaEkscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLDRCQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUNSLG1HQUFtRyxDQUFDO1lBQ3RHLE1BQU0sV0FBVyxHQUNmLDRGQUE0RixDQUFDO1lBQy9GLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQyw0QkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FDUixtRkFBbUYsQ0FBQztZQUN0RixNQUFNLFdBQVcsR0FDZiwySEFBMkgsQ0FBQztZQUM5SCxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxVQUFVLENBQzdCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsNEJBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQ1Isc0VBQXNFLENBQUM7WUFDekUsTUFBTSxXQUFXLEdBQ2Ysb09BQW9PLENBQUM7WUFDdk8scUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLDRCQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUNSLHFGQUFxRixDQUFDO1lBQ3hGLE1BQU0sV0FBVyxHQUNmLHdHQUF3RyxDQUFDO1lBQzNHLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQyw0QkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FDUiwyRUFBMkUsQ0FBQztZQUM5RSxNQUFNLFdBQVcsR0FDZiw4R0FBOEcsQ0FBQztZQUNqSCxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsNEJBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQ1IsK0RBQStELENBQUM7WUFDbEUsTUFBTSxXQUFXLEdBQ2YseUZBQXlGLENBQUM7WUFDNUYscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLDRCQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUNSLHVFQUF1RSxDQUFDO1lBQzFFLE1BQU0sV0FBVyxHQUNmLDBRQUEwUSxDQUFDO1lBQzdRLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQywyQkFBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUNSLHNFQUFzRSxDQUFDO1lBQ3pFLE1BQU0sV0FBVyxHQUNmLHNNQUFzTSxDQUFDO1lBQ3pNLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLHVCQUF1QixDQUFDLElBQWlCLEVBQUUsT0FBWTtRQUM3RCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQywwQ0FBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FBRyx5REFBeUQsQ0FBQztZQUN2RSxNQUFNLFdBQVcsR0FDZiwyT0FBMk8sQ0FBQztZQUM5TyxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsMENBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQ1IsK0hBQStILENBQUM7WUFDbEksTUFBTSxXQUFXLEdBQ2YsNkhBQTZILENBQUM7WUFDaEkscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLDBDQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUNSLDJMQUEyTCxDQUFDO1lBQzlMLE1BQU0sV0FBVyxHQUNmLGlIQUFpSCxDQUFDO1lBQ3BILHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQywwQ0FBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FBRyw2Q0FBNkMsQ0FBQztZQUMzRCxNQUFNLFdBQVcsR0FDZixxS0FBcUssQ0FBQztZQUN4SyxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxVQUFVLENBQzdCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsMENBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQ1IsMkVBQTJFLENBQUM7WUFDOUUsTUFBTSxXQUFXLEdBQ2YsaUdBQWlHLENBQUM7WUFDcEcscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLDBDQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUNSLHlIQUF5SCxDQUFDO1lBQzVILE1BQU0sV0FBVyxHQUNmLDZKQUE2SixDQUFDO1lBQ2hLLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQywwQ0FBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FBRywyREFBMkQsQ0FBQztZQUN6RSxNQUFNLFdBQVcsR0FDZix3UEFBd1AsQ0FBQztZQUMzUCxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxlQUFlLENBQUMsSUFBaUIsRUFBRSxPQUFZO1FBQ3JELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxrQkFBa0IsQ0FBQztZQUM3QyxDQUFDLDRCQUFlLENBQUMsSUFBSSxDQUFDLEVBQ3RCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsa0JBQWtCLENBQUM7WUFDbEMsTUFBTSxJQUFJLEdBQ1IsaUVBQWlFLENBQUM7WUFDcEUsTUFBTSxXQUFXLEdBQ2Ysd0ZBQXdGLENBQUM7WUFDM0YscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxrQkFBa0IsQ0FBQztZQUM3QyxDQUFDLDRCQUFlLENBQUMsSUFBSSxDQUFDLEVBQ3RCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsa0JBQWtCLENBQUM7WUFDbEMsTUFBTSxJQUFJLEdBQUcsd0RBQXdELENBQUM7WUFDdEUsTUFBTSxXQUFXLEdBQ2YsaU5BQWlOLENBQUM7WUFDcE4scUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssMkJBQTJCLENBQUMsSUFBaUIsRUFBRSxPQUFZO1FBQ2pFLElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLDBDQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUNSLDZEQUE2RCxDQUFDO1lBQ2hFLE1BQU0sV0FBVyxHQUNmLDJHQUEyRyxDQUFDO1lBQzlHLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsbUJBQW1CLENBQUM7WUFDOUMsQ0FBQywwQ0FBZ0IsQ0FBQyxJQUFJLENBQUMsRUFDdkI7WUFDQSxNQUFNLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQztZQUNuQyxNQUFNLElBQUksR0FDUiw2REFBNkQsQ0FBQztZQUNoRSxNQUFNLFdBQVcsR0FDZiwyR0FBMkcsQ0FBQztZQUM5RyxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG1CQUFtQixDQUFDO1lBQzlDLENBQUMsMENBQWdCLENBQUMsSUFBSSxDQUFDLEVBQ3ZCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUM7WUFDbkMsTUFBTSxJQUFJLEdBQ1IsMEhBQTBILENBQUM7WUFDN0gsTUFBTSxXQUFXLEdBQ2YsdUlBQXVJLENBQUM7WUFDMUkscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssa0JBQWtCLENBQUMsSUFBaUIsRUFBRSxPQUFZO1FBQ3hELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxrQkFBa0IsQ0FBQztZQUM3QyxDQUFDLGdDQUFlLENBQUMsSUFBSSxDQUFDLEVBQ3RCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsa0JBQWtCLENBQUM7WUFDbEMsTUFBTSxJQUFJLEdBQ1IsMEVBQTBFLENBQUM7WUFDN0UsTUFBTSxXQUFXLEdBQ2YsK0hBQStILENBQUM7WUFDbEkscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxrQkFBa0IsQ0FBQztZQUM3QyxDQUFDLGdDQUFlLENBQUMsSUFBSSxDQUFDLEVBQ3RCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsa0JBQWtCLENBQUM7WUFDbEMsTUFBTSxJQUFJLEdBQ1IseUZBQXlGLENBQUM7WUFDNUYsTUFBTSxXQUFXLEdBQ2YsaUlBQWlJLENBQUM7WUFDcEkscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsVUFBVSxDQUM3QixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxrQkFBa0IsQ0FBQztZQUM3QyxDQUFDLGdDQUFlLENBQUMsSUFBSSxDQUFDLEVBQ3RCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsa0JBQWtCLENBQUM7WUFDbEMsTUFBTSxJQUFJLEdBQ1IsOERBQThELENBQUM7WUFDakUsTUFBTSxXQUFXLEdBQ2Ysb0xBQW9MLENBQUM7WUFDdkwscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsVUFBVSxDQUM3QixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxrQkFBa0IsQ0FBQztZQUM3QyxDQUFDLGdDQUFlLENBQUMsSUFBSSxDQUFDLEVBQ3RCO1lBQ0EsTUFBTSxNQUFNLEdBQUcsa0JBQWtCLENBQUM7WUFDbEMsTUFBTSxJQUFJLEdBQUcseURBQXlELENBQUM7WUFDdkUsTUFBTSxXQUFXLEdBQ2YsK0tBQStLLENBQUM7WUFDbEwscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsVUFBVSxDQUM3QixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQztZQUM5QyxDQUFDLGlDQUFnQixDQUFDLElBQUksQ0FBQyxFQUN2QjtZQUNBLE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDO1lBQ25DLE1BQU0sSUFBSSxHQUNSLDhEQUE4RCxDQUFDO1lBQ2pFLE1BQU0sV0FBVyxHQUNmLGlIQUFpSCxDQUFDO1lBQ3BILHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFVBQVUsQ0FDN0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLG1CQUFtQixDQUFDLElBQWlCLEVBQUUsT0FBWTtRQUN6RCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQyxpQ0FBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUFHLG9EQUFvRCxDQUFDO1lBQ2xFLE1BQU0sV0FBVyxHQUNmLG9RQUFvUSxDQUFDO1lBQ3ZRLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFVBQVUsQ0FDN0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQyxpQ0FBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUFHLDBEQUEwRCxDQUFDO1lBQ3hFLE1BQU0sV0FBVyxHQUNmLHdIQUF3SCxDQUFDO1lBQzNILHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQyxpQ0FBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUNSLHdKQUF3SixDQUFDO1lBQzNKLE1BQU0sV0FBVyxHQUNmLHNHQUFzRyxDQUFDO1lBQ3pHLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFVBQVUsQ0FDN0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLENBQUM7WUFDN0MsQ0FBQyxpQ0FBZSxDQUFDLElBQUksQ0FBQyxFQUN0QjtZQUNBLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDO1lBQ2xDLE1BQU0sSUFBSSxHQUNSLHNGQUFzRixDQUFDO1lBQ3pGLE1BQU0sV0FBVyxHQUNmLHlKQUF5SixDQUFDO1lBQzVKLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDOztBQTE0Q0gsZ0RBMjRDQyIsInNvdXJjZXNDb250ZW50IjpbIi8qXG5Db3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cblNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4qL1xuaW1wb3J0IHsgQW5ub3RhdGlvbnMsIENmblJlc291cmNlLCBJQ29uc3RydWN0IH0gZnJvbSAnbW9ub2Nkayc7XG5pbXBvcnQgeyBOYWdQYWNrIH0gZnJvbSAnLi4vY29tbW9uJztcbmltcG9ydCB7XG4gIGF3c1NvbHV0aW9uc0F0aDEsXG4gIGF3c1NvbHV0aW9uc0VtcjIsXG4gIGF3c1NvbHV0aW9uc0VtcjYsXG4gIGF3c1NvbHV0aW9uc0VzaDEsXG4gIGF3c1NvbHV0aW9uc0VzaDIsXG4gIGF3c1NvbHV0aW9uc0VzaDMsXG4gIGF3c1NvbHV0aW9uc0VzaDQsXG4gIGF3c1NvbHV0aW9uc0VzaDUsXG4gIGF3c1NvbHV0aW9uc0VzaDcsXG4gIGF3c1NvbHV0aW9uc0VzaDgsXG4gIGF3c1NvbHV0aW9uc0VzaDksXG4gIGF3c1NvbHV0aW9uc0tkYTMsXG4gIGF3c1NvbHV0aW9uc0tkczEsXG4gIGF3c1NvbHV0aW9uc0tkZjEsXG4gIGF3c1NvbHV0aW9uc01zazIsXG4gIGF3c1NvbHV0aW9uc01zazYsXG4gIGF3c1NvbHV0aW9uc01zazMsXG4gIGF3c1NvbHV0aW9uc1FzMSxcbn0gZnJvbSAnLi9ydWxlcy9hbmFseXRpY3MnO1xuaW1wb3J0IHtcbiAgYXdzU29sdXRpb25zU25zMixcbiAgYXdzU29sdXRpb25zU3FzMixcbiAgYXdzU29sdXRpb25zU3FzMyxcbn0gZnJvbSAnLi9ydWxlcy9hcHBsaWNhdGlvbl9pbnRlZ3JhdGlvbic7XG5pbXBvcnQge1xuICBhd3NTb2x1dGlvbnNFYzIzLFxuICBhd3NTb2x1dGlvbnNFYzI2LFxuICBhd3NTb2x1dGlvbnNFYzI3LFxuICBhd3NTb2x1dGlvbnNFYzI4LFxuICBhd3NTb2x1dGlvbnNFYzI5LFxuICBhd3NTb2x1dGlvbnNFY3IxLFxuICBhd3NTb2x1dGlvbnNFY3M0LFxuICBhd3NTb2x1dGlvbnNFY3M3LFxuICBhd3NTb2x1dGlvbnNFbGIxLFxuICBhd3NTb2x1dGlvbnNFbGIyYSxcbiAgYXdzU29sdXRpb25zRWxiMmUsXG4gIGF3c1NvbHV0aW9uc0VsYjMsXG4gIGF3c1NvbHV0aW9uc0VsYjQsXG4gIGF3c1NvbHV0aW9uc0VsYjUsXG59IGZyb20gJy4vcnVsZXMvY29tcHV0ZS9pbmRleCc7XG5pbXBvcnQge1xuICBhd3NTb2x1dGlvbnNBZWMxLFxuICBhd3NTb2x1dGlvbnNBZWMzLFxuICBhd3NTb2x1dGlvbnNBZWM0LFxuICBhd3NTb2x1dGlvbnNBZWM1LFxuICBhd3NTb2x1dGlvbnNBZWM2LFxuICBhd3NTb2x1dGlvbnNEZGIzLFxuICBhd3NTb2x1dGlvbnNEZGI0LFxuICBhd3NTb2x1dGlvbnNEb2MxLFxuICBhd3NTb2x1dGlvbnNEb2MyLFxuICBhd3NTb2x1dGlvbnNEb2MzLFxuICBhd3NTb2x1dGlvbnNEb2M0LFxuICBhd3NTb2x1dGlvbnNEb2M1LFxuICBhd3NTb2x1dGlvbnNOMSxcbiAgYXdzU29sdXRpb25zTjIsXG4gIGF3c1NvbHV0aW9uc04zLFxuICBhd3NTb2x1dGlvbnNONCxcbiAgYXdzU29sdXRpb25zTjUsXG4gIGF3c1NvbHV0aW9uc1JkczEwLFxuICBhd3NTb2x1dGlvbnNSZHMxMSxcbiAgYXdzU29sdXRpb25zUmRzMTMsXG4gIGF3c1NvbHV0aW9uc1JkczE0LFxuICBhd3NTb2x1dGlvbnNSZHMxNixcbiAgYXdzU29sdXRpb25zUmRzMixcbiAgYXdzU29sdXRpb25zUmRzNixcbiAgYXdzU29sdXRpb25zUnMxLFxuICBhd3NTb2x1dGlvbnNSczEwLFxuICBhd3NTb2x1dGlvbnNSczIsXG4gIGF3c1NvbHV0aW9uc1JzMyxcbiAgYXdzU29sdXRpb25zUnM0LFxuICBhd3NTb2x1dGlvbnNSczUsXG4gIGF3c1NvbHV0aW9uc1JzNixcbiAgYXdzU29sdXRpb25zUnM4LFxuICBhd3NTb2x1dGlvbnNSczksXG59IGZyb20gJy4vcnVsZXMvZGF0YWJhc2VzL2luZGV4JztcbmltcG9ydCB7XG4gIGF3c1NvbHV0aW9uc0M5MSxcbiAgYXdzU29sdXRpb25zQ2IzLFxuICBhd3NTb2x1dGlvbnNDYjQsXG4gIGF3c1NvbHV0aW9uc0NiNSxcbn0gZnJvbSAnLi9ydWxlcy9kZXZlbG9wZXJfdG9vbHMnO1xuaW1wb3J0IHtcbiAgYXdzU29sdXRpb25zU20xLFxuICBhd3NTb2x1dGlvbnNTbTIsXG4gIGF3c1NvbHV0aW9uc1NtMyxcbn0gZnJvbSAnLi9ydWxlcy9tYWNoaW5lX2xlYXJuaW5nL2luZGV4JztcbmltcG9ydCB7XG4gIGF3c1NvbHV0aW9uc0FzMSxcbiAgYXdzU29sdXRpb25zQXMyLFxuICBhd3NTb2x1dGlvbnNBczMsXG59IGZyb20gJy4vcnVsZXMvbWFuYWdlbWVudF9hbmRfZ292ZXJuYW5jZSc7XG5pbXBvcnQge1xuICBhd3NTb2x1dGlvbnNNczEsXG4gIGF3c1NvbHV0aW9uc01zMTAsXG4gIGF3c1NvbHV0aW9uc01zNCxcbiAgYXdzU29sdXRpb25zTXM3LFxuICBhd3NTb2x1dGlvbnNNczgsXG59IGZyb20gJy4vcnVsZXMvbWVkaWFfc2VydmljZXMnO1xuaW1wb3J0IHtcbiAgYXdzU29sdXRpb25zQXBpZzEsXG4gIGF3c1NvbHV0aW9uc0FwaWc0LFxuICBhd3NTb2x1dGlvbnNBcGlnNixcbiAgYXdzU29sdXRpb25zQ2ZyMSxcbiAgYXdzU29sdXRpb25zQ2ZyMixcbiAgYXdzU29sdXRpb25zQ2ZyMyxcbiAgYXdzU29sdXRpb25zQ2ZyNSxcbiAgYXdzU29sdXRpb25zQ2ZyNixcbiAgYXdzU29sdXRpb25zVnBjMyxcbn0gZnJvbSAnLi9ydWxlcy9uZXR3b3JrX2FuZF9kZWxpdmVyeS9pbmRleCc7XG5pbXBvcnQge1xuICBhd3NTb2x1dGlvbnNDb2cxLFxuICBhd3NTb2x1dGlvbnNDb2cyLFxuICBhd3NTb2x1dGlvbnNDb2czLFxuICBhd3NTb2x1dGlvbnNDb2c3LFxuICBhd3NTb2x1dGlvbnNJYW00LFxuICBhd3NTb2x1dGlvbnNJYW01LFxuICBhd3NTb2x1dGlvbnNLbXM1LFxufSBmcm9tICcuL3J1bGVzL3NlY3VyaXR5X2FuZF9jb21wbGlhbmNlJztcbmltcG9ydCB7IGF3c1NvbHV0aW9uc1NmMSwgYXdzU29sdXRpb25zU2YyIH0gZnJvbSAnLi9ydWxlcy9zZXJ2ZXJsZXNzJztcbmltcG9ydCB7XG4gIGF3c1NvbHV0aW9uc0VmczEsXG4gIGF3c1NvbHV0aW9uc1MxLFxuICBhd3NTb2x1dGlvbnNTMixcbiAgYXdzU29sdXRpb25zUzMsXG59IGZyb20gJy4vcnVsZXMvc3RvcmFnZS9pbmRleCc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBBd3NTb2x1dGlvbnNDaGVja3MgZXh0ZW5kcyBOYWdQYWNrIHtcbiAgcHVibGljIHZpc2l0KG5vZGU6IElDb25zdHJ1Y3QpOiB2b2lkIHtcbiAgICBpZiAobm9kZSBpbnN0YW5jZW9mIENmblJlc291cmNlKSB7XG4gICAgICAvLyBHZXQgaWdub3JlcyBtZXRhZGF0YSBpZiBpdCBleGlzdHNcbiAgICAgIGNvbnN0IGlnbm9yZXMgPSBub2RlLmdldE1ldGFkYXRhKCdjZGtfbmFnJyk/LnJ1bGVzX3RvX3N1cHByZXNzO1xuICAgICAgdGhpcy5jaGVja0NvbXB1dGUobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrU3RvcmFnZShub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tEYXRhYmFzZXMobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrTmV0d29ya0RlbGl2ZXJ5KG5vZGUsIGlnbm9yZXMpO1xuICAgICAgdGhpcy5jaGVja01hbmFnZW1lbnRHb3Zlcm5hbmNlKG5vZGUsIGlnbm9yZXMpO1xuICAgICAgdGhpcy5jaGVja01hY2hpbmVMZWFybmluZyhub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tBbmFseXRpY3Mobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrU2VjdXJpdHlDb21wbGlhbmNlKG5vZGUsIGlnbm9yZXMpO1xuICAgICAgdGhpcy5jaGVja1NlcnZlcmxlc3Mobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrQXBwbGljYXRpb25JbnRlZ3JhdGlvbihub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tNZWRpYVNlcnZpY2VzKG5vZGUsIGlnbm9yZXMpO1xuICAgICAgdGhpcy5jaGVja0RldmVsb3BlclRvb2xzKG5vZGUsIGlnbm9yZXMpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBDb21wdXRlIFNlcnZpY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBJQ29uc3RydWN0IHRvIGV2YWx1YXRlXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrQ29tcHV0ZShub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KTogdm9pZCB7XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUVDMjMnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0VjMjMobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtRUMyMyc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBTZWN1cml0eSBHcm91cCBhbGxvd3MgZm9yIDAuMC4wLjAvMCBvciA6Oi8wIGluYm91bmQgYWNjZXNzLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdMYXJnZSBwb3J0IHJhbmdlcywgd2hlbiBvcGVuLCBleHBvc2UgaW5zdGFuY2VzIHRvIHVud2FudGVkIGF0dGFja3MuIE1vcmUgdGhhbiB0aGF0LCB0aGV5IG1ha2UgdHJhY2VhYmlsaXR5IG9mIHZ1bG5lcmFiaWxpdGllcyB2ZXJ5IGRpZmZpY3VsdC4gRm9yIGluc3RhbmNlLCB5b3VyIHdlYiBzZXJ2ZXJzIG1heSBvbmx5IHJlcXVpcmUgODAgYW5kIDQ0MyBwb3J0cyB0byBiZSBvcGVuLCBidXQgbm90IGFsbC4gT25lIG9mIHRoZSBtb3N0IGNvbW1vbiBtaXN0YWtlcyBvYnNlcnZlZCBpcyB3aGVuICBhbGwgcG9ydHMgZm9yIDAuMC4wLjAvMCByYW5nZSBhcmUgb3BlbiBpbiBhIHJ1c2ggdG8gYWNjZXNzIHRoZSBpbnN0YW5jZS4gRUMyIGluc3RhbmNlcyBtdXN0IGV4cG9zZSBvbmx5IHRvIHRob3NlIHBvcnRzIGVuYWJsZWQgb24gdGhlIGNvcnJlc3BvbmRpbmcgc2VjdXJpdHkgZ3JvdXAgbGV2ZWwuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtRUMyNicpICYmXG4gICAgICAhYXdzU29sdXRpb25zRWMyNihub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1FQzI2JztcbiAgICAgIGNvbnN0IGluZm8gPSAnVGhlIEVCUyB2b2x1bWUgaGFzIGVuY3J5cHRpb24gZGlzYWJsZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgXCJXaXRoIEVCUyBlbmNyeXB0aW9uLCB5b3UgYXJlbid0IHJlcXVpcmVkIHRvIGJ1aWxkLCBtYWludGFpbiwgYW5kIHNlY3VyZSB5b3VyIG93biBrZXkgbWFuYWdlbWVudCBpbmZyYXN0cnVjdHVyZS4gRUJTIGVuY3J5cHRpb24gdXNlcyBLTVMga2V5cyB3aGVuIGNyZWF0aW5nIGVuY3J5cHRlZCB2b2x1bWVzIGFuZCBzbmFwc2hvdHMuIFRoaXMgaGVscHMgcHJvdGVjdCBkYXRhIGF0IHJlc3QuXCI7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUVDMjcnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0VjMjcobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtRUMyNyc7XG4gICAgICBjb25zdCBpbmZvID0gJ1RoZSBTZWN1cml0eSBHcm91cCBkb2VzIG5vdCBoYXZlIGEgZGVzY3JpcHRpb24uJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0Rlc2NyaXB0aW9ucyBoZWxwIHNpbXBsaWZ5IG9wZXJhdGlvbnMgYW5kIHJlbW92ZSBhbnkgb3Bwb3J0dW5pdGllcyBmb3Igb3BlcmF0b3IgZXJyb3JzLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUVDMjgnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0VjMjgobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtRUMyOCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBFQzIgaW5zdGFuY2UvQXV0b1NjYWxpbmcgbGF1bmNoIGNvbmZpZ3VyYXRpb24gZG9lcyBub3QgaGF2ZSBkZXRhaWxlZCBtb25pdG9yaW5nIGVuYWJsZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ01vbml0b3JpbmcgZGF0YSBoZWxwcyBtYWtlIGJldHRlciBkZWNpc2lvbnMgb24gYXJjaGl0ZWN0aW5nIGFuZCBtYW5hZ2luZyBjb21wdXRlIHJlc291cmNlcy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1FQzI5JykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNFYzI5KG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUVDMjknO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgRUMyIGluc3RhbmNlIGlzIG5vdCBwYXJ0IG9mIGFuIEFTRyBhbmQgaGFzIFRlcm1pbmF0aW9uIFByb3RlY3Rpb24gZGlzYWJsZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1Rlcm1pbmF0aW9uIFByb3RlY3Rpb24gc2FmZXR5IGZlYXR1cmUgZW5hYmxlZCBpbiBvcmRlciB0byBwcm90ZWN0IHRoZSBpbnN0YW5jZXMgZnJvbSBiZWluZyBhY2NpZGVudGFsbHkgdGVybWluYXRlZC4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1FQ1IxJykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNFY3IxKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUVDUjEnO1xuICAgICAgY29uc3QgaW5mbyA9ICdUaGUgRUNSIFJlcG9zaXRvcnkgYWxsb3dzIG9wZW4gYWNjZXNzLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdSZW1vdmluZyAqIHByaW5jaXBhbHMgaW4gYW4gRUNSIFJlcG9zaXRvcnkgaGVscHMgcHJvdGVjdCBhZ2FpbnN0IHVuYXV0aG9yaXplZCBhY2Nlc3MuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtRUNTNCcpICYmXG4gICAgICAhYXdzU29sdXRpb25zRWNzNChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1FQ1M0JztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIEVDUyBDbHVzdGVyIGhhcyBDbG91ZFdhdGNoIENvbnRhaW5lciBJbnNpZ2h0cyBkaXNhYmxlZC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQ2xvdWRXYXRjaCBDb250YWluZXIgSW5zaWdodHMgYWxsb3cgb3BlcmF0b3JzIHRvIGdhaW4gYSBiZXR0ZXIgcGVyc3BlY3RpdmUgb24gaG93IHRoZSBjbHVzdGVy4oCZcyBhcHBsaWNhdGlvbnMgYW5kIG1pY3Jvc2VydmljZXMgYXJlIHBlcmZvcm1pbmcuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtRUNTNycpICYmXG4gICAgICAhYXdzU29sdXRpb25zRWNzNyhub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1FQ1M3JztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIEVDUyBUYXNrIERlZmluaXRpb24gZG9lcyBub3QgaGF2ZSBhd3Nsb2dzIGxvZ2dpbmcgZW5hYmxlZCBhdCB0aGUgbWluaW11bS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQ29udGFpbmVyIGxvZ2dpbmcgYWxsb3dzIG9wZXJhdG9ycyB0byB2aWV3IGFuZCBhZ2dyZWdhdGUgdGhlIGxvZ3MgZnJvbSB0aGUgY29udGFpbmVyLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUVMQjEnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0VsYjEobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtRUxCMSc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBDTEIgaXMgdXNlZCBmb3IgaW5jb21pbmcgSFRUUC9IVFRQUyB0cmFmZmljLiBVc2UgQUxCcyBpbnN0ZWFkLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdIVFRQL0hUVFBTIGFwcGxpY2F0aW9ucyAobW9ub2xpdGhpYyBvciBjb250YWluZXJpemVkKSBzaG91bGQgdXNlIHRoZSBBTEIgaW5zdGVhZCBvZiBUaGUgQ0xCIGZvciBlbmhhbmNlZCBpbmNvbWluZyB0cmFmZmljIGRpc3RyaWJ1dGlvbiwgYmV0dGVyIHBlcmZvcm1hbmNlIGFuZCBsb3dlciBjb3N0cy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1FTEIyYScpICYmXG4gICAgICAhYXdzU29sdXRpb25zRWxiMmEobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtRUxCMmEnO1xuICAgICAgY29uc3QgaW5mbyA9ICdUaGUgQUxCIGRvZXMgbm90IGhhdmUgYWNjZXNzIGxvZ3MgZW5hYmxlZC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQWNjZXNzIGxvZ3MgYWxsb3cgb3BlcmF0b3JzIHRvIHRvIGFuYWx5emUgdHJhZmZpYyBwYXR0ZXJucyBhbmQgaWRlbnRpZnkgYW5kIHRyb3VibGVzaG9vdCBzZWN1cml0eSBpc3N1ZXMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtRUxCMmUnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0VsYjJlKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUVMQjJlJztcbiAgICAgIGNvbnN0IGluZm8gPSAnVGhlIENMQiBkb2VzIG5vdCBoYXZlIGFjY2VzcyBsb2dzIGVuYWJsZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0FjY2VzcyBsb2dzIGFsbG93IG9wZXJhdG9ycyB0byB0byBhbmFseXplIHRyYWZmaWMgcGF0dGVybnMgYW5kIGlkZW50aWZ5IGFuZCB0cm91Ymxlc2hvb3Qgc2VjdXJpdHkgaXNzdWVzLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUVMQjMnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0VsYjMobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtRUxCMyc7XG4gICAgICBjb25zdCBpbmZvID0gJ1RoZSBDTEIgZG9lcyBub3QgaGF2ZSBjb25uZWN0aW9uIGRyYWluaW5nIGVuYWJsZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1dpdGggQ29ubmVjdGlvbiBEcmFpbmluZyBmZWF0dXJlIGVuYWJsZWQsIGlmIGFuIEVDMiBiYWNrZW5kIGluc3RhbmNlIGZhaWxzIGhlYWx0aCBjaGVja3MgVGhlIENMQiB3aWxsIG5vdCBzZW5kIGFueSBuZXcgcmVxdWVzdHMgdG8gdGhlIHVuaGVhbHRoeSBpbnN0YW5jZS4gSG93ZXZlciwgaXQgd2lsbCBzdGlsbCBhbGxvdyBleGlzdGluZyAoaW4tZmxpZ2h0KSByZXF1ZXN0cyB0byBjb21wbGV0ZSBmb3IgdGhlIGR1cmF0aW9uIG9mIHRoZSBjb25maWd1cmVkIHRpbWVvdXQuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtRUxCNCcpICYmXG4gICAgICAhYXdzU29sdXRpb25zRWxiNChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1FTEI0JztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIENMQiBkb2VzIG5vdCB1c2UgYXQgbGVhc3QgdHdvIEFacyB3aXRoIHRoZSBDcm9zcy1ab25lIExvYWQgQmFsYW5jaW5nIGZlYXR1cmUgZW5hYmxlZC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQ0xCcyBjYW4gZGlzdHJpYnV0ZSB0aGUgdHJhZmZpYyBldmVubHkgYWNyb3NzIGFsbCBiYWNrZW5kIGluc3RhbmNlcy4gVG8gdXNlIENyb3NzLVpvbmUgTG9hZCBCYWxhbmNpbmcgYXQgb3B0aW1hbCBsZXZlbCwgdGhlIHN5c3RlbSBzaG91bGQgbWFpbnRhaW4gYW4gZXF1YWwgRUMyIGNhcGFjaXR5IGRpc3RyaWJ1dGlvbiBpbiBlYWNoIG9mIHRoZSBBWnMgcmVnaXN0ZXJlZCB3aXRoIHRoZSBsb2FkIGJhbGFuY2VyLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUVMQjUnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0VsYjUobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtRUxCNSc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBDTEIgbGlzdGVuZXIgaXMgbm90IGNvbmZpZ3VyZWQgZm9yIHNlY3VyZSAoSFRUUHMgb3IgU1NMKSBwcm90b2NvbHMgZm9yIGNsaWVudCBjb21tdW5pY2F0aW9uLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdUaGUgSFRUUHMgb3IgU1NMIHByb3RvY29scyBlbmFibGUgc2VjdXJlIGNvbW11bmljYXRpb24gYnkgZW5jcnlwdGluZyB0aGUgY29tbXVuaWNhdGlvbiBiZXR3ZWVuIHRoZSBjbGllbnQgYW5kIHRoZSBsb2FkIGJhbGFuY2VyLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgU3RvcmFnZSBTZXJ2aWNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgSUNvbnN0cnVjdCB0byBldmFsdWF0ZVxuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja1N0b3JhZ2Uobm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSk6IHZvaWQge1xuICAgIGlmICghdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtUzEnKSAmJiAhYXdzU29sdXRpb25zUzEobm9kZSkpIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtUzEnO1xuICAgICAgY29uc3QgaW5mbyA9ICdUaGUgUzMgQnVja2V0IGhhcyBzZXJ2ZXIgYWNjZXNzIGxvZ3MgZGlzYWJsZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1RoZSBidWNrZXQgc2hvdWxkIGhhdmUgc2VydmVyIGFjY2VzcyBsb2dnaW5nIGVuYWJsZWQgdG8gcHJvdmlkZSBkZXRhaWxlZCByZWNvcmRzIGZvciB0aGUgcmVxdWVzdHMgdGhhdCBhcmUgbWFkZSB0byB0aGUgYnVja2V0Lic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKCF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1TMicpICYmICFhd3NTb2x1dGlvbnNTMihub2RlKSkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1TMic7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBTMyBCdWNrZXQgZG9lcyBub3QgaGF2ZSBwdWJsaWMgYWNjZXNzIHJlc3RyaWN0ZWQgYW5kIGJsb2NrZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1RoZSBidWNrZXQgc2hvdWxkIGhhdmUgcHVibGljIGFjY2VzcyByZXN0cmljdGVkIGFuZCBibG9ja2VkIHRvIHByZXZlbnQgdW5hdXRob3JpemVkIGFjY2Vzcy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmICghdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtUzMnKSAmJiAhYXdzU29sdXRpb25zUzMobm9kZSkpIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtUzMnO1xuICAgICAgY29uc3QgaW5mbyA9ICdUaGUgUzMgQnVja2V0IGRvZXMgbm90IGRlZmF1bHQgZW5jcnlwdGlvbiBlbmFibGVkLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdUaGUgYnVja2V0IHNob3VsZCBtaW5pbWFsbHkgaGF2ZSBTU0UgZW5hYmxlZCB0byBoZWxwIHByb3RlY3QgZGF0YS1hdC1yZXN0Lic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUVGUzEnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0VmczEobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtRUZTMSc7XG4gICAgICBjb25zdCBpbmZvID0gJ1RoZSBFRlMgaXMgbm90IGNvbmZpZ3VyZWQgZm9yIGVuY3J5cHRpb24gYXQgcmVzdC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQnkgdXNpbmcgYW4gZW5jcnlwdGVkIGZpbGUgc3lzdGVtLCBkYXRhIGFuZCBtZXRhZGF0YSBhcmUgYXV0b21hdGljYWxseSBlbmNyeXB0ZWQgYmVmb3JlIGJlaW5nIHdyaXR0ZW4gdG8gdGhlIGZpbGUgc3lzdGVtLiBTaW1pbGFybHksIGFzIGRhdGEgYW5kIG1ldGFkYXRhIGFyZSByZWFkLCB0aGV5IGFyZSBhdXRvbWF0aWNhbGx5IGRlY3J5cHRlZCBiZWZvcmUgYmVpbmcgcHJlc2VudGVkIHRvIHRoZSBhcHBsaWNhdGlvbi4gVGhlc2UgcHJvY2Vzc2VzIGFyZSBoYW5kbGVkIHRyYW5zcGFyZW50bHkgYnkgRUZTIHdpdGhvdXQgcmVxdWlyaW5nIG1vZGlmaWNhdGlvbiBvZiBhcHBsaWNhdGlvbnMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBEYXRhYmFzZSBTZXJ2aWNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgSUNvbnN0cnVjdCB0byBldmFsdWF0ZVxuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0RhdGFiYXNlcyhub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KTogdm9pZCB7XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLVJEUzInKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc1JkczIobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtUkRTMic7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBSRFMgaW5zdGFuY2Ugb3IgQXVyb3JhIGNsdXN0ZXIgZG9lcyBub3QgaGF2ZSBzdG9yYWdlIGVuY3J5cHRpb24gZW5hYmxlZC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnU3RvcmFnZSBlbmNyeXB0aW9uIGhlbHBzIHByb3RlY3QgZGF0YS1hdC1yZXN0IGJ5IGVuY3J5cHRpbmcgdGhlIHVuZGVybHlpbmcgc3RvcmFnZSwgYXV0b21hdGVkIGJhY2t1cHMsIHJlYWQgcmVwbGljYXMsIGFuZCBzbmFwc2hvdHMgZm9yIHRoZSBkYXRhYmFzZS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1SRFM2JykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNSZHM2KG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLVJEUzYnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgUkRTIEF1cm9yYSBNeVNRTC9Qb3N0Z3Jlc1NRTCBjbHVzdGVyIGRvZXMgbm90IGhhdmUgSUFNIERhdGFiYXNlIEF1dGhlbnRpY2F0aW9uIGVuYWJsZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgXCJXaXRoIElBTSBEYXRhYmFzZSBBdXRoZW50aWNhdGlvbiBlbmFibGVkLCB0aGUgc3lzdGVtIGRvZXNuJ3QgaGF2ZSB0byB1c2UgYSBwYXNzd29yZCB3aGVuIGNvbm5lY3RpbmcgdG8gdGhlIE15U1FML1Bvc3RncmVTUUwgZGF0YWJhc2UgaW5zdGFuY2VzLCBpbnN0ZWFkIGl0IHVzZXMgYW4gYXV0aGVudGljYXRpb24gdG9rZW4uXCI7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLVJEUzEwJykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNSZHMxMChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1SRFMxMCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBSRFMgaW5zdGFuY2Ugb3IgQXVyb3JhIGNsdXN0ZXIgZG9lcyBub3QgaGF2ZSBkZWxldGlvbiBwcm90ZWN0aW9uIGVuYWJsZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1RoZSBkZWxldGlvbiBwcm90ZWN0aW9uIGZlYXR1cmUgaGVscHMgcHJvdGVjdCB0aGUgZGF0YWJhc2UgZnJvbSBiZWluZyBhY2NpZGVudGFsbHkgZGVsZXRlZC4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1SRFMxMScpICYmXG4gICAgICAhYXdzU29sdXRpb25zUmRzMTEobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtUkRTMTEnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgUkRTIGluc3RhbmNlIG9yIEF1cm9yYSBjbHVzdGVyIHVzZXMgdGhlIGRlZmF1bHQgZW5kcG9pbnQgcG9ydC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnUG9ydCBvYmZ1c2NhdGlvbiAodXNpbmcgYSBub24gZGVmYXVsdCBlbmRwb2ludCBwb3J0KSBhZGRzIGFuIGFkZGl0aW9uYWwgbGF5ZXIgb2YgZGVmZW5zZSBhZ2FpbnN0IG5vbi10YXJnZXRlZCBhdHRhY2tzIChpLmUuIE15U1FML0F1cm9yYSBwb3J0IDMzMDYsIFNRTCBTZXJ2ZXIgcG9ydCAxNDMzLCBQb3N0Z3JlU1FMIHBvcnQgNTQzMiwgZXRjKS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1SRFMxMycpICYmXG4gICAgICAhYXdzU29sdXRpb25zUmRzMTMobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtUkRTMTMnO1xuICAgICAgY29uc3QgaW5mbyA9ICdUaGUgUkRTIGluc3RhbmNlIGlzIG5vdCBjb25maWd1cmVkIGZvciBhdXRvbWF0ZWQgYmFja3Vwcy4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPSAnQXV0b21hdGVkIGJhY2t1cHMgYWxsb3cgZm9yIHBvaW50LWluLXRpbWUgcmVjb3ZlcnkuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtUkRTMTQnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc1JkczE0KG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLVJEUzE0JztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIFJEUyBBdXJvcmEgTXlTUUwgY2x1c3RlciBkb2VzIG5vdCBoYXZlIEJhY2t0cmFjayBlbmFibGVkLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdCYWNrdHJhY2sgaGVscHMgb3JkZXIgdG8gcmV3aW5kIGNsdXN0ZXIgdGFibGVzIHRvIGEgc3BlY2lmaWMgdGltZSwgd2l0aG91dCB1c2luZyBiYWNrdXBzLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLVJEUzE2JykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNSZHMxNihub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1SRFMxNic7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBSRFMgQXVyb3JhIE15U1FMIHNlcnZlcmxlc3MgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGF1ZGl0LCBlcnJvciwgZ2VuZXJhbCwgYW5kIHNsb3dxdWVyeSBMb2cgRXhwb3J0cyBlbmFibGVkLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdUaGlzIGFsbG93cyBvcGVyYXRvcnMgdG8gdXNlIENsb3VkV2F0Y2ggdG8gdmlldyBsb2dzIHRvIGhlbHAgZGlhZ25vc2UgcHJvYmxlbXMgaW4gdGhlIGRhdGFiYXNlLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUREQjMnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0RkYjMobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtRERCMyc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBEeW5hbW9EQiB0YWJsZSBkb2VzIG5vdCBoYXZlIFBvaW50LWluLXRpbWUgUmVjb3ZlcnkgZW5hYmxlZC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnRHluYW1vREIgY29udGludW91cyBiYWNrdXBzIHJlcHJlc2VudCBhbiBhZGRpdGlvbmFsIGxheWVyIG9mIGluc3VyYW5jZSBhZ2FpbnN0IGFjY2lkZW50YWwgbG9zcyBvZiBkYXRhIG9uIHRvcCBvZiBvbi1kZW1hbmQgYmFja3Vwcy4gVGhlIER5bmFtb0RCIHNlcnZpY2UgY2FuIGJhY2sgdXAgdGhlIGRhdGEgd2l0aCBwZXItc2Vjb25kIGdyYW51bGFyaXR5IGFuZCByZXN0b3JlIGl0IHRvIGFueSBzaW5nbGUgc2Vjb25kIGZyb20gdGhlIHRpbWUgUElUUiB3YXMgZW5hYmxlZCB1cCB0byB0aGUgcHJpb3IgMzUgZGF5cy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkV2FybmluZyhcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUREQjQnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0RkYjQobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtRERCNCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBEQVggY2x1c3RlciBkb2VzIG5vdCBoYXZlIHNlcnZlci1zaWRlIGVuY3J5cHRpb24gZW5hYmxlZC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnRGF0YSBpbiBjYWNoZSwgY29uZmlndXJhdGlvbiBkYXRhIGFuZCBsb2cgZmlsZXMgc2hvdWxkIGJlIGVuY3J5cHRlZCB1c2luZyBTZXJ2ZXItU2lkZSBFbmNyeXB0aW9uIGluIG9yZGVyIHRvIHByb3RlY3QgZnJvbSB1bmF1dGhvcml6ZWQgYWNjZXNzIHRvIHRoZSB1bmRlcmx5aW5nIHN0b3JhZ2UuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtQUVDMScpICYmXG4gICAgICAhYXdzU29sdXRpb25zQWVjMShub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1BRUMxJztcbiAgICAgIGNvbnN0IGluZm8gPSAnVGhlIEVsYXN0aUNhY2hlIGNsdXN0ZXIgaXMgbm90IHByb3Zpc2lvbmVkIGluIGEgVlBDLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdQcm92aXNpb25pbmcgdGhlIGNsdXN0ZXIgd2l0aGluIGEgVlBDIGFsbG93cyBmb3IgYmV0dGVyIGZsZXhpYmlsaXR5IGFuZCBjb250cm9sIG92ZXIgdGhlIGNhY2hlIGNsdXN0ZXJzIHNlY3VyaXR5LCBhdmFpbGFiaWxpdHksIHRyYWZmaWMgcm91dGluZyBhbmQgbW9yZS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1BRUMzJykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNBZWMzKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUFFQzMnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgRWxhc3RpQ2FjaGUgUmVkaXMgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGJvdGggZW5jcnlwdGlvbiBpbiB0cmFuc2l0IGFuZCBhdCByZXN0IGVuYWJsZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0VuY3J5cHRpb24gaW4gdHJhbnNpdCBoZWxwcyBzZWN1cmUgY29tbXVuaWNhdGlvbnMgdG8gdGhlIGNsdXN0ZXIuIEVuY3J5cHRpb24gYXQgcmVzdCBoZWxwcyBwcm90ZWN0IGRhdGEgYXQgcmVzdCBmcm9tIHVuYXV0aG9yaXplZCBhY2Nlc3MuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtQUVDNCcpICYmXG4gICAgICAhYXdzU29sdXRpb25zQWVjNChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1BRUM0JztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIEVsYXN0aUNhY2hlIFJlZGlzIGNsdXN0ZXIgaXMgbm90IGRlcGxveWVkIGluIGEgTXVsdGktQVogY29uZmlndXJhdGlvbi4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnVGhlIGNsdXN0ZXIgc2hvdWxkIHVzZSBhIE11bHRpLUFaIGRlcGxveW1lbnQgY29uZmlndXJhdGlvbiBmb3IgaGlnaCBhdmFpbGFiaWxpdHkuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtQUVDNScpICYmXG4gICAgICAhYXdzU29sdXRpb25zQWVjNShub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1BRUM1JztcbiAgICAgIGNvbnN0IGluZm8gPSAnVGhlIEVsYXN0aUNhY2hlIGNsdXN0ZXIgdXNlcyB0aGUgZGVmYXVsdCBlbmRwb2ludCBwb3J0Lic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdQb3J0IG9iZnVzY2F0aW9uICh1c2luZyBhIG5vbiBkZWZhdWx0IGVuZHBvaW50IHBvcnQpIGFkZHMgYW4gYWRkaXRpb25hbCBsYXllciBvZiBkZWZlbnNlIGFnYWluc3Qgbm9uLXRhcmdldGVkIGF0dGFja3MgKGkuZS4gUmVkaXMgcG9ydCA2Mzc5IGFuZCBNZW1jYWNoZWQgcG9ydCAxMTIxMSkuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtQUVDNicpICYmXG4gICAgICAhYXdzU29sdXRpb25zQWVjNihub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1BRUM2JztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIEVsYXN0aUNhY2hlIFJlZGlzIGNsdXN0ZXIgZG9lcyBub3QgdXNlIFJlZGlzIEFVVEggZm9yIHVzZXIgYXV0aGVudGljYXRpb24uJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1JlZGlzIGF1dGhlbnRpY2F0aW9uIHRva2VucyBlbmFibGUgUmVkaXMgdG8gcmVxdWlyZSBhIHRva2VuIChwYXNzd29yZCkgYmVmb3JlIGFsbG93aW5nIGNsaWVudHMgdG8gZXhlY3V0ZSBjb21tYW5kcywgdGhlcmVieSBpbXByb3ZpbmcgZGF0YSBzZWN1cml0eS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmICghdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtTjEnKSAmJiAhYXdzU29sdXRpb25zTjEobm9kZSkpIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtTjEnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgTmVwdHVuZSBEQiBjbHVzdGVyIGlzIG5vdCBkZXBsb3llZCBpbiBhIE11bHRpLUFaIGNvbmZpZ3VyYXRpb24uJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1RoZSBjbHVzdGVyIHNob3VsZCB1c2UgYSBNdWx0aS1BWiBkZXBsb3ltZW50IGNvbmZpZ3VyYXRpb24gZm9yIGhpZ2ggYXZhaWxhYmlsaXR5Lic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKCF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1OMicpICYmICFhd3NTb2x1dGlvbnNOMihub2RlKSkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1OMic7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBOZXB0dW5lIERCIGluc3RhbmNlIGRvZXMgaGF2ZSBBdXRvIE1pbm9yIFZlcnNpb24gVXBncmFkZSBlbmFibGVkLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdUaGUgTmVwdHVuZSBzZXJ2aWNlIHJlZ3VsYXJseSByZWxlYXNlcyBlbmdpbmUgdXBkYXRlcy4gRW5hYmxpbmcgQXV0byBNaW5vciBWZXJzaW9uIFVwZ3JhZGUgd2lsbCBhbGxvdyB0aGUgc2VydmljZSB0byBhdXRvbWF0aWNhbGx5IGFwcGx5IHRoZXNlIHVwZ3JhZGVzIHRvIERCIEluc3RhbmNlcy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmICghdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtTjMnKSAmJiAhYXdzU29sdXRpb25zTjMobm9kZSkpIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtTjMnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgTmVwdHVuZSBEQiBjbHVzdGVyIGRvZXMgbm90IGhhdmUgYSByZWFzb25hYmxlIG1pbmltdW0gYmFja3VwIHJldGVudGlvbiBwZXJpb2QgY29uZmlndXJlZC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnVGhlIHJldGVudGlvbiBwZXJpb2QgcmVwcmVzZW50cyB0aGUgbnVtYmVyIG9mIGRheXMgdG8gcmV0YWluIGF1dG9tYXRlZCBzbmFwc2hvdHMuIEEgbWluaW11bSByZXRlbnRpb24gcGVyaW9kIG9mIDcgZGF5cyBpcyByZWNvbW1lbmRlZCBidXQgY2FuIGJlIGFkanVzdCB0byBtZWV0IHN5c3RlbSByZXF1aXJlbWVudHMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLU40JykgJiYgIWF3c1NvbHV0aW9uc040KG5vZGUpKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLU40JztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIE5lcHR1bmUgREIgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGVuY3J5cHRpb24gYXQgcmVzdCBlbmFibGVkLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdFbmNyeXB0aW5nIGRhdGEtYXQtcmVzdCBwcm90ZWN0cyBkYXRhIGNvbmZpZGVudGlhbGl0eSBhbmQgcHJldmVudHMgdW5hdXRob3JpemVkIHVzZXJzIGZyb20gYWNjZXNzaW5nIHNlbnNpdGl2ZSBpbmZvcm1hdGlvbi4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmICghdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtTjUnKSAmJiAhYXdzU29sdXRpb25zTjUobm9kZSkpIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtTjUnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgTmVwdHVuZSBEQiBjbHVzdGVyIGRvZXMgbm90IGhhdmUgSUFNIERhdGFiYXNlIEF1dGhlbnRpY2F0aW9uIGVuYWJsZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgXCJXaXRoIElBTSBEYXRhYmFzZSBBdXRoZW50aWNhdGlvbiBlbmFibGVkLCB0aGUgc3lzdGVtIGRvZXNuJ3QgaGF2ZSB0byB1c2UgYSBwYXNzd29yZCB3aGVuIGNvbm5lY3RpbmcgdG8gdGhlIGNsdXN0ZXIuXCI7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLVJTMScpICYmXG4gICAgICAhYXdzU29sdXRpb25zUnMxKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLVJTMSc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBSZWRzaGlmdCBjbHVzdGVyIHBhcmFtZXRlciBncm91cCBtdXN0IGhhdmUgdGhlIFwicmVxdWlyZV9zc2xcIiBwYXJhbWV0ZXIgZW5hYmxlZC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnRW5hYmxpbmcgdGhlIFwicmVxdWlyZV9zc2xcIiBwYXJhbWV0ZXIgc2VjdXJlcyBkYXRhLWluLXRyYW5zaXQgYnkgZW5jcnlwdGluZyB0aGUgY29ubmVjdGlvbiBiZXR3ZWVuIHRoZSBjbGllbnRzIGFuZCB0aGUgUmVkc2hpZnQgY2x1c3RlcnMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtUlMyJykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNSczIobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtUlMyJztcbiAgICAgIGNvbnN0IGluZm8gPSAnVGhlIFJlZHNoaWZ0IGNsdXN0ZXIgaXMgbm90IHByb3Zpc2lvbmVkIGluIGEgVlBDLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdQcm92aXNpb25pbmcgdGhlIGNsdXN0ZXIgd2l0aGluIGEgVlBDIGFsbG93cyBmb3IgYmV0dGVyIGZsZXhpYmlsaXR5IGFuZCBjb250cm9sIG92ZXIgdGhlIFJlZHNoaWZ0IGNsdXN0ZXJzIHNlY3VyaXR5LCBhdmFpbGFiaWxpdHksIHRyYWZmaWMgcm91dGluZyBhbmQgbW9yZS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1SUzMnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc1JzMyhub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1SUzMnO1xuICAgICAgY29uc3QgaW5mbyA9ICdUaGUgUmVkc2hpZnQgY2x1c3RlciB1c2VzIHRoZSBkZWZhdWx0IFwiYXdzdXNlclwiIHVzZXJuYW1lLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdVc2luZyBhIGN1c3RvbSBtYXN0ZXIgdXNlciBuYW1lIGluc3RlYWQgb2YgdGhlIGRlZmF1bHQgbWFzdGVyIHVzZXIgbmFtZSAoaS5lLiBcImF3c3VzZXJcIikgcHJvdmlkZXMgYW4gYWRkaXRpb25hbCBsYXllciBvZiBkZWZlbnNlIGFnYWluc3Qgbm9uLXRhcmdldGVkIGF0dGFja3MuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtUlM0JykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNSczQobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtUlM0JztcbiAgICAgIGNvbnN0IGluZm8gPSAnVGhlIFJlZHNoaWZ0IGNsdXN0ZXIgdXNlcyB0aGUgZGVmYXVsdCBlbmRwb2ludCBwb3J0Lic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdQb3J0IG9iZnVzY2F0aW9uICh1c2luZyBhIG5vbiBkZWZhdWx0IGVuZHBvaW50IHBvcnQpIGFkZHMgYW4gYWRkaXRpb25hbCBsYXllciBvZiBkZWZlbnNlIGFnYWluc3Qgbm9uLXRhcmdldGVkIGF0dGFja3MgKGkuZS4gUmVkc2hpZnQgcG9ydCA1NDM5KS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1SUzUnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc1JzNShub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1SUzUnO1xuICAgICAgY29uc3QgaW5mbyA9ICdUaGUgUmVkc2hpZnQgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGF1ZGl0IGxvZ2dpbmcgZW5hYmxlZC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQXVkaXQgbG9nZ2luZyBoZWxwcyBvcGVyYXRvcnMgdHJvdWJsZXNob290IGlzc3VlcyBhbmQgZW5zdXJlIHNlY3VyaXR5Lic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLVJTNicpICYmXG4gICAgICAhYXdzU29sdXRpb25zUnM2KG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLVJTNic7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBSZWRzaGlmdCBjbHVzdGVyIGRvZXMgbm90IGhhdmUgZW5jcnlwdGlvbiBhdCByZXN0IGVuYWJsZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0VuY3J5cHRpbmcgZGF0YS1hdC1yZXN0IHByb3RlY3RzIGRhdGEgY29uZmlkZW50aWFsaXR5Lic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLVJTOCcpICYmXG4gICAgICAhYXdzU29sdXRpb25zUnM4KG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLVJTOCc7XG4gICAgICBjb25zdCBpbmZvID0gJ1RoZSBSZWRzaGlmdCBjbHVzdGVyIGlzIHB1YmxpY2x5IGFjY2Vzc2libGUuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0Rpc2FibGluZyBwdWJsaWMgYWNjZXNzaWJpbGl0eSBoZWxwcyBtaW5pbWl6ZSBzZWN1cml0eSByaXNrcy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1SUzknKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc1JzOShub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1SUzknO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgUmVkc2hpZnQgY2x1c3RlciBkb2VzIG5vdCBoYXZlIHZlcnNpb24gdXBncmFkZSBlbmFibGVkLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdWZXJzaW9uIFVwZ3JhZGUgbXVzdCBlbmFibGVkIHRvIGVuYWJsZSB0aGUgY2x1c3RlciB0byBhdXRvbWF0aWNhbGx5IHJlY2VpdmUgdXBncmFkZXMgZHVyaW5nIHRoZSBtYWludGVuYW5jZSB3aW5kb3cuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtUlMxMCcpICYmXG4gICAgICAhYXdzU29sdXRpb25zUnMxMChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1SUzEwJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIFJlZHNoaWZ0IGNsdXN0ZXIgZG9lcyBub3QgaGF2ZSBhIHJldGVudGlvbiBwZXJpb2QgZm9yIGF1dG9tYXRlZCBzbmFwc2hvdHMgY29uZmlndXJlZC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnVGhlIHJldGVudGlvbiBwZXJpb2QgcmVwcmVzZW50cyB0aGUgbnVtYmVyIG9mIGRheXMgdG8gcmV0YWluIGF1dG9tYXRlZCBzbmFwc2hvdHMuIEEgcG9zaXRpdmUgcmV0ZW50aW9uIHBlcmlvZCBzaG91bGQgYmUgc2V0IHRvIGNvbmZpZ3VyZSB0aGlzIGZlYXR1cmUuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtRE9DMScpICYmXG4gICAgICAhYXdzU29sdXRpb25zRG9jMShub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1ET0MxJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIERvY3VtZW50IERCIGNsdXN0ZXIgZG9lcyBub3QgaGF2ZSBlbmNyeXB0aW9uIGF0IHJlc3QgZW5hYmxlZC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnRW5jcnlwdGluZyBkYXRhLWF0LXJlc3QgcHJvdGVjdHMgZGF0YSBjb25maWRlbnRpYWxpdHkgYW5kIHByZXZlbnRzIHVuYXV0aG9yaXplZCB1c2VycyBmcm9tIGFjY2Vzc2luZyBzZW5zaXRpdmUgaW5mb3JtYXRpb24uJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtRE9DMicpICYmXG4gICAgICAhYXdzU29sdXRpb25zRG9jMihub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1ET0MyJztcbiAgICAgIGNvbnN0IGluZm8gPSAnVGhlIERvY3VtZW50IERCIGNsdXN0ZXIgdXNlcyB0aGUgZGVmYXVsdCBlbmRwb2ludCBwb3J0Lic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdQb3J0IG9iZnVzY2F0aW9uICh1c2luZyBhIG5vbiBkZWZhdWx0IGVuZHBvaW50IHBvcnQpIGFkZHMgYW4gYWRkaXRpb25hbCBsYXllciBvZiBkZWZlbnNlIGFnYWluc3Qgbm9uLXRhcmdldGVkIGF0dGFja3MgKGkuZS4gTW9uZ29EQiBwb3J0IDI3MDE3KS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1ET0MzJykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNEb2MzKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLURPQzMnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgRG9jdW1lbnQgREIgY2x1c3RlciBkb2VzIG5vdCBoYXZlIHRoZSB1c2VybmFtZSBhbmQgcGFzc3dvcmQgc3RvcmVkIGluIFNlY3JldHMgTWFuYWdlci4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICBcIlNlY3JldHMgTWFuYWdlciBlbmFibGVzIG9wZXJhdG9ycyB0byByZXBsYWNlIGhhcmRjb2RlZCBjcmVkZW50aWFscyBpbiB5b3VyIGNvZGUsIGluY2x1ZGluZyBwYXNzd29yZHMsIHdpdGggYW4gQVBJIGNhbGwgdG8gU2VjcmV0cyBNYW5hZ2VyIHRvIHJldHJpZXZlIHRoZSBzZWNyZXQgcHJvZ3JhbW1hdGljYWxseS4gVGhpcyBoZWxwcyBlbnN1cmUgdGhlIHNlY3JldCBjYW4ndCBiZSBjb21wcm9taXNlZCBieSBzb21lb25lIGV4YW1pbmluZyBzeXN0ZW0gY29kZSwgYmVjYXVzZSB0aGUgc2VjcmV0IG5vIGxvbmdlciBleGlzdHMgaW4gdGhlIGNvZGUuIEFsc28sIG9wZXJhdG9ycyBjYW4gY29uZmlndXJlIFNlY3JldHMgTWFuYWdlciB0byBhdXRvbWF0aWNhbGx5IHJvdGF0ZSB0aGUgc2VjcmV0IGZvciB5b3UgYWNjb3JkaW5nIHRvIGEgc3BlY2lmaWVkIHNjaGVkdWxlLiBUaGlzIGVuYWJsZXMgeW91IHRvIHJlcGxhY2UgbG9uZy10ZXJtIHNlY3JldHMgd2l0aCBzaG9ydC10ZXJtIG9uZXMsIHNpZ25pZmljYW50bHkgcmVkdWNpbmcgdGhlIHJpc2sgb2YgY29tcHJvbWlzZS5cIjtcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtRE9DNCcpICYmXG4gICAgICAhYXdzU29sdXRpb25zRG9jNChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1ET0M0JztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIERvY3VtZW50IERCIGNsdXN0ZXIgZG9lcyBub3QgaGF2ZSBhIHJlYXNvbmFibGUgbWluaW11bSBiYWNrdXAgcmV0ZW50aW9uIHBlcmlvZCBjb25maWd1cmVkLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdUaGUgcmV0ZW50aW9uIHBlcmlvZCByZXByZXNlbnRzIHRoZSBudW1iZXIgb2YgZGF5cyB0byByZXRhaW4gYXV0b21hdGVkIHNuYXBzaG90cy4gQSBtaW5pbXVtIHJldGVudGlvbiBwZXJpb2Qgb2YgNyBkYXlzIGlzIHJlY29tbWVuZGVkIGJ1dCBjYW4gYmUgYWRqdXN0IHRvIG1lZXQgc3lzdGVtIHJlcXVpcmVtZW50cy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1ET0M1JykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNEb2M1KG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLURPQzUnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgRG9jdW1lbnQgREIgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGF1dGhlbnRpY2F0ZSwgY3JlYXRlSW5kZXgsIGFuZCBkcm9wQ29sbGVjdGlvbiBMb2cgRXhwb3J0cyBlbmFibGVkLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdUaGlzIGFsbG93cyBvcGVyYXRvcnMgdG8gdXNlIENsb3VkV2F0Y2ggdG8gdmlldyBsb2dzIHRvIGhlbHAgZGlhZ25vc2UgcHJvYmxlbXMgaW4gdGhlIGRhdGFiYXNlLiBUaGUgZXZlbnRzIHJlY29yZGVkIGJ5IHRoZSBBV1MgRG9jdW1lbnREQiBhdWRpdCBsb2dzIGluY2x1ZGUgc3VjY2Vzc2Z1bCBhbmQgZmFpbGVkIGF1dGhlbnRpY2F0aW9uIGF0dGVtcHRzLCBjcmVhdGluZyBpbmRleGVzIG9yIGRyb3BwaW5nIGEgY29sbGVjdGlvbiBpbiBhIGRhdGFiYXNlIHdpdGhpbiB0aGUgRG9jdW1lbnREQiBjbHVzdGVyLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgTmV0d29yayBhbmQgRGVsaXZlcnkgU2VydmljZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIElDb25zdHJ1Y3QgdG8gZXZhbHVhdGVcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tOZXR3b3JrRGVsaXZlcnkobm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSk6IHZvaWQge1xuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1WUEMzJykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNWcGMzKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLVZQQzMnO1xuICAgICAgY29uc3QgaW5mbyA9ICdBIE5ldHdvcmsgQUNMIG9yIE5ldHdvcmsgQUNMIGVudHJ5IGhhcyBiZWVuIGltcGxlbWVudGVkLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdOZXR3b3JrIEFDTHMgc2hvdWxkIGJlIHVzZWQgc3BhcmluZ2x5IGZvciB0aGUgZm9sbG93aW5nIHJlYXNvbnM6IHRoZXkgY2FuIGJlIGNvbXBsZXggdG8gbWFuYWdlLCB0aGV5IGFyZSBzdGF0ZWxlc3MsIGV2ZXJ5IElQIGFkZHJlc3MgbXVzdCBiZSBleHBsaWNpdGx5IG9wZW5lZCBpbiBlYWNoIChpbmJvdW5kL291dGJvdW5kKSBkaXJlY3Rpb24sIGFuZCB0aGV5IGFmZmVjdCBhIGNvbXBsZXRlIHN1Ym5ldC4gVXNlIHNlY3VyaXR5IGdyb3VwcyB3aGVuIHBvc3NpYmxlIGFzIHRoZXkgYXJlIHN0YXRlZnVsIGFuZCBlYXNpZXIgdG8gbWFuYWdlLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRXYXJuaW5nKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtQ0ZSMScpICYmXG4gICAgICAhYXdzU29sdXRpb25zQ2ZyMShub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1DRlIxJztcbiAgICAgIGNvbnN0IGluZm8gPSAnVGhlIENsb3VkRnJvbnQgZGlzdHJpYnV0aW9uIG1heSByZXF1aXJlIEdlbyByZXN0cmljdGlvbnMuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0dlbyByZXN0cmljdGlvbiBtYXkgbmVlZCB0byBiZSBlbmFibGVkIGZvciB0aGUgZGlzdHJpYnV0aW9uIGluIG9yZGVyIHRvIGFsbG93IG9yIGRlbnkgYSBjb3VudHJ5IGluIG9yZGVyIHRvIGFsbG93IG9yIHJlc3RyaWN0IHVzZXJzIGluIHNwZWNpZmljIGxvY2F0aW9ucyBmcm9tIGFjY2Vzc2luZyBjb250ZW50Lic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRXYXJuaW5nKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtQ0ZSMicpICYmXG4gICAgICAhYXdzU29sdXRpb25zQ2ZyMihub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1DRlIyJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIENsb3VkRnJvbnQgZGlzdHJpYnV0aW9uIG1heSByZXF1aXJlIGludGVncmF0aW9uIHdpdGggQVdTIFdBRi4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnVGhlIFdlYiBBcHBsaWNhdGlvbiBGaXJld2FsbCBjYW4gaGVscCBwcm90ZWN0IGFnYWluc3QgYXBwbGljYXRpb24tbGF5ZXIgYXR0YWNrcyB0aGF0IGNhbiBjb21wcm9taXNlIHRoZSBzZWN1cml0eSBvZiB0aGUgc3lzdGVtIG9yIHBsYWNlIHVubmVjZXNzYXJ5IGxvYWQgb24gdGhlbS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkV2FybmluZyhcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUNGUjMnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0NmcjMobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtQ0ZSMyc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbnMgZG9lcyBub3QgaGF2ZSBhY2Nlc3MgbG9nZ2luZyBlbmFibGVkLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdFbmFibGluZyBhY2Nlc3MgbG9ncyBoZWxwcyBvcGVyYXRvcnMgdHJhY2sgYWxsIHZpZXdlciByZXF1ZXN0cyBmb3IgdGhlIGNvbnRlbnQgZGVsaXZlcmVkIHRocm91Z2ggdGhlIENvbnRlbnQgRGVsaXZlcnkgTmV0d29yay4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1DRlI1JykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNDZnI1KG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUNGUjUnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgQ2xvdWRGcm9udCBkaXN0cmlidXRpb25zIHVzZXMgU1NMdjMgb3IgVExTdjEgZm9yIGNvbW11bmljYXRpb24gdG8gdGhlIG9yaWdpbi4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnVnVsbmVyYWJpbGl0aWVzIGhhdmUgYmVlbiBhbmQgY29udGludWUgdG8gYmUgZGlzY292ZXJlZCBpbiB0aGUgZGVwcmVjYXRlZCBTU0wgYW5kIFRMUyBwcm90b2NvbHMuIFVzaW5nIGEgc2VjdXJpdHkgcG9saWN5IHdpdGggbWluaW11bSBUTFN2MS4xIG9yIFRMU3YxLjIgYW5kIGFwcHJvcHJpYXRlIHNlY3VyaXR5IGNpcGhlcnMgZm9yIEhUVFBTIGhlbHBzIHByb3RlY3Qgdmlld2VyIGNvbm5lY3Rpb25zLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUNGUjYnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0NmcjYobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtQ0ZSNic7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbiBkb2VzIG5vdCB1c2UgYW4gb3JpZ2luIGFjY2VzcyBpZGVudGl0eSBhbiBTMyBvcmlnaW4uJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ09yaWdpbiBhY2Nlc3MgaWRlbnRpdGllcyBoZWxwIHdpdGggc2VjdXJpdHkgYnkgcmVzdHJpY3RpbmcgYW55IGRpcmVjdCBhY2Nlc3MgdG8gb2JqZWN0cyB0aHJvdWdoIFMzIFVSTHMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtQVBJRzEnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0FwaWcxKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUFQSUcxJztcbiAgICAgIGNvbnN0IGluZm8gPSAnVGhlIEFQSSBkb2VzIG5vdCBoYXZlIGFjY2VzcyBsb2dnaW5nIGVuYWJsZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0VuYWJsaW5nIGFjY2VzcyBsb2dzIGhlbHBzIG9wZXJhdG9ycyB2aWV3IHdobyBhY2Nlc3NlZCBhbiBBUEkgYW5kIGhvdyB0aGUgY2FsbGVyIGFjY2Vzc2VkIHRoZSBBUEkuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtQVBJRzQnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0FwaWc0KG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUFQSUc0JztcbiAgICAgIGNvbnN0IGluZm8gPSAnVGhlIEFQSSBkb2VzIG5vdCBpbXBsZW1lbnQgYXV0aG9yaXphdGlvbi4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnSW4gbW9zdCBjYXNlcyBhbiBBUEkgbmVlZHMgdG8gaGF2ZSBhbiBhdXRoZW50aWNhdGlvbiBhbmQgYXV0aG9yaXphdGlvbiBpbXBsZW1lbnRhdGlvbiBzdHJhdGVneS4gVGhpcyBpbmNsdWRlcyB1c2luZyBzdWNoIGFwcHJvYWNoZXMgYXMgSUFNLCBDb2duaXRvIFVzZXIgUG9vbHMsIEN1c3RvbSBhdXRob3JpemVyLCBldGMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtQVBJRzYnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0FwaWc2KG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUFQSUc2JztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIFJFU1QgQVBJIFN0YWdlIGRvZXMgbm90IGhhdmUgQ2xvdWRXYXRjaCBsb2dnaW5nIGVuYWJsZWQgZm9yIGFsbCBtZXRob2RzLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdFbmFibGluZyBDbG91ZFdhdGNoIGxvZ3MgYXQgdGhlIHN0YWdlIGxldmVsIGhlbHBzIG9wZXJhdG9ycyB0byB0cmFjayBhbmQgYW5hbHl6ZSBleGVjdXRpb24gYmVoYXZpb3IgYXQgdGhlIEFQSSBzdGFnZSBsZXZlbC4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIE1hbmFnZW1lbnQgYW5kIEdvdmVybmFuY2UgU2VydmljZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIElDb25zdHJ1Y3QgdG8gZXZhbHVhdGVcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tNYW5hZ2VtZW50R292ZXJuYW5jZShub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KTogdm9pZCB7XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUFTMScpICYmXG4gICAgICAhYXdzU29sdXRpb25zQXMxKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUFTMSc7XG4gICAgICBjb25zdCBpbmZvID0gJ1RoZSBBdXRvIFNjYWxpbmcgR3JvdXAgZG9lcyBub3QgaGF2ZSBhIGNvb2xkb3duIHBlcmlvZC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQSBjb29sZG93biBwZXJpb2QgdGVtcG9yYXJpbHkgc3VzcGVuZHMgYW55IHNjYWxpbmcgYWN0aXZpdGllcyBpbiBvcmRlciB0byBhbGxvdyB0aGUgbmV3bHkgbGF1bmNoZWQgRUMyIGluc3RhbmNlKHMpIHNvbWUgdGltZSB0byBzdGFydCBoYW5kbGluZyB0aGUgYXBwbGljYXRpb24gdHJhZmZpYy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1BUzInKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0FzMihub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1BUzInO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgQXV0byBTY2FsaW5nIEdyb3VwIGRvZXMgbm90IGhhdmUgcHJvcGVybHkgY29uZmlndXJlZCBoZWFsdGggY2hlY2tzLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdUaGUgaGVhbHRoIGNoZWNrIGZlYXR1cmUgZW5hYmxlcyB0aGUgc2VydmljZSB0byBkZXRlY3Qgd2hldGhlciBpdHMgcmVnaXN0ZXJlZCBFQzIgaW5zdGFuY2VzIGFyZSBoZWFsdGh5IG9yIG5vdC4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1BUzMnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0FzMyhub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1BUzMnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgQXV0byBTY2FsaW5nIEdyb3VwIGRvZXMgbm90IGhhdmUgbm90aWZpY2F0aW9ucyBjb25maWd1cmVkIGZvciBhbGwgc2NhbGluZyBldmVudHMuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ05vdGlmaWNhdGlvbnMgb24gRUMyIGluc3RhbmNlIGxhdW5jaCwgbGF1bmNoIGVycm9yLCB0ZXJtaW5hdGlvbiwgYW5kIHRlcm1pbmF0aW9uIGVycm9ycyBhbGxvdyBvcGVyYXRvcnMgdG8gZ2FpbiBiZXR0ZXIgaW5zaWdodHMgaW50byBzeXN0ZW1zIGF0dHJpYnV0ZXMgc3VjaCBhcyBhY3Rpdml0eSBhbmQgaGVhbHRoLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgTWFjaGluZSBMZWFybmluZyBTZXJ2aWNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgSUNvbnN0cnVjdCB0byBldmFsdWF0ZVxuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja01hY2hpbmVMZWFybmluZyhub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KTogdm9pZCB7XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLVNNMScpICYmXG4gICAgICAhYXdzU29sdXRpb25zU20xKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLVNNMSc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBTYWdlTWFrZXIgbm90ZWJvb2sgaW5zdGFuY2UgaXMgbm90IHByb3Zpc2lvbmVkIGluc2lkZSBhIFZQQy4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnUHJvdmlzaW9uaW5nIHRoZSBub3RlYm9vayBpbnN0YW5jZXMgaW5zaWRlIGEgVlBDIGVuYWJsZXMgdGhlIG5vdGVib29rIHRvIGFjY2VzcyBWUEMtb25seSByZXNvdXJjZXMgc3VjaCBhcyBFRlMgZmlsZSBzeXN0ZW1zLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLVNNMicpICYmXG4gICAgICAhYXdzU29sdXRpb25zU20yKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLVNNMic7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBTYWdlTWFrZXIgbm90ZWJvb2sgaW5zdGFuY2UgZG9lcyBub3QgaGF2ZSBhbiBlbmNyeXB0ZWQgc3RvcmFnZSB2b2x1bWUuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0VuY3J5cHRpbmcgc3RvcmFnZSB2b2x1bWVzIGhlbHBzIHByb3RlY3QgU2FnZU1ha2VyIGRhdGEtYXQtcmVzdC4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1TTTMnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc1NtMyhub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1TTTMnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgU2FnZU1ha2VyIG5vdGVib29rIGluc3RhbmNlIGhhcyBkaXJlY3QgaW50ZXJuZXQgYWNjZXNzIGVuYWJsZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0Rpc2FibGluZyBwdWJsaWMgYWNjZXNzaWJpbGl0eSBoZWxwcyBtaW5pbWl6ZSBzZWN1cml0eSByaXNrcy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIEFuYWx5dGljcyBTZXJ2aWNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgSUNvbnN0cnVjdCB0byBldmFsdWF0ZVxuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0FuYWx5dGljcyhub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KTogdm9pZCB7XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUFUSDEnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0F0aDEobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtQVRIMSc7XG4gICAgICBjb25zdCBpbmZvID0gJ1RoZSBBdGhlbmEgd29ya2dyb3VwIGRvZXMgbm90IGVuY3J5cHQgcXVlcnkgcmVzdWx0cy4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnRW5jcnlwdGluZyBxdWVyeSByZXN1bHRzIHN0b3JlZCBpbiBTMyBoZWxwcyBzZWN1cmUgZGF0YSB0byBtZWV0IGNvbXBsaWFuY2UgcmVxdWlyZW1lbnRzIGZvciBkYXRhLWF0LXJlc3QgZW5jcnlwdGlvbi4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1FTVIyJykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNFbXIyKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUVNUjInO1xuICAgICAgY29uc3QgaW5mbyA9ICdUaGUgRU1SIGNsdXN0ZXIgZG9lcyBub3QgaGF2ZSBTMyBsb2dnaW5nIGVuYWJsZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1VwbG9hZGluZyBsb2dzIHRvIFMzIGVuYWJsZXMgdGhlIHN5c3RlbSB0byBrZWVwIHRoZSBsb2dnaW5nIGRhdGEgZm9yIGhpc3RvcmljYWwgcHVycG9zZXMgb3IgdG8gdHJhY2sgYW5kIGFuYWx5emUgdGhlIGNsdXN0ZXJzIGJlaGF2aW9yLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUVNUjYnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0VtcjYobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtRU1SNic7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBFTVIgY2x1c3RlciBkb2VzIG5vdCBpbXBsZW1lbnQgYXV0aGVudGljYXRpb24gdmlhIGFuIEVDMiBLZXkgUGFpciBvciBLZXJiZXJvcy4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnU1NIIGNsaWVudHMgY2FuIHVzZSBhbiBFQzIga2V5IHBhaXIgdG8gYXV0aGVudGljYXRlIHRvIGNsdXN0ZXIgaW5zdGFuY2VzLiBBbHRlcm5hdGl2ZWx5LCB3aXRoIEVNUiByZWxlYXNlIHZlcnNpb24gNS4xMC4wIG9yIGxhdGVyLCBzb2x1dGlvbnMgY2FuIGNvbmZpZ3VyZSBLZXJiZXJvcyB0byBhdXRoZW50aWNhdGUgdXNlcnMgYW5kIFNTSCBjb25uZWN0aW9ucyB0byB0aGUgbWFzdGVyIG5vZGUuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtRVNIMScpICYmXG4gICAgICAhYXdzU29sdXRpb25zRXNoMShub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1FU0gxJztcbiAgICAgIGNvbnN0IGluZm8gPSAnVGhlIEVTIGRvbWFpbiBpcyBub3QgcHJvdmlzaW9uZWQgaW5zaWRlIGEgVlBDLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdQcm92aXNpb25pbmcgdGhlIGRvbWFpbiB3aXRoaW4gYSBWUEMgZW5hYmxlcyBiZXR0ZXIgZmxleGliaWxpdHkgYW5kIGNvbnRyb2wgb3ZlciB0aGUgY2x1c3RlcnMgYWNjZXNzIGFuZCBzZWN1cml0eSBhcyB0aGlzIGZlYXR1cmUga2VlcHMgYWxsIHRyYWZmaWMgYmV0d2VlbiB0aGUgVlBDIGFuZCBFbGFzdGljc2VhcmNoIGRvbWFpbnMgd2l0aGluIHRoZSBBV1MgbmV0d29yayBpbnN0ZWFkIG9mIGdvaW5nIG92ZXIgdGhlIHB1YmxpYyBJbnRlcm5ldC4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1FU0gyJykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNFc2gyKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUVTSDInO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgRVMgZG9tYWluIGRvZXMgbm90IGhhdmUgbm9kZS10by1ub2RlIGVuY3J5cHRpb24gZW5hYmxlZC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnRW5hYmxpbmcgdGhlIG5vZGUtdG8tbm9kZSBlbmNyeXB0aW9uIGZlYXR1cmUgYWRkcyBhbiBleHRyYSBsYXllciBvZiBkYXRhIHByb3RlY3Rpb24gb24gdG9wIG9mIHRoZSBleGlzdGluZyBFUyBzZWN1cml0eSBmZWF0dXJlcyBzdWNoIGFzIEhUVFBTIGNsaWVudCB0byBjbHVzdGVyIGVuY3J5cHRpb24gYW5kIGRhdGEtYXQtcmVzdCBlbmNyeXB0aW9uLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUVTSDMnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0VzaDMobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtRVNIMyc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBFUyBkb21haW4gZG9lcyBub3Qgb25seSBncmFudCBhY2Nlc3MgdmlhIGFsbG93bGlzdGVkIElQIGFkZHJlc3Nlcy4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnVXNpbmcgYWxsb3dsaXN0ZWQgSVAgYWRkcmVzc2VzIGhlbHBzIHByb3RlY3QgdGhlIGRvbWFpbiBhZ2FpbnN0IHVuYXV0aG9yaXplZCBhY2Nlc3MuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtRVNINCcpICYmXG4gICAgICAhYXdzU29sdXRpb25zRXNoNChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1FU0g0JztcbiAgICAgIGNvbnN0IGluZm8gPSAnVGhlIEVTIGRvbWFpbiBkb2VzIG5vdCB1c2UgZGVkaWNhdGVkIG1hc3RlciBub2Rlcy4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnVXNpbmcgZGVkaWNhdGVkIG1hc3RlciBub2RlcyBoZWxwcyBpbXByb3ZlIGVudmlyb25tZW50YWwgc3RhYmlsaXR5IGJ5IG9mZmxvYWRpbmcgYWxsIHRoZSBtYW5hZ2VtZW50IHRhc2tzIGZyb20gdGhlIGRhdGEgbm9kZXMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtRVNINScpICYmXG4gICAgICAhYXdzU29sdXRpb25zRXNoNShub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1FU0g1JztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIEVTIGRvbWFpbiBkb2VzIG5vdCBhbGxvdyBmb3IgdW5zaWduZWQgcmVxdWVzdHMgb3IgYW5vbnltb3VzIGFjY2Vzcy4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnUmVzdHJpY3RpbmcgcHVibGljIGFjY2VzcyBoZWxwcyBwcmV2ZW50IHVuYXV0aG9yaXplZCBhY2Nlc3MgYW5kIHByZXZlbnRzIGFueSB1bnNpZ25lZCByZXF1ZXN0cyB0byBiZSBtYWRlIHRvIHRoZSByZXNvdXJjZXMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtRVNINycpICYmXG4gICAgICAhYXdzU29sdXRpb25zRXNoNyhub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1FU0g3JztcbiAgICAgIGNvbnN0IGluZm8gPSAnVGhlIEVTIGRvbWFpbiBkb2VzIG5vdCBoYXZlIFpvbmUgQXdhcmVuZXNzIGVuYWJsZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0VuYWJsaW5nIGNyb3NzLXpvbmUgcmVwbGljYXRpb24gKFpvbmUgQXdhcmVuZXNzKSBpbmNyZWFzZXMgdGhlIGF2YWlsYWJpbGl0eSBvZiB0aGUgRVMgZG9tYWluIGJ5IGFsbG9jYXRpbmcgdGhlIG5vZGVzIGFuZCByZXBsaWNhdGUgdGhlIGRhdGEgYWNyb3NzIHR3byBBWnMgaW4gdGhlIHNhbWUgcmVnaW9uIGluIG9yZGVyIHRvIHByZXZlbnQgZGF0YSBsb3NzIGFuZCBtaW5pbWl6ZSBkb3dudGltZSBpbiB0aGUgZXZlbnQgb2Ygbm9kZSBvciBBWiBmYWlsdXJlLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUVTSDgnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0VzaDgobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtRVNIOCc7XG4gICAgICBjb25zdCBpbmZvID0gJ1RoZSBFUyBkb21haW4gZG9lcyBub3QgaGF2ZSBlbmNyeXB0aW9uIGF0IHJlc3QgZW5hYmxlZC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnRW5jcnlwdGluZyBkYXRhLWF0LXJlc3QgcHJvdGVjdHMgZGF0YSBjb25maWRlbnRpYWxpdHkgYW5kIHByZXZlbnRzIHVuYXV0aG9yaXplZCB1c2VycyBmcm9tIGFjY2Vzc2luZyBzZW5zaXRpdmUgaW5mb3JtYXRpb24uJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtRVNIOScpICYmXG4gICAgICAhYXdzU29sdXRpb25zRXNoOShub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1FU0g5JztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIEVTIGRvbWFpbiBkb2VzIG5vdCBtaW5pbWFsbHkgcHVibGlzaCBTRUFSQ0hfU0xPV19MT0dTIGFuZCBJTkRFWF9TTE9XX0xPR1MgdG8gQ2xvdWRXYXRjaCBMb2dzLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdUaGVzZSBsb2dzIGVuYWJsZSBvcGVyYXRvcnMgdG8gZ2FpbiBmdWxsIGluc2lnaHQgaW50byB0aGUgcGVyZm9ybWFuY2Ugb2YgdGhlc2Ugb3BlcmF0aW9ucy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1LREEzJykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNLZGEzKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUtEQTMnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgS2luZXNpcyBEYXRhIEFuYWx5dGljcyBGbGluayBBcHBsaWNhdGlvbiBkb2VzIG5vdCBoYXZlIGNoZWNrcG9pbnRpbmcgZW5hYmxlZC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQ2hlY2twb2ludHMgYXJlIGJhY2t1cHMgb2YgYXBwbGljYXRpb24gc3RhdGUgdGhhdCBLREEgYXV0b21hdGljYWxseSBjcmVhdGVzIHBlcmlvZGljYWxseSBhbmQgdXNlcyB0byByZXN0b3JlIGZyb20gZmF1bHRzLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRXYXJuaW5nKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtS0RTMScpICYmXG4gICAgICAhYXdzU29sdXRpb25zS2RzMShub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1LRFMxJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIEtpbmVzaXMgRGF0YSBTdHJlYW0gZG9lcyBub3QgaGFzIHNlcnZlci1zaWRlIGVuY3J5cHRpb24gZW5hYmxlZC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICBcIkRhdGEgaXMgZW5jcnlwdGVkIGJlZm9yZSBpdCdzIHdyaXR0ZW4gdG8gdGhlIEtpbmVzaXMgc3RyZWFtIHN0b3JhZ2UgbGF5ZXIsIGFuZCBkZWNyeXB0ZWQgYWZ0ZXIgaXTigJlzIHJldHJpZXZlZCBmcm9tIHN0b3JhZ2UuIFRoaXMgYWxsb3dzIHRoZSBzeXN0ZW0gdG8gbWVldCBzdHJpY3QgcmVndWxhdG9yeSByZXF1aXJlbWVudHMgYW5kIGVuaGFuY2UgdGhlIHNlY3VyaXR5IG9mIHN5c3RlbSBkYXRhLlwiO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1LREYxJykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNLZGYxKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUtERjEnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgS2luZXNpcyBEYXRhIEZpcmVob3NlIGRlbGl2ZXJ5IHN0cmVhbSBkb2VzIGhhdmUgc2VydmVyLXNpZGUgZW5jcnlwdGlvbiBlbmFibGVkLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdUaGlzIGFsbG93cyB0aGUgc3lzdGVtIHRvIG1lZXQgc3RyaWN0IHJlZ3VsYXRvcnkgcmVxdWlyZW1lbnRzIGFuZCBlbmhhbmNlIHRoZSBzZWN1cml0eSBvZiBzeXN0ZW0gZGF0YS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1NU0syJykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNNc2syKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLU1TSzInO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgTVNLIGNsdXN0ZXIgdXNlcyBwbGFpbnRleHQgY29tbXVuaWNhdGlvbiBiZXR3ZWVuIGNsaWVudHMgYW5kIGJyb2tlcnMuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1RMUyBvbmx5IGNvbW11bmljYXRpb24gc2VjdXJlcyBkYXRhLWluLXRyYW5zaXQgYnkgZW5jcnlwdGluZyB0aGUgY29ubmVjdGlvbiBiZXR3ZWVuIHRoZSBjbGllbnRzIGFuZCBicm9rZXJzLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLU1TSzMnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc01zazMobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtTVNLMyc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBNU0sgY2x1c3RlciB1c2VzIHBsYWludGV4dCBjb21tdW5pY2F0aW9uIGJldHdlZW4gYnJva2Vycy4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnVExTIGNvbW11bmljYXRpb24gc2VjdXJlcyBkYXRhLWluLXRyYW5zaXQgYnkgZW5jcnlwdGluZyB0aGUgY29ubmVjdGlvbiBiZXR3ZWVuIGJyb2tlcnMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtTVNLNicpICYmXG4gICAgICAhYXdzU29sdXRpb25zTXNrNihub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1NU0s2JztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIE1TSyBjbHVzdGVyIGRvZXMgbm90IHNlbmQgYnJva2VyIGxvZ3MgdG8gYSBzdXBwb3J0ZWQgZGVzdGluYXRpb24uJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0Jyb2tlciBsb2dzIGVuYWJsZSBvcGVyYXRvcnMgdG8gdHJvdWJsZXNob290IEFwYWNoZSBLYWZrYSBhcHBsaWNhdGlvbnMgYW5kIHRvIGFuYWx5emUgdGhlaXIgY29tbXVuaWNhdGlvbnMgd2l0aCB0aGUgTVNLIGNsdXN0ZXIuIFRoZSBjbHVzdGVyIGNhbiBkZWxpdmVyIGxvZ3MgdG8gdGhlIGZvbGxvd2luZyByZXNvdXJjZXM6IGEgQ2xvdWRXYXRjaCBsb2cgZ3JvdXAsIGFuIFMzIGJ1Y2tldCwgYSBLaW5lc2lzIERhdGEgRmlyZWhvc2UgZGVsaXZlcnkgc3RyZWFtLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLVFTMScpICYmXG4gICAgICAhYXdzU29sdXRpb25zUXMxKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLVFTMSc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBRdWlja3NpZ2h0IGRhdGEgc291cmNlcyBjb25uZWN0aW9uIGlzIG5vdCBjb25maWd1cmVkIHRvIHVzZSBTU0wuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1NTTCBzZWN1cmVzIGNvbW11bmljYXRpb25zIHRvIGRhdGEgc291cmNlcywgZXNwZWNpYWxseSB3aGVuIHVzaW5nIHB1YmxpYyBuZXR3b3Jrcy4gVXNpbmcgU1NMIHdpdGggUXVpY2tTaWdodCByZXF1aXJlcyB0aGUgdXNlIG9mIGNlcnRpZmljYXRlcyBzaWduZWQgYnkgYSBwdWJsaWNseS1yZWNvZ25pemVkIGNlcnRpZmljYXRlIGF1dGhvcml0eS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIFNlY3VyaXR5IGFuZCBDb21wbGlhbmNlIFNlcnZpY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBJQ29uc3RydWN0IHRvIGV2YWx1YXRlXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrU2VjdXJpdHlDb21wbGlhbmNlKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpOiB2b2lkIHtcbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtSUFNNCcpICYmXG4gICAgICAhYXdzU29sdXRpb25zSWFtNChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1JQU00JztcbiAgICAgIGNvbnN0IGluZm8gPSAnVGhlIElBTSB1c2VyLCByb2xlLCBvciBncm91cCB1c2VzIEFXUyBtYW5hZ2VkIHBvbGljaWVzLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdBbiBBV1PCoG1hbmFnZWQgcG9saWN5wqBpcyBhIHN0YW5kYWxvbmXCoHBvbGljecKgdGhhdCBpcyBjcmVhdGVkIGFuZCBhZG1pbmlzdGVyZWQgYnkgQVdTLiBDdXJyZW50bHksIG1hbnkgQVdTIG1hbmFnZWQgcG9saWNpZXMgZG8gbm90IHJlc3RyaWN0IHJlc291cmNlIHNjb3BlLiBSZXBsYWNlIEFXUyBtYW5hZ2VkIHBvbGljaWVzIHdpdGggc3lzdGVtIHNwZWNpZmljIChjdXN0b21lcikgbWFuYWdlZCBwb2xpY2llcy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1JQU01JykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNJYW01KG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUlBTTUnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgSUFNIGVudGl0eSBjb250YWlucyB3aWxkY2FyZCBwZXJtaXNzaW9ucyBhbmQgZG9lcyBub3QgaGF2ZSBhIGNka19uYWcgcnVsZSBzdXBwcmVzc2lvbiB3aXRoIGV2aWRlbmNlIGZvciB0aG9zZSBwZXJtaXNzaW9uLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdNZXRhZGF0YSBleHBsYWluaW5nIHRoZSBldmlkZW5jZSAoZS5nLiB2aWEgc3VwcG9ydGluZyBsaW5rcykgZm9yIHdpbGRjYXJkIHBlcm1pc3Npb25zIGFsbG93cyBmb3IgdHJhbnNwYXJlbmN5IHRvIG9wZXJhdG9ycy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1DT0cxJykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNDb2cxKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUNPRzEnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgQ29nbml0byB1c2VyIHBvb2wgZG9lcyBub3QgaGF2ZSBhIHBhc3N3b3JkIHBvbGljeSB0aGF0IG1pbmltYWxseSBzcGVjaWZ5IGEgcGFzc3dvcmQgbGVuZ3RoIG9mIGF0IGxlYXN0IDggY2hhcmFjdGVycywgYXMgd2VsbCBhcyByZXF1aXJpbmcgdXBwZXJjYXNlLCBudW1lcmljLCBhbmQgc3BlY2lhbCBjaGFyYWN0ZXJzLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdTdHJvbmcgcGFzc3dvcmQgcG9saWNpZXMgaW5jcmVhc2Ugc3lzdGVtIHNlY3VyaXR5IGJ5IGVuY291cmFnaW5nIHVzZXJzIHRvIGNyZWF0ZSByZWxpYWJsZSBhbmQgc2VjdXJlIHBhc3N3b3Jkcy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1DT0cyJykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNDb2cyKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUNPRzInO1xuICAgICAgY29uc3QgaW5mbyA9ICdUaGUgQ29nbml0byB1c2VyIHBvb2wgZG9lcyBub3QgcmVxdWlyZSBNRkEuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ011bHRpLWZhY3RvciBhdXRoZW50aWNhdGlvbiAoTUZBKSBpbmNyZWFzZXMgc2VjdXJpdHkgZm9yIHRoZSBhcHBsaWNhdGlvbiBieSBhZGRpbmcgYW5vdGhlciBhdXRoZW50aWNhdGlvbiBtZXRob2QsIGFuZCBub3QgcmVseWluZyBzb2xlbHkgb24gdXNlciBuYW1lIGFuZCBwYXNzd29yZC4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkV2FybmluZyhcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUNPRzMnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0NvZzMobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtQ09HMyc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBDb2duaXRvIHVzZXIgcG9vbCBkb2VzIG5vdCBoYXZlIEFkdmFuY2VkU2VjdXJpdHlNb2RlIHNldCB0byBFTkZPUkNFRC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQWR2YW5jZWQgc2VjdXJpdHkgZmVhdHVyZXMgZW5hYmxlIHRoZSBzeXN0ZW0gdG8gZGV0ZWN0IGFuZCBhY3QgdXBvbiBtYWxpY2lvdXMgc2lnbi1pbiBhdHRlbXB0cy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1DT0c3JykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNDb2c3KG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUNPRzcnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgQ29nbml0byBpZGVudGl0eSBwb29sIGFsbG93cyBmb3IgdW5hdXRoZW50aWNhdGVkIGxvZ2lucyBhbmQgZG9lcyBub3QgaGF2ZSBhIGNka19uYWcgcnVsZSBzdXBwcmVzc2lvbiB3aXRoIGEgcmVhc29uLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdJbiBtYW55IGNhc2VzIGFwcGxpY2F0aW9ucyBkbyBub3Qgd2FycmFudCB1bmF1dGhlbnRpY2F0ZWQgZ3Vlc3QgYWNjZXNzIGFwcGxpY2F0aW9ucy4gTWV0YWRhdGEgZXhwbGFpbmluZyB0aGUgdXNlIGNhc2UgYWxsb3dzIGZvciB0cmFuc3BhcmVuY3kgdG8gb3BlcmF0b3JzLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUtNUzUnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0ttczUobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtS01TNSc7XG4gICAgICBjb25zdCBpbmZvID0gJ1RoZSBLTVMgU3ltbWV0cmljIENNSyBkb2VzIG5vdCBoYXZlIEtleSBSb3RhdGlvbiBlbmFibGVkLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdLTVMgS2V5IFJvdGF0aW9uIGFsbG93IGEgc3lzdGVtIHRvIHNldCBhbiB5ZWFybHkgcm90YXRpb24gc2NoZWR1bGUgZm9yIGEgQ01LIHNvIHdoZW4gYSBjdXN0b21lciBtYXN0ZXIga2V5IGlzIHJlcXVpcmVkIHRvIGVuY3J5cHQgbmV3IGRhdGEsIHRoZSBLTVMgc2VydmljZSBjYW4gYXV0b21hdGljYWxseSB1c2UgdGhlIGxhdGVzdCB2ZXJzaW9uIG9mIHRoZSBIU0EgYmFja2luZyBrZXkgdG8gcGVyZm9ybSB0aGUgZW5jcnlwdGlvbi4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIFNlcnZlcmxlc3MgU2VydmljZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIElDb25zdHJ1Y3QgdG8gZXZhbHVhdGVcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tTZXJ2ZXJsZXNzKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpOiB2b2lkIHtcbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtU0YxJykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNTZjEobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtU0YxJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIFN0ZXAgRnVuY3Rpb24gZG9lcyBub3QgbG9nIFwiQUxMXCIgZXZlbnRzIHRvIENsb3VkV2F0Y2ggTG9ncy4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnTG9nZ2luZyBcIkFMTFwiIGV2ZW50cyB0byBDbG91ZFdhdGNoIGxvZ3MgaGVscCBvcGVyYXRvcnMgdHJvdWJsZXNob290IGFuZCBhdWRpdCBzeXN0ZW1zLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLVNGMicpICYmXG4gICAgICAhYXdzU29sdXRpb25zU2YyKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLVNGMic7XG4gICAgICBjb25zdCBpbmZvID0gJ1RoZSBTdGVwIEZ1bmN0aW9uIGRvZXMgbm90IGhhdmUgWC1SYXkgdHJhY2luZyBlbmFibGVkLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdYLXJheSBwcm92aWRlcyBhbiBlbmQtdG8tZW5kIHZpZXcgb2YgaG93IGFuIGFwcGxpY2F0aW9uIGlzIHBlcmZvcm1pbmcuIFRoaXMgaGVscHMgb3BlcmF0b3JzIHRvIGRpc2NvdmVyIHBlcmZvcm1hbmNlIGlzc3VlcywgZGV0ZWN0IHBlcm1pc3Npb24gcHJvYmxlbXMsIGFuZCB0cmFjayByZXF1ZXN0cyBtYWRlIHRvIGFuZCBmcm9tIG90aGVyIEFXUyBzZXJ2aWNlcy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIEFwcGxpY2F0aW9uIEludGVncmF0aW9uIFNlcnZpY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBJQ29uc3RydWN0IHRvIGV2YWx1YXRlXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrQXBwbGljYXRpb25JbnRlZ3JhdGlvbihub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KTogdm9pZCB7XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLVNOUzInKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc1NuczIobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtU05TMic7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBTTlMgVG9waWMgZG9lcyBub3QgaGF2ZSBzZXJ2ZXItc2lkZSBlbmNyeXB0aW9uIGVuYWJsZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1NlcnZlciBzaWRlIGVuY3J5cHRpb24gYWRkcyBhZGRpdGlvbmFsIHByb3RlY3Rpb24gb2Ygc2Vuc2l0aXZlIGRhdGEgZGVsaXZlcmVkIGFzIG1lc3NhZ2VzIHRvIHN1YnNjcmliZXJzLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLVNRUzInKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc1NxczIobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtU1FTMic7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBTUVMgUXVldWUgZG9lcyBub3QgaGF2ZSBzZXJ2ZXItc2lkZSBlbmNyeXB0aW9uIGVuYWJsZWQuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1NlcnZlciBzaWRlIGVuY3J5cHRpb24gYWRkcyBhZGRpdGlvbmFsIHByb3RlY3Rpb24gb2Ygc2Vuc2l0aXZlIGRhdGEgZGVsaXZlcmVkIGFzIG1lc3NhZ2VzIHRvIHN1YnNjcmliZXJzLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLVNRUzMnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc1NxczMobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtU1FTMyc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBTUVMgcXVldWUgZG9lcyBub3QgaGF2ZSBhIGRlYWQtbGV0dGVyIHF1ZXVlIChETFEpIGVuYWJsZWQgb3IgaGF2ZSBhIGNka19uYWcgcnVsZSBzdXBwcmVzc2lvbiBpbmRpY2F0aW5nIGl0IGlzIGEgRExRLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdVc2luZyBhIERMUSBoZWxwcyBtYWludGFpbiB0aGUgcXVldWUgZmxvdyBhbmQgYXZvaWQgbG9zaW5nIGRhdGEgYnkgZGV0ZWN0aW5nIGFuZCBtaXRpZ2F0aW5nIGZhaWx1cmVzIGFuZCBzZXJ2aWNlIGRpc3J1cHRpb25zIG9uIHRpbWUuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBNZWRpYSBTZXJ2aWNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgSUNvbnN0cnVjdCB0byBldmFsdWF0ZVxuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja01lZGlhU2VydmljZXMobm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSk6IHZvaWQge1xuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1NUzEnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc01zMShub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1NUzEnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgTWVkaWFTdG9yZSBjb250YWluZXIgZG9lcyBub3QgaGF2ZSBjb250YWluZXIgYWNjZXNzIGxvZ2dpbmcgZW5hYmxlZC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnVGhlIGNvbnRhaW5lciBzaG91bGQgaGF2ZSBhY2Nlc3MgbG9nZ2luZyBlbmFibGVkIHRvIHByb3ZpZGUgZGV0YWlsZWQgcmVjb3JkcyBmb3IgdGhlIHJlcXVlc3RzIHRoYXQgYXJlIG1hZGUgdG8gdGhlIGNvbnRhaW5lci4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1NUzQnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc01zNChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1NUzQnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgTWVkaWFTdG9yZSBjb250YWluZXIgZG9lcyBub3QgZGVmaW5lIGEgbWV0cmljIHBvbGljeSB0byBzZW5kIG1ldHJpY3MgdG8gQ2xvdWRXYXRjaC4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnVXNpbmcgYSBjb21iaW5hdGlvbiBvZiBNZWRpYVN0b3JlIG1ldHJpY3MgYW5kIENsb3VkV2F0Y2ggYWxhcm1zIGhlbHBzIG9wZXJhdG9ycyBnYWluIGJldHRlciBpbnNpZ2h0cyBpbnRvIGNvbnRhaW5lciBvcGVyYXRpb25zLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRXYXJuaW5nKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtTVM3JykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNNczcobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtTVM3JztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIE1lZGlhU3RvcmUgY29udGFpbmVyIGRvZXMgbm90IGRlZmluZSBhIGNvbnRhaW5lciBwb2xpY3kuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1VzaW5nIGEgY29udGFpbmVyIHBvbGljeSBoZWxwcyBmb2xsb3cgdGhlIHN0YW5kYXJkIHNlY3VyaXR5IGFkdmljZSBvZiBncmFudGluZyBsZWFzdCBwcml2aWxlZ2UsIG9yIGdyYW50aW5nIG9ubHkgdGhlIHBlcm1pc3Npb25zIHJlcXVpcmVkIHRvIGFsbG93IG5lZWRlZCBhY2Nlc3MgdG8gdGhlIGNvbnRhaW5lci4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkV2FybmluZyhcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLU1TOCcpICYmXG4gICAgICAhYXdzU29sdXRpb25zTXM4KG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLU1TOCc7XG4gICAgICBjb25zdCBpbmZvID0gJ1RoZSBNZWRpYVN0b3JlIGNvbnRhaW5lciBkb2VzIG5vdCBkZWZpbmUgYSBDT1JTIHBvbGljeS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnVXNpbmcgYSBDT1JTIHBvbGljeSBoZWxwcyBmb2xsb3cgdGhlIHN0YW5kYXJkIHNlY3VyaXR5IGFkdmljZSBvZiBncmFudGluZyBsZWFzdCBwcml2aWxlZ2UsIG9yIGdyYW50aW5nIG9ubHkgdGhlIHBlcm1pc3Npb25zIHJlcXVpcmVkIHRvIGFsbG93IG5lZWRlZCBhY2Nlc3MgdG8gdGhlIGNvbnRhaW5lci4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkV2FybmluZyhcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLU1TMTAnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc01zMTAobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtTVMxMCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBNZWRpYVN0b3JlIGNvbnRhaW5lciBkb2VzIG5vdCBkZWZpbmUgYSBsaWZlY3ljbGUgcG9saWN5Lic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdNYW55IHVzZSBjYXNlcyB3YXJyYW50IHRoZSB1c2FnZSBvZiBsaWZlY3ljbGUgY29uZmlndXJhdGlvbnMgdG8gbWFuYWdlIGNvbnRhaW5lciBvYmplY3RzIGR1cmluZyB0aGVpciBsaWZldGltZS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkV2FybmluZyhcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgRGV2ZWxvcGVyIFRvb2xzIFNlcnZpY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBJQ29uc3RydWN0IHRvIGV2YWx1YXRlXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrRGV2ZWxvcGVyVG9vbHMobm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSk6IHZvaWQge1xuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1DQjMnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0NiMyhub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1DQjMnO1xuICAgICAgY29uc3QgaW5mbyA9ICdUaGUgQ29kZUJ1aWxkIHByb2plY3QgaGFzIHByaXZpbGVnZWQgbW9kZSBlbmFibGVkLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdQcml2aWxlZ2VkIGdyYW50cyBlbGV2YXRlZCByaWdodHMgdG8gdGhlIHN5c3RlbSwgd2hpY2ggaW50cm9kdWNlcyBhZGRpdGlvbmFsIHJpc2suIFByaXZpbGVnZWQgbW9kZSBzaG91bGQgb25seSBiZSBzZXQgdG8gdHJ1ZSBvbmx5IGlmIHRoZSBidWlsZCBwcm9qZWN0IGlzIHVzZWQgdG8gYnVpbGQgRG9ja2VyIGltYWdlcy4gT3RoZXJ3aXNlLCBhIGJ1aWxkIHRoYXQgYXR0ZW1wdHMgdG8gaW50ZXJhY3Qgd2l0aCB0aGUgRG9ja2VyIGRhZW1vbiBmYWlscy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkV2FybmluZyhcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnQXdzU29sdXRpb25zLUNCNCcpICYmXG4gICAgICAhYXdzU29sdXRpb25zQ2I0KG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnQXdzU29sdXRpb25zLUNCNCc7XG4gICAgICBjb25zdCBpbmZvID0gJ1RoZSBDb2RlQnVpbGQgcHJvamVjdCBkb2VzIG5vdCB1c2UgYSBDTUsgZm9yIGVuY3J5cHRpb24uJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1VzaW5nIGEgQ01LIGhlbHBzIGZvbGxvdyB0aGUgc3RhbmRhcmQgc2VjdXJpdHkgYWR2aWNlIG9mIGdyYW50aW5nIGxlYXN0IHByaXZpbGVnZSB0byBvYmplY3RzIGdlbmVyYXRlZCBieSB0aGUgcHJvamVjdC4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0F3c1NvbHV0aW9ucy1DQjUnKSAmJlxuICAgICAgIWF3c1NvbHV0aW9uc0NiNShub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0F3c1NvbHV0aW9ucy1DQjUnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgQ29kZWJ1aWxkIHByb2plY3QgZG9lcyBub3QgdXNlIGltYWdlcyBwcm92aWRlZCBieSB0aGUgQ29kZUJ1aWxkIHNlcnZpY2Ugb3IgaGF2ZSBhIGNka19uYWcgc3VwcHJlc3Npb24gcnVsZSBleHBsYWluaW5nIHRoZSBuZWVkIGZvciBhIGN1c3RvbSBpbWFnZS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnRXhwbGFpbmluZyBkaWZmZXJlbmNlcy9lZGl0cyB0byBEb2NrZXIgaW1hZ2VzIGhlbHBzIG9wZXJhdG9ycyBiZXR0ZXIgdW5kZXJzdGFuZCBzeXN0ZW0gZGVwZW5kZW5jaWVzLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRXYXJuaW5nKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdBd3NTb2x1dGlvbnMtQzkxJykgJiZcbiAgICAgICFhd3NTb2x1dGlvbnNDOTEobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdBd3NTb2x1dGlvbnMtQzkxJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIENsb3VkOSBpbnN0YW5jZSBkb2VzIG5vdCB1c2UgYSBuby1pbmdyZXNzIEVDMiBpbnN0YW5jZSB3aXRoIEFXUyBTeXN0ZW1zIE1hbmFnZXIuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1NTTSBhZGRzIGFuIGFkZGl0aW9uYWwgbGF5ZXIgb2YgcHJvdGVjdGlvbiBhcyBpdCBhbGxvd3Mgb3BlcmF0b3JzIHRvIGNvbnRyb2wgYWNjZXNzIHRocm91Z2ggSUFNIHBlcm1pc3Npb25zIGFuZCBkb2VzIG5vdCByZXF1aXJlIG9wZW5pbmcgaW5ib3VuZCBwb3J0cy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICB9XG59XG4iXX0=