"""
IGRF-13 model parameters.

Notes
-----
This is used in cotrans_lib.cdipdir

"""


def set_igrf_params():
    """
    Set the IGRF-13 parameters.

    Parameters
    ----------
    none

    Returns
    -------
    minyear: int
        Minimum year for parameters, 1960
    maxyear: int
        Maximum validity year, 2025 (2020-2025 with interpolation)
    ga: dict of float
        Dictionary of "g" parameters.
    ha: dict of float
        Dictionary of "h" parameters.
    dg: list of float
        Parameters for 2020-2025 interpolation.
    dh: list of float
        Parameters for 2020-2025 interpolation.

    Notes
    -----
    International Geomagnetic Reference Field (IGRF-13) model
    for time interval 1970 to 2020.
    Used to compute geodipole axis direction, function cdipdir().
    """
    minyear = 1970  # Some functions do not work with negative unix times.
    maxyear = 2025  # max year plus 5 years with interpolation
    ga = {}
    ha = {}
    ga[1965] = [0.0, -30334.0, -2119.0, -1662.0, 2997.0, 1594.0, 1297.0, -2038.0, 1292.0, 856.0, 957.0, 804.0, 479.0, -390.0, 252.0, -219.0, 358.0, 254.0, -31.0, -157.0, -62.0, 45.0, 61.0, 8.0, -228.0, 4.0, 1.0, -111.0, 75.0, -57.0, 4.0, 13.0, -26.0, -6.0, 13.0, 1.0, 13.0, 5.0, -4.0, -14.0, 0.0, 8.0, -1.0, 11.0, 4.0, 8.0, 10.0, 2.0, -13.0, 10.0, -1.0, -1.0, 5.0, 1.0, -2.0, -2.0, -3.0, 2.0, -5.0, -2.0, 4.0, 4.0, 0.0, 2.0, 2.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
    ha[1965] = [0.0, 0.0, 5776.0, 0.0, -2016.0, 114.0, 0.0, -404.0, 240.0, -165.0, 0.0, 148.0, -269.0, 13.0, -269.0, 0.0, 19.0, 128.0, -126.0, -97.0, 81.0, 0.0, -11.0, 100.0, 68.0, -32.0, -8.0, -7.0, 0.0, -61.0, -27.0, -2.0, 6.0, 26.0, -23.0, -12.0, 0.0, 7.0, -12.0, 9.0, -16.0, 4.0, 24.0, -3.0, -17.0, 0.0, -22.0, 15.0, 7.0, -4.0, -5.0, 10.0, 10.0, -4.0, 1.0, 0.0, 2.0, 1.0, 2.0, 6.0, -4.0, 0.0, -2.0, 3.0, 0.0, -6.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
    ga[1970] = [0.0, -30220.0, -2068.0, -1781.0, 3000.0, 1611.0, 1287.0, -2091.0, 1278.0, 838.0, 952.0, 800.0, 461.0, -395.0, 234.0, -216.0, 359.0, 262.0, -42.0, -160.0, -56.0, 43.0, 64.0, 15.0, -212.0, 2.0, 3.0, -112.0, 72.0, -57.0, 1.0, 14.0, -22.0, -2.0, 13.0, -2.0, 14.0, 6.0, -2.0, -13.0, -3.0, 5.0, 0.0, 11.0, 3.0, 8.0, 10.0, 2.0, -12.0, 10.0, -1.0, 0.0, 3.0, 1.0, -1.0, -3.0, -3.0, 2.0, -5.0, -1.0, 6.0, 4.0, 1.0, 0.0, 3.0, -1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
    ha[1970] = [0.0, 0.0, 5737.0, 0.0, -2047.0, 25.0, 0.0, -366.0, 251.0, -196.0, 0.0, 167.0, -266.0, 26.0, -279.0, 0.0, 26.0, 139.0, -139.0, -91.0, 83.0, 0.0, -12.0, 100.0, 72.0, -37.0, -6.0, 1.0, 0.0, -70.0, -27.0, -4.0, 8.0, 23.0, -23.0, -11.0, 0.0, 7.0, -15.0, 6.0, -17.0, 6.0, 21.0, -6.0, -16.0, 0.0, -21.0, 16.0, 6.0, -4.0, -5.0, 10.0, 11.0, -2.0, 1.0, 0.0, 1.0, 1.0, 3.0, 4.0, -4.0, 0.0, -1.0, 3.0, 1.0, -4.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
    ga[1975] = [0.0, -30100.0, -2013.0, -1902.0, 3010.0, 1632.0, 1276.0, -2144.0, 1260.0, 830.0, 946.0, 791.0, 438.0, -405.0, 216.0, -218.0, 356.0, 264.0, -59.0, -159.0, -49.0, 45.0, 66.0, 28.0, -198.0, 1.0, 6.0, -111.0, 71.0, -56.0, 1.0, 16.0, -14.0, 0.0, 12.0, -5.0, 14.0, 6.0, -1.0, -12.0, -8.0, 4.0, 0.0, 10.0, 1.0, 7.0, 10.0, 2.0, -12.0, 10.0, -1.0, -1.0, 4.0, 1.0, -2.0, -3.0, -3.0, 2.0, -5.0, -2.0, 5.0, 4.0, 1.0, 0.0, 3.0, -1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
    ha[1975] = [0.0, 0.0, 5675.0, 0.0, -2067.0, -68.0, 0.0, -333.0, 262.0, -223.0, 0.0, 191.0, -265.0, 39.0, -288.0, 0.0, 31.0, 148.0, -152.0, -83.0, 88.0, 0.0, -13.0, 99.0, 75.0, -41.0, -4.0, 11.0, 0.0, -77.0, -26.0, -5.0, 10.0, 22.0, -23.0, -12.0, 0.0, 6.0, -16.0, 4.0, -19.0, 6.0, 18.0, -10.0, -17.0, 0.0, -21.0, 16.0, 7.0, -4.0, -5.0, 10.0, 11.0, -3.0, 1.0, 0.0, 1.0, 1.0, 3.0, 4.0, -4.0, -1.0, -1.0, 3.0, 1.0, -5.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
    ga[1980] = [0.0, -29992.0, -1956.0, -1997.0, 3027.0, 1663.0, 1281.0, -2180.0, 1251.0, 833.0, 938.0, 782.0, 398.0, -419.0, 199.0, -218.0, 357.0, 261.0, -74.0, -162.0, -48.0, 48.0, 66.0, 42.0, -192.0, 4.0, 14.0, -108.0, 72.0, -59.0, 2.0, 21.0, -12.0, 1.0, 11.0, -2.0, 18.0, 6.0, 0.0, -11.0, -7.0, 4.0, 3.0, 6.0, -1.0, 5.0, 10.0, 1.0, -12.0, 9.0, -3.0, -1.0, 7.0, 2.0, -5.0, -4.0, -4.0, 2.0, -5.0, -2.0, 5.0, 3.0, 1.0, 2.0, 3.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
    ha[1980] = [0.0, 0.0, 5604.0, 0.0, -2129.0, -200.0, 0.0, -336.0, 271.0, -252.0, 0.0, 212.0, -257.0, 53.0, -297.0, 0.0, 46.0, 150.0, -151.0, -78.0, 92.0, 0.0, -15.0, 93.0, 71.0, -43.0, -2.0, 17.0, 0.0, -82.0, -27.0, -5.0, 16.0, 18.0, -23.0, -10.0, 0.0, 7.0, -18.0, 4.0, -22.0, 9.0, 16.0, -13.0, -15.0, 0.0, -21.0, 16.0, 9.0, -5.0, -6.0, 9.0, 10.0, -6.0, 2.0, 0.0, 1.0, 0.0, 3.0, 6.0, -4.0, 0.0, -1.0, 4.0, 0.0, -6.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
    ga[1985] = [0.0, -29873.0, -1905.0, -2072.0, 3044.0, 1687.0, 1296.0, -2208.0, 1247.0, 829.0, 936.0, 780.0, 361.0, -424.0, 170.0, -214.0, 355.0, 253.0, -93.0, -164.0, -46.0, 53.0, 65.0, 51.0, -185.0, 4.0, 16.0, -102.0, 74.0, -62.0, 3.0, 24.0, -6.0, 4.0, 10.0, 0.0, 21.0, 6.0, 0.0, -11.0, -9.0, 4.0, 4.0, 4.0, -4.0, 5.0, 10.0, 1.0, -12.0, 9.0, -3.0, -1.0, 7.0, 1.0, -5.0, -4.0, -4.0, 3.0, -5.0, -2.0, 5.0, 3.0, 1.0, 2.0, 3.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
    ha[1985] = [0.0, 0.0, 5500.0, 0.0, -2197.0, -306.0, 0.0, -310.0, 284.0, -297.0, 0.0, 232.0, -249.0, 69.0, -297.0, 0.0, 47.0, 150.0, -154.0, -75.0, 95.0, 0.0, -16.0, 88.0, 69.0, -48.0, -1.0, 21.0, 0.0, -83.0, -27.0, -2.0, 20.0, 17.0, -23.0, -7.0, 0.0, 8.0, -19.0, 5.0, -23.0, 11.0, 14.0, -15.0, -11.0, 0.0, -21.0, 15.0, 9.0, -6.0, -6.0, 9.0, 9.0, -7.0, 2.0, 0.0, 1.0, 0.0, 3.0, 6.0, -4.0, 0.0, -1.0, 4.0, 0.0, -6.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
    ga[1990] = [0.0, -29775.0, -1848.0, -2131.0, 3059.0, 1686.0, 1314.0, -2239.0, 1248.0, 802.0, 939.0, 780.0, 325.0, -423.0, 141.0, -214.0, 353.0, 245.0, -109.0, -165.0, -36.0, 61.0, 65.0, 59.0, -178.0, 3.0, 18.0, -96.0, 77.0, -64.0, 2.0, 26.0, -1.0, 5.0, 9.0, 0.0, 23.0, 5.0, -1.0, -10.0, -12.0, 3.0, 4.0, 2.0, -6.0, 4.0, 9.0, 1.0, -12.0, 9.0, -4.0, -2.0, 7.0, 1.0, -6.0, -3.0, -4.0, 2.0, -5.0, -2.0, 4.0, 3.0, 1.0, 3.0, 3.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
    ha[1990] = [0.0, 0.0, 5406.0, 0.0, -2279.0, -373.0, 0.0, -284.0, 293.0, -352.0, 0.0, 247.0, -240.0, 84.0, -299.0, 0.0, 46.0, 154.0, -153.0, -69.0, 97.0, 0.0, -16.0, 82.0, 69.0, -52.0, 1.0, 24.0, 0.0, -80.0, -26.0, 0.0, 21.0, 17.0, -23.0, -4.0, 0.0, 10.0, -19.0, 6.0, -22.0, 12.0, 12.0, -16.0, -10.0, 0.0, -20.0, 15.0, 11.0, -7.0, -7.0, 9.0, 8.0, -7.0, 2.0, 0.0, 2.0, 1.0, 3.0, 6.0, -4.0, 0.0, -2.0, 3.0, -1.0, -6.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
    ga[1995] = [0.0, -29692.0, -1784.0, -2200.0, 3070.0, 1681.0, 1335.0, -2267.0, 1249.0, 759.0, 940.0, 780.0, 290.0, -418.0, 122.0, -214.0, 352.0, 235.0, -118.0, -166.0, -17.0, 68.0, 67.0, 68.0, -170.0, -1.0, 19.0, -93.0, 77.0, -72.0, 1.0, 28.0, 5.0, 4.0, 8.0, -2.0, 25.0, 6.0, -6.0, -9.0, -14.0, 9.0, 6.0, -5.0, -7.0, 4.0, 9.0, 3.0, -10.0, 8.0, -8.0, -1.0, 10.0, -2.0, -8.0, -3.0, -6.0, 2.0, -4.0, -1.0, 4.0, 2.0, 2.0, 5.0, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
    ha[1995] = [0.0, 0.0, 5306.0, 0.0, -2366.0, -413.0, 0.0, -262.0, 302.0, -427.0, 0.0, 262.0, -236.0, 97.0, -306.0, 0.0, 46.0, 165.0, -143.0, -55.0, 107.0, 0.0, -17.0, 72.0, 67.0, -58.0, 1.0, 36.0, 0.0, -69.0, -25.0, 4.0, 24.0, 17.0, -24.0, -6.0, 0.0, 11.0, -21.0, 8.0, -23.0, 15.0, 11.0, -16.0, -4.0, 0.0, -20.0, 15.0, 12.0, -6.0, -8.0, 8.0, 5.0, -8.0, 3.0, 0.0, 1.0, 0.0, 4.0, 5.0, -5.0, -1.0, -2.0, 1.0, -2.0, -7.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
    ga[2000] = [0.0, -29619.4, -1728.2, -2267.7, 3068.4, 1670.9, 1339.6, -2288.0, 1252.1, 714.5, 932.3, 786.8, 250.0, -403.0, 111.3, -218.8, 351.4, 222.3, -130.4, -168.6, -12.9, 72.3, 68.2, 74.2, -160.9, -5.9, 16.9, -90.4, 79.0, -74.0, 0.0, 33.3, 9.1, 6.9, 7.3, -1.2, 24.4, 6.6, -9.2, -7.9, -16.6, 9.1, 7.0, -7.9, -7.0, 5.0, 9.4, 3.0, -8.4, 6.3, -8.9, -1.5, 9.3, -4.3, -8.2, -2.6, -6.0, 1.7, -3.1, -0.5, 3.7, 1.0, 2.0, 4.2, 0.3, -1.1, 2.7, -1.7, -1.9, 1.5, -0.1, 0.1, -0.7, 0.7, 1.7, 0.1, 1.2, 4.0, -2.2, -0.3, 0.2, 0.9, -0.2, 0.9, -0.5, 0.3, -0.3, -0.4, -0.1, -0.2, -0.4, -0.2, -0.9, 0.3, 0.1, -0.4, 1.3, -0.4, 0.7, -0.4, 0.3, -0.1, 0.4, 0.0, 0.1]
    ha[2000] = [0.0, 0.0, 5186.1, 0.0, -2481.6, -458.0, 0.0, -227.6, 293.4, -491.1, 0.0, 272.6, -231.9, 119.8, -303.8, 0.0, 43.8, 171.9, -133.1, -39.3, 106.3, 0.0, -17.4, 63.7, 65.1, -61.2, 0.7, 43.8, 0.0, -64.6, -24.2, 6.2, 24.0, 14.8, -25.4, -5.8, 0.0, 11.9, -21.5, 8.5, -21.5, 15.5, 8.9, -14.9, -2.1, 0.0, -19.7, 13.4, 12.5, -6.2, -8.4, 8.4, 3.8, -8.2, 4.8, 0.0, 1.7, 0.0, 4.0, 4.9, -5.9, -1.2, -2.9, 0.2, -2.2, -7.4, 0.0, 0.1, 1.3, -0.9, -2.6, 0.9, -0.7, -2.8, -0.9, -1.2, -1.9, -0.9, 0.0, -0.4, 0.3, 2.5, -2.6, 0.7, 0.3, 0.0, 0.0, 0.3, -0.9, -0.4, 0.8, 0.0, -0.9, 0.2, 1.8, -0.4, -1.0, -0.1, 0.7, 0.3, 0.6, 0.3, -0.2, -0.5, -0.9]
    ga[2005] = [0.0, -29554.63, -1669.05, -2337.24, 3047.69, 1657.76, 1336.3, -2305.83, 1246.39, 672.51, 920.55, 797.96, 210.65, -379.86, 100.0, -227.0, 354.41, 208.95, -136.54, -168.05, -13.55, 73.6, 69.56, 76.74, -151.34, -14.58, 14.58, -86.36, 79.88, -74.46, -1.65, 38.73, 12.3, 9.37, 5.42, 1.94, 24.8, 7.62, -11.73, -6.88, -18.11, 10.17, 9.36, -11.25, -4.87, 5.58, 9.76, 3.58, -6.94, 5.01, -10.76, -1.25, 8.76, -6.66, -9.22, -2.17, -6.12, 1.42, -2.35, -0.15, 3.06, 0.29, 2.06, 3.77, -0.21, -2.09, 2.95, -1.6, -1.88, 1.44, -0.31, 0.29, -0.79, 0.53, 1.8, 0.16, 0.96, 3.99, -2.15, -0.29, 0.21, 0.89, -0.38, 0.96, -0.3, 0.46, -0.35, -0.36, 0.08, -0.49, -0.08, -0.16, -0.88, 0.3, 0.28, -0.43, 1.18, -0.37, 0.75, -0.26, 0.35, -0.05, 0.41, -0.1, -0.18]
    ha[2005] = [0.0, 0.0, 5077.99, 0.0, -2594.5, -515.43, 0.0, -198.86, 269.72, -524.72, 0.0, 282.07, -225.23, 145.15, -305.36, 0.0, 42.72, 180.25, -123.45, -19.57, 103.85, 0.0, -20.33, 54.75, 63.63, -63.53, 0.24, 50.94, 0.0, -61.14, -22.57, 6.82, 25.35, 10.93, -26.32, -4.64, 0.0, 11.2, -20.88, 9.83, -19.71, 16.22, 7.61, -12.76, -0.06, 0.0, -20.11, 12.69, 12.67, -6.72, -8.16, 8.1, 2.92, -7.73, 6.01, 0.0, 2.19, 0.1, 4.46, 4.76, -6.58, -1.01, -3.47, -0.86, -2.31, -7.93, 0.0, 0.26, 1.44, -0.77, -2.27, 0.9, -0.58, -2.69, -1.08, -1.58, -1.9, -1.39, 0.0, -0.55, 0.23, 2.38, -2.63, 0.61, 0.4, 0.01, 0.02, 0.28, -0.87, -0.34, 0.88, 0.0, -0.76, 0.33, 1.72, -0.54, -1.07, -0.04, 0.63, 0.21, 0.53, 0.38, -0.22, -0.57, -0.82]
    ga[2010] = [0.0, -29496.57, -1586.42, -2396.06, 3026.34, 1668.17, 1339.85, -2326.54, 1232.1, 633.73, 912.66, 808.97, 166.58, -356.83, 89.4, -230.87, 357.29, 200.26, -141.05, -163.17, -8.03, 72.78, 68.69, 75.92, -141.4, -22.83, 13.1, -78.09, 80.44, -75.0, -4.55, 45.24, 14.0, 10.46, 1.64, 4.92, 24.41, 8.21, -14.5, -5.59, -19.34, 11.61, 10.85, -14.05, -3.54, 5.5, 9.45, 3.45, -5.27, 3.13, -12.38, -0.76, 8.43, -8.42, -10.08, -1.94, -6.24, 0.89, -1.07, -0.16, 2.45, -0.33, 2.13, 3.09, -1.03, -2.8, 3.05, -1.48, -2.03, 1.65, -0.51, 0.54, -0.79, 0.37, 1.79, 0.12, 0.75, 3.75, -2.12, -0.21, 0.3, 1.04, -0.63, 0.95, -0.11, 0.52, -0.39, -0.37, 0.21, -0.77, 0.04, -0.09, -0.89, 0.31, 0.42, -0.45, 1.08, -0.31, 0.78, -0.18, 0.38, 0.02, 0.42, -0.26, -0.26]
    ha[2010] = [0.0, 0.0, 4944.26, 0.0, -2708.54, -575.73, 0.0, -160.4, 251.75, -537.03, 0.0, 286.48, -211.03, 164.46, -309.72, 0.0, 44.58, 189.01, -118.06, -0.01, 101.04, 0.0, -20.9, 44.18, 61.54, -66.26, 3.02, 55.4, 0.0, -57.8, -21.2, 6.54, 24.96, 7.03, -27.61, -3.28, 0.0, 10.84, -20.03, 11.83, -17.41, 16.71, 6.96, -10.74, 1.64, 0.0, -20.54, 11.51, 12.75, -7.14, -7.42, 7.97, 2.14, -6.08, 7.01, 0.0, 2.73, -0.1, 4.71, 4.44, -7.22, -0.96, -3.95, -1.99, -1.97, -8.31, 0.0, 0.13, 1.67, -0.66, -1.76, 0.85, -0.39, -2.51, -1.27, -2.11, -1.94, -1.86, 0.0, -0.87, 0.27, 2.13, -2.49, 0.49, 0.59, 0.0, 0.13, 0.27, -0.86, -0.23, 0.87, 0.0, -0.87, 0.3, 1.66, -0.59, -1.14, -0.07, 0.54, 0.1, 0.49, 0.44, -0.25, -0.53, -0.79]
    ga[2015] = [0.0, -29441.46, -1501.77, -2445.88, 3012.2, 1676.35, 1350.33, -2352.26, 1225.85, 581.69, 907.42, 813.68, 120.49, -334.85, 70.38, -232.91, 360.14, 192.35, -140.94, -157.4, 4.3, 69.55, 67.57, 72.79, -129.85, -28.93, 13.14, -70.85, 81.29, -75.99, -6.79, 51.82, 15.07, 9.32, -2.88, 6.61, 23.98, 8.89, -16.78, -3.16, -20.56, 13.33, 11.76, -15.98, -2.02, 5.33, 8.83, 3.02, -3.22, 0.67, -13.2, -0.1, 8.68, -9.06, -10.54, -2.01, -6.26, 0.17, 0.55, -0.55, 1.7, -0.67, 2.13, 2.33, -1.8, -3.59, 3, -1.4, -2.3, 2.08, -0.79, 0.58, -0.7, 0.14, 1.7, -0.22, 0.44, 3.49, -2.09, -0.16, 0.46, 1.23, -0.89, 0.85, 0.1, 0.54, -0.37, -0.43, 0.22, -0.94, -0.03, -0.02, -0.92, 0.42, 0.63, -0.42, 0.96, -0.19, 0.81, -0.13, 0.38, 0.08, 0.46, -0.35, -0.36]
    ha[2015] = [0.0, 0.0, 4795.99, 0.0, -2845.41, -642.17, 0.0, -115.29, 245.04, -538.7, 0.0, 283.54, -188.43, 180.95, -329.23, 0.0, 46.98, 196.98, -119.14, 15.98, 100.12, 0.0, -20.61, 33.3, 58.74, -66.64, 7.35, 62.41, 0.0, -54.27, -19.53, 5.59, 24.45, 3.27, -27.5, -2.32, 0.0, 10.04, -18.26, 13.18, -14.6, 16.16, 5.69, -9.1, 2.26, 0.0, -21.77, 10.76, 11.74, -6.74, -6.88, 7.79, 1.04, -3.89, 8.44, 0.0, 3.28, -0.4, 4.55, 4.4, -7.92, -0.61, -4.16, -2.85, -1.12, -8.72, 0.0, 0, 2.11, -0.6, -1.05, 0.76, -0.2, -2.12, -1.44, -2.57, -2.01, -2.34, 0.0, -1.08, 0.37, 1.75, -2.19, 0.27, 0.72, -0.09, 0.29, 0.23, -0.89, -0.16, 0.72, 0.0, -0.88, 0.49, 1.56, -0.5, -1.24, -0.1, 0.42, -0.04, 0.48, 0.48, -0.3, -0.43, -0.71]
    ga[2020] = [0.0, -29404.8, -1450.9, -2499.6, 2982.0, 1677.0, 1363.2, -2381.2, 1236.2, 525.7, 903.0, 809.5, 86.3, -309.4, 48.0, -234.3, 363.2, 187.8, -140.7, -151.2, 13.5, 66.0, 65.5, 72.9, -121.5, -36.2, 13.5, -64.7, 80.6, -76.7, -8.2, 56.5, 15.8, 6.4, -7.2, 9.8, 23.7, 9.7, -17.6, -0.5, -21.1, 15.3, 13.7, -16.5, -0.3, 5.0, 8.4, 2.9, -1.5, -1.1, -13.2, 1.1, 8.8, -9.3, -11.9, -1.9, -6.2, -0.1, 1.7, -0.9, 0.7, -0.9, 1.9, 1.4, -2.4, -3.8, 3.0, -1.4, -2.5, 2.3, -0.9, 0.3, -0.7, -0.1, 1.4, -0.6, 0.2, 3.1, -2.0, -0.1, 0.5, 1.3, -1.2, 0.7, 0.3, 0.5, -0.3, -0.5, 0.1, -1.1, -0.3, 0.1, -0.9, 0.5, 0.7, -0.3, 0.8, 0.0, 0.8, 0.0, 0.4, 0.1, 0.5, -0.5, -0.4]
    ha[2020] = [0.0, 0.0, 4652.5, 0.0, -2991.6, -734.6, 0.0, -82.1, 241.9, -543.4, 0.0, 281.9, -158.4, 199.7, -349.7, 0.0, 47.7, 208.3, -121.2, 32.3, 98.9, 0.0, -19.1, 25.1, 52.8, -64.5, 8.9, 68.1, 0.0, -51.5, -16.9, 2.2, 23.5, -2.2, -27.2, -1.8, 0.0, 8.4, -15.3, 12.8, -11.7, 14.9, 3.6, -6.9, 2.8, 0.0, -23.4, 11.0, 9.8, -5.1, -6.3, 7.8, 0.4, -1.4, 9.6, 0.0, 3.4, -0.2, 3.6, 4.8, -8.6, -0.1, -4.3, -3.4, -0.1, -8.8, 0.0, 0.0, 2.5, -0.6, -0.4, 0.6, -0.2, -1.7, -1.6, -3.0, -2.0, -2.6, 0.0, -1.2, 0.5, 1.4, -1.8, 0.1, 0.8, -0.2, 0.6, 0.2, -0.9, 0.0, 0.5, 0.0, -0.9, 0.6, 1.4, -0.4, -1.3, -0.1, 0.3, -0.1, 0.5, 0.5, -0.4, -0.4, -0.6]
    dg = [0.0, 5.7, 7.4, -11.0, -7.0, -2.1, 2.2, -5.9, 3.1, -12.0, -1.2, -1.6, -5.9, 5.2, -5.1, -0.3, 0.5, -0.6, 0.2, 1.3, 0.9, -0.5, -0.3, 0.4, 1.3, -1.4, 0.0, 0.9, -0.1, -0.2, 0.0, 0.7, 0.1, -0.5, -0.8, 0.8, 0.0, 0.1, -0.1, 0.4, -0.1, 0.4, 0.3, -0.1, 0.4, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
    dh = [0.0, 0.0, -25.9, 0.0, -30.2, -22.4, 0.0, 6.0, -1.1, 0.5, 0.0, -0.1, 6.5, 3.6, -5.0, 0.0, 0.0, 2.5, -0.6, 3.0, 0.3, 0.0, 0.0, -1.6, -1.3, 0.8, 0.0, 1.0, 0.0, 0.6, 0.6, -0.8, -0.2, -1.1, 0.1, 0.3, 0.0, -0.2, 0.6, -0.2, 0.5, -0.3, -0.4, 0.5, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]

    return minyear, maxyear, ga, ha, dg, dh
