import os
import traceback
from typing import Optional

import typer
from git import Repo
from github import Github
from github.PaginatedList import PaginatedList
from github.PullRequest import PullRequest

from repo_tools.common import utils
from repo_tools.common.help_texts import HelpText
from repo_tools.common.utils import (
    pull_request_to_string,
    query_yes_no,
    print_with_pad,
    print_header,
    OUTPUT_PADDING,
)

github_app = typer.Typer()

pr_app = typer.Typer()
github_app.add_typer(pr_app, name="pr")


@pr_app.command(help="Create a PR on GitHub")
def create(
    target_branch: str = typer.Option(...),
    title: str = typer.Option(...),
    body: str = typer.Option(...),
    reviewers: str = typer.Option(
        ..., help="Comma separated list of github usernames to add for review"
    ),
    glob: Optional[str] = typer.Option(
        "",
        "--glob",
        "-g",
        help=HelpText.GLOB,
    ),
    feature: Optional[str] = typer.Option(
        "",
        "--feature",
        "-f",
        help=HelpText.FEATURE,
    ),
    include: Optional[str] = typer.Option("", "--include", "-i", help=HelpText.INCLUDE),
    exclude: Optional[str] = typer.Option("", "--exclude", "-e", help=HelpText.EXCLUDE),
    dry_run: bool = typer.Option(False, help="execute a dry run, does not create PRs"),
    draft: bool = typer.Option(False, help="Create a PR in draft state"),
):
    github_api_client = Github(utils.get_github_oauth_token())
    github_org = utils.get_github_org()
    projects = utils.get_projects(feature, include, exclude, glob_pattern=glob)
    reviewers = reviewers.split(",")

    for project in projects:
        try:
            repo = Repo(project.abs_path)
            current_branch = str(repo.active_branch)

            if dry_run:
                typer.echo(
                    typer.style(
                        f"DRY RUN --> {project.name}:",
                        fg=typer.colors.YELLOW,
                        bold=True,
                    )
                )
                typer.echo(
                    f"   Running without --dry-run flag will create a PR for {project.name}\n"
                    f"   Merge '{current_branch}' into '{target_branch}'\n"
                    f"   Title: {title}"
                )
                continue

            repository = github_api_client.get_repo(f"{github_org}/{project.name}")
            pull_request = repository.create_pull(
                title=title,
                body=f"{body} \n\n - autogenerated PR",
                head=current_branch,
                base=target_branch,
                draft=draft,
            )
            pull_request.create_review_request(reviewers=reviewers)
            typer.echo(
                typer.style(f"{project.name}:", fg=typer.colors.GREEN, bold=True)
            )
            typer.echo(
                f"   PR created: {pull_request.html_url}, review assigned to {', '.join(reviewers)}"
            )
        except Exception as exc:
            typer.echo(typer.style(f"{project.name}:", fg=typer.colors.RED, bold=True))
            typer.echo(f"   GitHub PR creation failed: {exc}")
            continue


@pr_app.command(help="Update a PR on GitHub")
def update(
    title: Optional[str] = typer.Option(None, help="New PR title"),
    body: Optional[str] = typer.Option(
        None, help="Text to append/replace the body with"
    ),
    overwrite: bool = typer.Option(False, help="Whether to overwrite the body of a PR"),
    reviewers: str = typer.Option(
        "", help="Comma separated list of github usernames to add for review"
    ),
    glob: Optional[str] = typer.Option(
        "",
        "--glob",
        "-g",
        help=HelpText.GLOB,
    ),
    feature: Optional[str] = typer.Option(
        "",
        "--feature",
        "-f",
        help=HelpText.FEATURE,
    ),
    include: Optional[str] = typer.Option("", "--include", "-i", help=HelpText.INCLUDE),
    exclude: Optional[str] = typer.Option("", "--exclude", "-e", help=HelpText.EXCLUDE),
    dry_run: bool = typer.Option(False, help="execute a dry run, does not create PRs"),
    draft: Optional[bool] = typer.Option(None, help="Set the PR as draft"),
    safe: bool = typer.Option(True, help="Will prompt for each update"),
    default_yes: bool = typer.Option(
        False, help="Set 'yes' as default yes/no questions answer"
    ),
):
    github_api_client = Github(utils.get_github_oauth_token())
    github_org = utils.get_github_org()
    projects = utils.get_projects(feature, include, exclude, glob_pattern=glob)
    reviewers = [r for r in reviewers.split(",") if r]

    if not reviewers and body is None and title is None:
        typer.echo(typer.style("GENERIC:", fg=typer.colors.RED, bold=True))
        print_with_pad("One of --reviewers, --body or --title must be set")
        return

    if draft is not None:
        typer.echo(typer.style("GENERIC:", fg=typer.colors.RED, bold=True))
        print_with_pad("Changing 'draft' state is not allowed yet by GitHub API...")
        return

    for project in projects:
        try:
            typer.echo(
                typer.style(f"{project.name}:", fg=typer.colors.GREEN, bold=True)
            )
            repo = Repo(project.abs_path)

            repository = github_api_client.get_repo(f"{github_org}/{project.name}")
            org = repository.organization.login
            query_head = f"{org}:{repo.active_branch}"
            pull_requests: PaginatedList[PullRequest] = repository.get_pulls(
                state="opened", sort="created", head=query_head
            )

            if pull_requests.totalCount == 0:
                typer.echo(
                    typer.style(f"{project.name}:", fg=typer.colors.RED, bold=True)
                )
                print_with_pad("GitHub PR update failed: could not find any PRs")
                continue
            if pull_requests.totalCount > 1:
                typer.echo(
                    typer.style(f"{project.name}:", fg=typer.colors.RED, bold=True)
                )
                print_with_pad(
                    "GitHub PR update failed: there is more then one pull request opened"
                )
                for pr in pull_requests:
                    print_with_pad(f"{repository.name}, {pr.title}, {pr.html_url}")
                continue
            pr = pull_requests[0]
            print_with_pad(
                f"Updating {repository.name!r}, PR {pr.title!r} ({pr.html_url}):"
            )

            args = {}

            if title is not None:
                args["title"] = title
                print_header("Title")
                print_with_pad(f"Old: {pr.title!r}")
                print_with_pad(f"New: {title!r}")
            if body is not None:
                body = bytes(body, "utf-8").decode("unicode_escape")
                if not overwrite:
                    body = f"{pr.body}\n{body}"
                args["body"] = body
                print_header("Body")
                print_with_pad(f"Old: {pr.body!r}")
                print_with_pad(f"New: {body!r}")
            reviewers_clean = set()
            if reviewers:
                user_review_requests, _ = pr.get_review_requests()
                old_reviewers = sorted([r.login for r in user_review_requests])
                reviewers_clean = list(set(reviewers) - set(old_reviewers))
                old_reviewers_string = ", ".join(old_reviewers)
                new_reviewers_string = ", ".join(old_reviewers + reviewers_clean)
                print_header("Reviewers")
                print_with_pad(f"Old: {old_reviewers_string}")
                print_with_pad(f"New: {new_reviewers_string}")
            if dry_run:
                print_with_pad("Dry run: no write operations are sent to GitHub")
                continue
            if safe and not query_yes_no(
                OUTPUT_PADDING + "Update this PR?", default_yes=default_yes
            ):
                print_with_pad("Update aborted!")
                continue
            if reviewers_clean:
                pr.create_review_request(reviewers_clean)
            pr.edit(**args)
            print_with_pad(f"PR updated: {pr.html_url}, PR state after changes:")
            print_with_pad(pull_request_to_string(pr))
        except Exception:
            typer.echo(typer.style(f"{project.name}:", fg=typer.colors.RED, bold=True))
            print_with_pad(f"GitHub PR creation failed: {traceback.format_exc()}")
            continue


@github_app.command(
    help="Clone all repositories from the configured GitHub organisation"
)
def clone_all(
    path: Optional[str] = typer.Argument(
        ".", help="target path to clone all repositories to."
    )
):
    cwd = os.path.abspath(os.path.join(os.getcwd(), path))
    os.makedirs(cwd, exist_ok=True)
    github_api_client = Github(utils.get_github_oauth_token())
    github_org_name = utils.get_github_org()

    org = github_api_client.get_organization(github_org_name)

    repositories = org.get_repos()
    typer.echo(f"Found {repositories.totalCount} repositories in {github_org_name}.")
    confirm = typer.confirm(
        f"Are you sure you want to clone all repositories to {cwd} ?"
    )

    if not confirm:
        typer.echo("Not continuing...")
        raise typer.Abort()

    for repo in repositories:
        typer.echo(typer.style(f"Cloning {repo.full_name}...", bold=True))
        clone_command = f"git clone {repo.ssh_url}"
        out, status_code = utils.execute_command(clone_command, cwd=cwd)
        utils.display_command_output(repo.full_name, out, status_code)
