# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['cavcalc', 'cavcalc._arguments', 'cavcalc.functions', 'cavcalc.parameters']

package_data = \
{'': ['*']}

install_requires = \
['Pint>=0.19.2,<0.20.0', 'matplotlib>=3.6.1,<4.0.0', 'numpy>=1.23.3,<2.0.0']

extras_require = \
{':python_version < "3.9"': ['importlib-metadata>=4.0,<5.0']}

entry_points = \
{'console_scripts': ['cavcalc = cavcalc.__main__:cli_entry']}

setup_kwargs = {
    'name': 'cavcalc',
    'version': '1.0.0',
    'description': 'A program for computing Fabry-Perot optical cavity parameters.',
    'long_description': '![](https://gitlab.com/sjrowlinson/cavcalc/raw/master/images/logo_cavcalc.png)\n\nA command line program and Python module for computing parameters associated with linear, Fabry-Perot optical cavities.\n\n- Find the documentation at: https://cavcalc.readthedocs.io/en/latest/\n- Follow the latest changes: https://gitlab.com/sjrowlinson/cavcalc\n- See the entry on PyPI: https://pypi.org/project/cavcalc/\n\n## Installation\nTo install `cavcalc`, simply run:\n\n```\npip install cavcalc\n```\n\nCheck that the installation was successful with:\n\n```bash\ncavcalc --version\n```\n\nif you see something along the lines of\n\n```\ncavcalc v1.0.0\n```\n\nthen you should be ready to start using `cavcalc`!\n\n**Note**: As is often recommended with the installation of Python packages (especially those with dependencies\non packages such as `numpy` and `matplotlib`, as is the case here), you should prefer to install `cavcalc` in\na suitable virtual environment. See [the official documentation on Python virtual environments](https://docs.python.org/3/tutorial/venv.html)\nfor details on how to set up these if you are unfamiliar with the topic.\n\n## Example usage via the CLI\n\nFor details on available arguments run `cavcalc -h` on the command line.\n\nSome examples follow on how to use `cavcalc`. See [the documentation on using cavcalc](https://cavcalc.readthedocs.io/en/latest/using/index.html)\nfor more in-depth examples and guides.\n\n### Computing single parameters\n\nYou can ask for, e.g., the beam size on the mirrors of a symmetric cavity given its length and stability factor of the mirrors (gs) with:\n\n```bash\ncavcalc w -L 4000 -gs 0.91\n```\n\nThis would result in an output of:\n\n```\nGiven:\n\tCavity length = 4000 m\n\n\tStability g-factor of both mirrors = 0.91\n\n\tWavelength of beam = 1064 nm\n\nComputed:\n\tRadius of beam at mirrors = 57.16193267930482 mm\n```\n\nUnits for both inputs and outputs can also be specified:\n\n```bash\ncavcalc w -u mm -L 10km -gouy 145deg\n```\n\nThis requests the beam radius (in mm) on the mirrors of a symmetric cavity of length 10km given that the\nround-trip Gouy phase is 145 degrees; resulting in the following output:\n\n```\nGiven:\n\tCavity length = 10 km\n\n\tRound-trip Gouy phase = 145 deg\n\n\tWavelength of beam = 1064 nm\n\nComputed:\n\tRadius of beam at mirrors = 59.59174828941794 mm\n```\n\nSupport for units is provided via the package [Pint](https://pint.readthedocs.io/en/stable/index.html), so any units\ndefined in the Pint unit-registry can be used in cavcalc.\n\n### Computing all available parameters\n\nA compute target of `all` is the default choice which is used to calculate all parameters which can be determined\nfrom the arguments specified. For example, using approximate values of the Advanced LIGO arm cavity parameters,\n\n```bash\ncavcalc -L 4km -Rc1 1934 -Rc2 2245 -T1 0.014 -L1 37.5e-6 -T2 5e-6 -L2 37.5e-6\n```\n\ngives the following output:\n\n```\nGiven:\n\tLoss of first mirror = 3.75e-05\n\tLoss of second mirror = 3.75e-05\n\tTransmission of first mirror = 0.014\n\tTransmission of second mirror = 5e-06\n\n\tCavity length = 4 km\n\n\tRadius of curvature of first mirror = 1934 m\n\tRadius of curvature of second mirror = 2245 m\n\n\tWavelength of beam = 1064 nm\n\nComputed:\n\tFSR = 37474.05725 Hz\n\tFWHM = 84.56921734107604 Hz\n\tMode separation frequency = 4988.072188176178 Hz\n\tPole frequency = 42.28460867053802 Hz\n\n\tFinesse = 443.11699254426594\n\tReflectivity of first mirror = 0.9859625\n\tReflectivity of second mirror = 0.9999574999999999\n\n\tPosition of beam waist (from first mirror) = 1837.2153886417168 m\n\n\tRadius of beam at first mirror = 53.421066433049255 mm\n\tRadius of beam at second mirror = 62.448079883230896 mm\n\tRadius of beam at waist = 11.950538458990879 mm\n\n\tStability g-factor of cavity = 0.8350925761717987\n\tStability g-factor of first mirror = -1.0682523267838677\n\tStability g-factor of second mirror = -0.7817371937639199\n\n\tRound-trip Gouy phase = 312.0813565565169 deg\n```\n\n### Units of output\n\nThe default behaviour for the output parameter units is to grab the relevant parameter type option under the `[units]` header\nof the `cavcalc.ini` configuration file. When installing `cavcalc`, this file is written to a new `cavcalc/` directory within\nyour config directory (i.e. typically `~/.config/cavcalc/cavcalc.ini` under Unix systems). See the comments in this file for\ndetails on the options available for the output units of each parameter type.\n\n`cavcalc` attempts to read a `cavcalc.ini` config file from several locations in this fixed order:\n\n- Firstly from the current working directory, then\n- from `$XDG_CONFIG_HOME/cavcalc/` (or `%APPDATA%/cavcalc/` on Windows), then\n- the final read attempt is from the within the source of the package directory itself.\n\nThe config options from these read attempts are loaded in a standard way; that is, any options appearing\nfirst in the sequence defined above will take priority. If any of the above read attempts fails, then this\nwill be a silent failure; the only situation where an error could occur would be when *all* of the above\nread attempts fail (which is very unlikely to happen).\n\nNote that if you specify a `-u` argument when running `cavcalc` from the CLI, then this takes priority over\nthe options in the config file (as we saw in an example above).\n\n### Evaluating, and plotting, parameters over data ranges\n\nParameters can be computed over ranges of data using:\n\n* the data range syntax: `-<param_name> "start stop num [<units>]"`,\n* or data from an input file with `-<param_name> <file>`.\n\nWe can use data-ranges to compute, and plot, arrays of target values, e.g:\n\n```bash\ncavcalc w -L "1 10 100 km" -Rc 5.1km --plot\n```\n\nThis results in a plot (see below) showing how the beam radius at the mirrors of a symmetric cavity varies from\na cavity length of 1 km to 10 km with 100 data points, with the radii of curvature of both mirrors fixed at 5.1 km.\n\n![](https://gitlab.com/sjrowlinson/cavcalc/raw/master/images/symmcav_ws_vs_lengths.png)\n\n\nAlternatively one could use a file of data, e.g:\n\n```bash\ncavcalc gouy -L 5cm -w beam_radii.txt --plot --saveplot symmcav_gouy_vs_ws.png\n```\n\nThis then computes the round-trip Gouy phase (in degrees) of a symmetric cavity of length 5cm\nusing beam radii data stored in a file `beam_radii.txt`, and plots the results (see below). Note also that\nyou can save the resulting figure using the `--saveplot <filename>` syntax as seen in the above command.\n\n![](https://gitlab.com/sjrowlinson/cavcalc/raw/master/images/symmcav_gouy_vs_ws.png)\n\nFrom the plot above you can also see that cavcalc supports automatically plotting of quantities which can be\ndual-valued. In this case, the Gouy phase can be one of two values for each beam radius; this is due to the\nnature of the equations which govern the Fabry-Perot cavity dynamics.\n\n### Image / density plots via `--mesh`\n\nWhen multiple arguments are given as data-ranges, one can use the `--mesh` option to construct mesh-grids\nof these parameters. This allows cavcalc to automatically produce image plots. For example:\n\n```bash\ncavcalc w -L "1 10 100 km" -gouy "20 120 100 deg" --mesh true --plot\n```\n\ncomputes the radius of the beam on the mirrors of a symmetric cavity, against both the cavity length and\nround-trip Gouy phase on a grid. This results in the plot shown below. Note that we simply used `--mesh true`\nhere, which automatically determines the ordering of the mesh-grid parameters based on the order in which\nthese parameters were given. One could replace the above with, e.g., `--mesh "gouy,L"` to reverse the order\nof the mesh-grid; and thereby flip the parameter axes on any automated plots.\n\n![](https://gitlab.com/sjrowlinson/cavcalc/raw/master/images/symmcav_w_vs_L_gouy.png)\n\nA matplotlib compliant colour-map can be specified when making an image plot using the `--cmap <name>` option. For example,\nthe following command gives the plot shown below.\n\n```bash\ncavcalc gouy -L 12um -g1 "-2 2 499" -g2 "-2 2 499" --mesh true --plot --cmap Spectral_r\n```\n\n![](https://gitlab.com/sjrowlinson/cavcalc/raw/master/images/asymmcav_w0_vs_g1g2.png)\n\n## A note on g-factors\n\nStability (g) factors are split into four different parameters for implementation purposes and to\nhopefully make it clearer as to which argument is being used and whether the resulting cavity\ncomputations are for a symmetric or asymmetric cavity. These arguments are detailed here:\n\n- `-gs` : The symmetric, singular stability factor. This represents the individual g-factors of **both**\n          cavity mirrors. Use this to define a *symmetric* cavity where the overall cavity g-factor is\n\t\t  then simply `g = gs * gs`.\n- `-g` : The overall cavity stability factor. This is the product of the individual g-factors of the\n         cavity mirrors. Use this to define a *symmetric* cavity where the individual g-factors of **both**\n\t\t mirrors are then `gs = +/- sqrt(g)`.\n- `-g1` : The stability factor of the first cavity mirror. Use this to define an *asymmetric* cavity\n          along with the argument `-g2` such that the overall cavity g-factor is then `g = g1 * g2`.\n- `-g2` : The stability factor of the second cavity mirror. Use this to define an *asymmetric* cavity\n          along with the argument `-g1` such that the overall cavity g-factor is then `g = g1 * g2`.\n\n---\n\n## Using `cavcalc` programmatically\n\nAs well as providing a CLI, cavcalc has a full API which allows users to interact with this tool\nvia Python. The recommended method for doing this is to use the single-function interface via\n[`cavcalc.calculate`](https://cavcalc.readthedocs.io/en/latest/api/generated/cavcalc.calculate.calculate.html#cavcalc.calculate.calculate). This\nfunction works similarly to the CLI, where a target can be specified along with a variable number of keyword\narguments corresponding to the physical parameters. This function then returns one of two output objects (`SingleOutput`\nif a target was given, `MultiOutput` otherwise); see [`cavcalc.output`](https://cavcalc.readthedocs.io/en/latest/api/cavcalc.output.html#module-cavcalc.output)\nfor details.\n\nFor example, the following script will compute all available targets from the cavity length and mirror radii\nof curvature provided:\n\n```python\nimport cavcalc as cc\n\n# Specifying no target means all possible targets are computed\nout = cc.calculate(L="4km", Rc1=1934, Rc2=2245)\n\n# Printing the output object results in the same output as\n# you would see when running via the CLI\nprint(out)\n```\n\nresults in:\n\n```\nGiven:\n\tCavity length = 4 km\n\n\tRadius of curvature of first mirror = 1934 m\n\tRadius of curvature of second mirror = 2245 m\n\n\tWavelength of beam = 1064 nm\n\nComputed:\n\tFSR = 37474.05725 Hz\n\tMode separation frequency = 4988.072188176178 Hz\n\n\tPosition of beam waist (from first mirror) = 1837.2153886417168 m\n\n\tRadius of beam at first mirror = 53.421066433049255 mm\n\tRadius of beam at second mirror = 62.448079883230896 mm\n\tRadius of beam at waist = 11.950538458990879 mm\n\n\tStability g-factor of cavity = 0.8350925761717987\n\tStability g-factor of first mirror = -1.0682523267838677\n\tStability g-factor of second mirror = -0.7817371937639199\n\n\tRound-trip Gouy phase = 312.0813565565169 deg\n```\n\nAn extra feature of the API is the ability to use the `cavcalc.configure` function for overriding\ndefault behaviour. For example, in the script below we use this in a context-managed scope to\ntemporarily use microns for any beam radius parameters, mm for distances, and GHz for any frequencies:\n\n```python\nimport cavcalc as cc\n\n# Temporarily override units...\nwith cc.configure(beamsizes="um", distances="mm", frequencies="GHz"):\n    out = cc.calculate(L="8mm", gouy=121)\n    print(out)\n\n# ... previous state (using loaded config options) will\n# be restored on exit from the with block above\n```\n\nresulting in:\n\n```\nGiven:\n    Cavity length = 8 mm\n\n    Round-trip Gouy phase = 121 deg\n\n    Wavelength of beam = 1064 nm\n\nComputed:\n    FSR = 18.737028625 GHz\n    Mode separation frequency = 6.297723510069445 GHz\n\n    Position of beam waist (from first mirror) = 4.0 mm\n\n    Radius of curvature of both mirrors = 0.01576117284251957 m\n\n    Radius of beam at mirrors = 55.79464044247193 µm\n    Radius of beam at waist = 48.19739141432035 µm\n\n    Stability g-factor of cavity = 0.2424809625449729\n    Stability g-factor of both mirrors = 0.4924235601034671\n\n    Divergence angle = 0.4026158371180008 deg\n```\n',
    'author': 'Samuel Rowlinson',
    'author_email': 'samueljrowlinson@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://cavcalc.readthedocs.io/en/latest/',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
