# pytest-resource-usage

Add running times and peak memory usage of tests to the output of
[pytest](https://pytest.org). Two ways of measuring memory usage are
implemented. The first uses `tracemalloc`, which is unaffected by
swapping. Since `tracemalloc` is a standard library, this way of
measuring introduces no dependencies beyond `pytest`. However, measuring
with `tracemalloc` can have a high overhead. The other way of measuring
memory usage is by means of sampling the unique set size. This method
has lower overhead, but can be less accurate and is only available if
the `psutil` package is installed.

The `pytest-resource-usage` package is intentionally kept fairly simple.
If you want something more feature-rich and ambitious, you should use
[pytest-monitor](https://github.com/CFMTech/pytest-monitor).


## Example usage

Reporting is triggered by the presence of one or more of the following markers:
* `report_duration`,
* `report_tracemalloc`,
* `report_uss` (requires `psutil`).

```python
from time import sleep

import pytest


@pytest.mark.report_duration
def test_sleep():
    sleep(99)


@pytest.mark.report_uss
@pytest.mark.report_tracemalloc
@pytest.mark.report_duration
@pytest.mark.parametrize("elements", [2_000_000, 1_000_000])
def test_tracemalloc_overhead(elements):
    _ = list(range(elements))


@pytest.mark.report_uss(interval=0.01)
@pytest.mark.report_duration
def test_unique_set_size():
    _ = list(range(1_000_000))


@pytest.mark.report_duration
def test_no_overhead():
    _ = list(range(1_000_000))
```

Running the above tests produces the following `pytest` output.

```
================================================================ test session starts =================================================================
plugins: pytest_resource_usage-0.2.0
collected 5 items                                                                                                                                    

tests/test_readme.py .....                                                                                                                     [100%]

=================================================================== resource usage ===================================================================
tests/test_readme.py::test_sleep (call) running time: 0:01:39
tests/test_readme.py::test_tracemalloc_overhead[2000000] (call) peak allocated memory: 72MB, peak unique set size: 243MB, running time: 0.627 seconds
tests/test_readme.py::test_tracemalloc_overhead[1000000] (call) peak allocated memory: 36MB, peak unique set size: 115MB, running time: 0.329 seconds
tests/test_readme.py::test_unique_set_size (call) peak unique set size: 25.7MB, running time: 0.047 seconds
tests/test_readme.py::test_no_overhead (call) running time: 0.020 seconds
=========================================================== 5 passed in 100.13s (0:01:40) ============================================================
```
