"""
This module contains functions for constructing SGE submission scripts for
running Gaussian calculations
"""

import os


def create_data_file(com_f_name, n_cores):
    """
    Creates .DATA file which is copy of .com file with number of
    cores (threads) and memory specified.

    Parameters
    ----------
        f_name : str
            File name of .com file generated by gaussian_suite gen function
        n_cores : int
            Number of cores (threads) to request. Memory will be set to
            2*n_cores in gigabytes

    Returns
    -------
        None
    """

    f_name_head = os.path.splitext(com_f_name)[0]

    data_f_name = "{}.DATA".format(f_name_head)

    # Copy .com file to .data file
    # Replace MEMORY in data file
    with open(data_f_name, "w") as fd, open(com_f_name, "r") as fc:

        fd.write("%nproc={:d}\n".format(n_cores))
        for line in fc:
            if "MEMORYGB" in line:
                fd.write("%mem={:d}GB\n".format(n_cores * 2))
            else:
                fd.write("{}".format(line))

    return


def create_submission_script(f_name, n_cores, gen_fchk=True, short=False):
    """
    Creates submission script for CSF SGE queue

    Parameters
    ----------
        f_name : str
            File name of .data file generated by create_data_file function
        n_cores : int
            Number of cores (threads) to request. Memory will be set to
            2*n_cores in gigabytes
        gen_fchk : bool, default True
            If true, generates formatted checkpoint file using formchk utility
        short : bool, default False
            If true, request short queue (1 hour and 12 core max.)

    Returns
    -------
        None
    """

    f_name_head = os.path.splitext(f_name)[0]

    # Check n_cores if short requested
    if short and n_cores > 12:
        print("Short queue requested with >12 cores")
        print("Program aborts")
        exit()

    # Get checkpoint file name
    with open(f_name, "r") as fd:
        for line in fd:
            if "%chk" in line:
                chk_f_name = line.split('=')[-1]

    with open("{}.sh".format(f_name_head), "w") as fj:
        fj.write("#$ -S /bin/bash\n")
        fj.write("#$ -N {}\n".format(f_name_head))
        fj.write("#$ -pe smp.pe {:d}\n".format(n_cores))
        if short:
            fj.write("#$ -l short\n")
        fj.write("#$ -cwd\n")
        fj.write("#$ -V\n")
        fj.write("#$ -o {}.out\n".format(f_name_head))
        fj.write("#$ -e {}.err\n".format(f_name_head))
        fj.write("export JNAME={}\n".format(f_name_head))
        fj.write("export CurDir={}\n".format(os.getcwd()))
        fj.write(". /etc/profile\n")
        fj.write("module load apps/binapps/gaussian/g09d01_em64t\n")
        fj.write(
            'export GAUSS_SCRDIR=/scratch/$USER/Gaussian/gau_temp_$JOB_ID\n'
        )
        fj.write('mkdir -p $GAUSS_SCRDIR\n')
        fj.write('cp $CurDir/$JNAME.DATA $GAUSS_SCRDIR\n')
        fj.write('$g09root/g09/g09 < ./$JNAME.DATA > $CurDir/$JNAME.log\n')
        fj.write('cp -pr *.log $CurDir/.\n')
        fj.write('cp -pr *.chk $CurDir/.\n')
        fj.write('cp fort.7 $CurDir/$JNAME.fort7\n')
        fj.write("\n")
        fj.write('cd $CurDir/.\n')
        if gen_fchk:
            fj.write('formchk {}'.format(chk_f_name))
        fj.write(
            '#Job {} submitted from $PWD and running in scrdir in the log file\n'.format( # noqa
                f_name_head
            )
        ) # noqa

    os.system("chmod +x {}.sh".format(f_name_head))

    return
