"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntegTesting = void 0;
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ifw = require(".");
class IntegTesting {
    constructor() {
        const app = new cdk.App();
        const env = {
            region: process.env.CDK_INTEG_REGION || process.env.CDK_DEFAULT_REGION || 'us-east-1',
            account: process.env.CDK_INTEG_ACCOUNT || process.env.CDK_DEFAULT_ACCOUNT,
        };
        const stack = new cdk.Stack(app, 'integ-stack', { env });
        const database = new aws_cdk_lib_1.aws_timestream.CfnDatabase(stack, 'Database', {
            databaseName: 'FleetWise',
        });
        const table = new aws_cdk_lib_1.aws_timestream.CfnTable(stack, 'Table', {
            databaseName: 'FleetWise',
            tableName: 'FleetWise',
        });
        table.node.addDependency(database);
        const role = new aws_cdk_lib_1.aws_iam.Role(stack, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('iotfleetwise.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AdministratorAccess'),
            ],
        });
        const signalCatalog = new ifw.SignalCatalog(stack, 'SignalCatalog', {
            database,
            table,
            role,
            description: 'my signal catalog',
            nodes: [
                new ifw.SignalCatalogBranch('Vehicle', 'Vehicle'),
                new ifw.SignalCatalogSensor('EngineTorque', 'Vehicle.EngineTorque', 'DOUBLE'),
            ],
        });
        const model_a = new ifw.VehicleModel(stack, 'ModelA', {
            signalCatalog,
            name: 'modelA',
            description: 'Model A vehicle',
            networkInterfaces: [new ifw.CanVehicleInterface('1', 'vcan0')],
            signals: [
                new ifw.CanVehicleSignal('EngineTorque', 'Vehicle.EngineTorque', '1', 401, // messageId
                1.0, // factor
                true, // isBigEndian
                false, // isSigned
                8, // length
                0.0, // offset
                9),
            ],
        });
        const vin100 = new ifw.Vehicle(stack, 'vin100', {
            vehicleId: 'vin100',
            vehicleModel: model_a,
            createIotThing: true,
        });
        new cdk.CfnOutput(stack, 'certificateId', { value: vin100.certificateId });
        const vpc = aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(stack, 'VPC', { isDefault: true });
        const securityGroup = new aws_cdk_lib_1.aws_ec2.SecurityGroup(stack, 'SecurityGroup', {
            vpc,
            allowAllOutbound: true,
        });
        securityGroup.addIngressRule(aws_cdk_lib_1.aws_ec2.Peer.anyIpv4(), aws_cdk_lib_1.aws_ec2.Port.tcp(22), 'SSH access');
        securityGroup.addIngressRule(aws_cdk_lib_1.aws_ec2.Peer.anyIpv4(), aws_cdk_lib_1.aws_ec2.Port.allIcmp(), 'ping');
        // EC2 role
        const ec2_role = new aws_cdk_lib_1.aws_iam.Role(stack, 'ec2Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AdministratorAccess'),
            ],
        });
        // Ubuntu 18.04 for Arm64
        const machineImage = aws_cdk_lib_1.aws_ec2.MachineImage.fromSsmParameter('/aws/service/canonical/ubuntu/server/18.04/stable/current/arm64/hvm/ebs-gp2/ami-id', { os: aws_cdk_lib_1.aws_ec2.OperatingSystemType.LINUX });
        // Create the Vehicle simulator
        const instance = new aws_cdk_lib_1.aws_ec2.Instance(stack, 'VehicleSim', {
            vpc: vpc,
            instanceType: new aws_cdk_lib_1.aws_ec2.InstanceType('m6g.xlarge'),
            machineImage,
            securityGroup,
            keyName: 'salamida',
            role: ec2_role,
            vpcSubnets: {
                subnetGroupName: 'Public',
            },
            resourceSignalTimeout: cdk.Duration.minutes(30),
        });
        const sourceUrl = 'https://github.com/aws/aws-iot-fleetwise-edge.git';
        const userData = `\
    set -euo pipefail

    # Wait for any existing package install to finish
    i=0
    while true; do
        if sudo fuser /var/{lib/{dpkg,apt/lists},cache/apt/archives}/lock >/dev/null 2>&1; then
            i=0
        else
            i=\`expr $i + 1\`
            if expr $i \>= 10 > /dev/null; then
                break
            fi
        fi
        sleep 1
    done

    # Upgrade system and reboot if required
    apt update && apt upgrade -y
    if [ -f /var/run/reboot-required ]; then
      # Delete the UserData info file so that we run again after reboot
      rm -f /var/lib/cloud/instances/*/sem/config_scripts_user
      reboot
      exit
    fi

    # Install helper scripts:
    apt update && apt install -y python3-setuptools
    mkdir -p /opt/aws/bin
    wget https://s3.amazonaws.com/cloudformation-examples/aws-cfn-bootstrap-py3-latest.tar.gz
    python3 -m easy_install --script-dir /opt/aws/bin aws-cfn-bootstrap-py3-latest.tar.gz
    rm aws-cfn-bootstrap-py3-latest.tar.gz

    # On error, signal back to cfn:
    error_handler() {
      /opt/aws/bin/cfn-signal --success false --stack ${stack.stackName} --resource ${instance.instance.logicalId} --region ${stack.region}
    }
    trap error_handler ERR

    # Increase pid_max:
    echo 1048576 > /proc/sys/kernel/pid_max
    # Disable syslog:
    systemctl stop syslog.socket rsyslog.service
    # Remove journald rate limiting and set max size:
    printf "RateLimitBurst=0\nSystemMaxUse=1G\n" >> /etc/systemd/journald.conf

    # Install packages
    apt update && apt install -y git ec2-instance-connect htop jq unzip

    # Install AWS CLI:
    curl "https://awscli.amazonaws.com/awscli-exe-linux-aarch64.zip" -o "awscliv2.zip"
    unzip awscliv2.zip
    ./aws/install
    rm awscliv2.zip

    # Download source
    cd /home/ubuntu
    if echo ${sourceUrl} | grep -q 's3://'; then
      sudo -u ubuntu aws s3 cp ${sourceUrl} aws-iot-fleetwise-edge.zip
      sudo -u ubuntu unzip aws-iot-fleetwise-edge.zip -d aws-iot-fleetwise-edge
    else
      sudo -u ubuntu git clone ${sourceUrl} aws-iot-fleetwise-edge
    fi
    cd aws-iot-fleetwise-edge
    
    # Install SocketCAN modules:
    ./tools/install-socketcan.sh --bus-count 1
    
    # Install CAN Simulator
    ./tools/install-cansim.sh --bus-count 1
    
    # Install FWE credentials and config file
    mkdir /etc/aws-iot-fleetwise
    echo -n "${vin100.certificatePem}" > /etc/aws-iot-fleetwise/certificate.pem
    echo -n "${vin100.privateKey}" > /etc/aws-iot-fleetwise/private-key.key
    ./tools/configure-fwe.sh \
      --input-config-file "configuration/static-config.json" \
      --output-config-file "/etc/aws-iot-fleetwise/config-0.json" \
      --vehicle-id vin100 \
      --endpoint-url "${vin100.endpointAddress}" \
      --topic-prefix '$aws/iotfleetwise/' \
      --can-bus0 "vcan0"

    # Install source deps
    ./tools/install-deps-native.sh

    # Build source
    sudo -u ubuntu ./tools/build-fwe-native.sh

    # Install FWE
    ./tools/install-fwe.sh
    
    # Signal init complete:
    /opt/aws/bin/cfn-signal --stack ${stack.stackName} --resource ${instance.instance.logicalId} --region ${stack.region}`;
        instance.addUserData(userData);
        new cdk.CfnOutput(stack, 'Vehicle IP Address', { value: instance.instancePublicIp });
        new cdk.CfnOutput(stack, 'Vehicle ssh command', { value: 'ssh -i salamida.pem -o IdentitiesOnly=yes ec2-user@' + instance.instancePublicIp });
        new ifw.Campaign(stack, 'Campaign1', {
            name: 'FwTimeBasedCampaign1',
            target: vin100,
            collectionScheme: new ifw.TimeBasedCollectionScheme(cdk.Duration.seconds(10)),
            signals: [
                new ifw.CampaignSignal('Vehicle.EngineTorque'),
            ],
            autoApprove: true,
        });
        new ifw.Fleet(stack, 'Fleet2', {
            fleetId: 'fleet2',
            signalCatalog,
            vehicles: [vin100],
        });
        this.stack = [stack];
    }
}
exports.IntegTesting = IntegTesting;
process.env.GITLAB_REGISTRATION_TOKEN = 'mock';
process.env.CDK_INTEG_REGION = 'eu-central-1';
new IntegTesting();
//# sourceMappingURL=data:application/json;base64,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