# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['opensearch_logger']

package_data = \
{'': ['*']}

install_requires = \
['opensearch-py>=1.0,<2.0']

setup_kwargs = {
    'name': 'opensearch-logger',
    'version': '1.0.0',
    'description': 'OpenSearch logging handler',
    'long_description': '# Opensearch Logger for Python\n\n<p>\n    <a href="https://pypi.org/pypi/opensearch-logger"><img alt="Package version" src="https://img.shields.io/pypi/v/opensearch-logger?logo=python&logoColor=white&color=blue"></a>\n    <a href="https://pypi.org/pypi/opensearch-logger"><img alt="Supported python versions" src="https://img.shields.io/pypi/pyversions/opensearch-logger?logo=python&logoColor=white"></a>\n    <a href="https://pypi.org/pypi/opensearch-logger"><img alt="Package stability" src="https://img.shields.io/pypi/status/opensearch-logger?logo=python&logoColor=white&color=blue"></a>\n    <a href="https://codecov.io/gh/vduseev/opensearch-logger"><img alt="Code coverage" src="https://img.shields.io/codecov/c/github/vduseev/opensearch-logger?logo=codecov&logoColor=white&color=white"></a>\n    <a href="https://github.com/vduseev/opensearch-logger/actions/workflows/test.yml"><img alt="Tests (main branch)" src="https://img.shields.io/github/workflow/status/vduseev/opensearch-logger/Test/main?logo=github"></a>\n    <a href="https://pypi.org/pypi/opensearch-logger"><img alt="License" src="https://img.shields.io/pypi/l/opensearch-logger"></a>\n</p>\n\nThis library provides a standard Python logging handler compatible with [Opensearch][opensearch] suite.\n\nThe **goals** of this project are\n\n* to provide a **simple** and direct logging from Python to Opensearch without *fluentd*, *logstash* or other middleware;\n* keep it up to date with the growing difference between Opensearch and Elasticsearch projects;\n* keep the library easy to use, robust, and simple.\n\nThe library has been open-sourced from an internal project where it has been successfully used in production\nsince the release of Opensearch 1.0.\n\nGenerated log records follow the [Elastic Common Schema (ECS)][ecs] field naming convention.\nFor better performance it is recommended to set up a proper mapping for you logging indices but everything will\nwork even without it. You can find a ready to use [compatible JSON mapping][ecs-mapping] in the repository.\n\n## Installation\n\n```shell\npip install opensearch-logger\n```\n\n## Usage\n\nJust add the handler to your logger as follows\n\n```python\nimport logging\nfrom opensearch_logger import OpensearchHandler\n\nhandler = OpensearchHandler(\n    index_name="my-logs",\n    hosts=["https://localhost:9200"],\n    http_auth=("admin", "admin"),\n    use_ssl=True,\n    verify_certs=False,\n    ssl_assert_hostname=False,\n    ssl_show_warn=False,\n)\n\nlogger = logging.getLogger(__name__)\nlogger.setLevel(logging.INFO)\nlogger.addHandler(handler)\n```\n\nTo log into Opensearch, simply use the regular logging commands:\n\n```python\nlogger.info("This message will be indexed in Opensearch")\n\n# Report extra fields\nstart_time = time.perf_counter()\nheavy_database_operation()\nelapsed_time = time.perf_counter() - start_time\n\nlogger.info(f"Database operation took {elapsed_time:.3f} seconds", extra={"elapsed_time": elapsed_time})\n```\n\n## Configuration\n\nThe `OpensearchHandler` constructor several parameters from the first table below to control name of the index,\nbuffering, and some general behavior. None of this parameters are mandatory.\n\nAll other keyword arguments are passed directly to the underlying Opensearch python client.\nFull list of connection parameters can be found in [`opensearch-py`][opensearch-py] docs.\nAt least one connection parameter **must** be provided, otherwise a `TypeError` will be thrown.\n\n## Logging parameters\n\n| Parameter | Default | Description |\n| - | - | - |\n| `index_name` | `"python-logs"` | Base name of the Opensearch index name that will be created. |\n| `index_rotate` | `DAILY` | Frequency that controls what date is appended to index name during its creation. `OpensearchHandler.DAILY`. |\n| `index_date_format` | `"%Y.%m.%d"` | Format of the date that gets appended to the base index name. |\n| `index_name_sep` | `"-"` | Separator string between `index_name` and the date, appended to the index name. |\n| `buffer_size` | `1000` | Number of log records which when reached on the internal buffer results in a flush to Opensearch. |\n| `flush_frequency` | `1` | Float representing how often the buffer will be flushed (in seconds). |\n| `extra_fields` | `{}` | Nested dictionary with all the additional fields that you would like to add to all logs. |\n| `raise_on_index_exc` | `False` | Raise exception if indexing to Opensearch fails. |\n\n## Connection parameters\n\nHere are few examples of the connection parameters supported by the Opensearch client.\n\n| Parameter | Example | Description |\n| - | - | - |\n| `hosts` | `["https://localhost:9200"]` | The list of hosts to connect to. Multiple hosts are allowed. |\n| `http_auth` | `("admin", "admin")` | Username and password to authenticate against the Opensearch servers. |\n| `http_compress` | `True` | Enables gzip compression for request bodies. |\n| `use_ssl` | `True` | Whether communications should be SSL encrypted. |\n| `verify_certs` | `False` | Whether the SSL certificates are validated or not. |\n| `ssl_assert_hostname` | `False` | Verify authenticity of host for encrypted connections. |\n| `ssl_show_warn` | `False` | Enable warning for SSL connections. |\n| `ca_carts` | `"/var/lib/root-ca.pem"` | CA bundle path for using intermediate CAs with your root CA. |\n\n## Configuring using logging.config or in Django\n\nAs most other log handlers, `opensearch-logger` support configuration via `logging.config` facility.\nJust specify the `opensearch_logger.OpensearchHandler` as one of the handlers and provide parameters to it.\n\nFull guide on tweaking `logging.config` can be found in the [official python documentation][logging-config].\n\n```python\nimport logging.config\n\nLOGGING = {\n    "version": 1,\n    "disable_existing_loggers": False,\n    "handlers": {\n        "file": {\n            "level": "DEBUG",\n            "class": "logging.handlers.RotatingFileHandler",\n            "filename": "./debug.log",\n            "maxBytes": 102400,\n            "backupCount": 5,\n        },\n        "opensearch": {\n            "level": "INFO",\n            "class": "opensearch_logger.OpensearchHandler",\n            "hosts": [{"host": "localhost", "port": 9200}],\n            "index_name": "my-logs",\n            "extra_fields": {"App": "test", "Environment": "dev"},\n            "use_ssl": True,\n            "verify_certs": False,\n        },\n    },\n    "loggers": {\n        "root": {\n            "handlers": ["file", "opensearch"],\n            "level": "INFO",\n            "propogate": False,\n        },\n        "django": {\n            "handlers": ["file","opensearch"],\n            "level": "DEBUG",\n            "propagate": True,\n        },\n    },\n}\n\nlogging.config.dictConfig(LOGGING)\n```\n\n## Dependencies\n\nThis library uses the following packages\n\n* [`opensearch-py`][opensearch-py]\n\n## Building from source & Developing\n\nThis package uses [`pyenv`][pyenv] (optional) and [Poetry][poetry] for development purposes.\nIt also uses Docker to run Opensearch container for integration testing during development.\n\n1. Clone the repo.\n1. Instruct poetry to use a proper Python version and install dependencies.\n\n   ```shell\n   poetry env use 3.8.12\n   poetry install\n   ```\n\n1. Run tests\n\n   **WARNING**: You need opensearch running on `https://localhost:9200` to run the tests.\n   Part of the tests verifies that correct number of logs actually gets into Opensearch.\n   Alternatively, you can specify the `TEST_OPENSEARCH_HOST` variable and set it to a different value pointing\n   to the running Opensearch server.\n\n   There are not many tests, but they run with **5 seconds cooldown each** to allow Opensearch to process the\n   newly sent log records properly and verify their count.\n\n   Small helper scripts are available in the `tests/` directory to start and stop Opensearch using Docker.\n\n   ```shell\n   # Give it 5-10 seconds to initialize before running tests\n   tests/start-opensearch-docker.sh\n\n   # Run tests\n   poetry run pytest\n\n   # Run coverage tests\n   poetry run pytest --cov --cov-report=html\n\n   # Run mypy typing verification\n   poetry run pytest --mypy opensearch_logger --strict-markers\n\n   # Run flake8 to make sure code style is correct\n   poetry run flake8\n\n   # Turn off Opensearch\n   tests/stop-opensearch-docker.sh\n   ```\n\n   Before turning the Opensearch container off, it is possible to check that the records are actually there.\n\n   ```shell\n   # Verify index is in place and has required number of records\n   $ curl -k -XGET "https://admin:admin@localhost:9200/_cat/indices/test*?v&s=index"\n   health status index                             uuid                   pri rep docs.count docs.deleted store.size pri.store.size\n   yellow open   test-opensearch-logger-2021.11.08 N0BEEnG2RIuPP0l8RZE0Dg   1   1          7            0     29.7kb         29.7kb\n   ```\n\n1. Build a package\n\n   ```shell\n   poetry build\n   ```\n\n## Contributions\n\nContributions are welcome! 👏  🎉\n\nPlease create a GitHub issue and a Pull Request that references that issue as well as your proposed changes.\nYour Pull Request will be automatically tested using GitHub actions.\n\nAfter your pull request will be accepted, it will be merged and the version of the library will be bumped\nand released to PyPI.\n\n## History\n\nThis is a fork of [Python Elasticsearch ECS Log handler][python-elasticsearch-ecs-logger] project\nwhich was in turn forked from [Python Elasticsearch Logger][python-elasticsearch-logger] project.\nWhile original is perfectly suitable for logging to Elasticsearch, due to the split between\nOpensearch and Elasticsearch it makes sense to make a fork entirely tailored to work with Opensearch\nand based on the official [`opensearch-py`][opensearch-py] Python library.\n\nThe API between `python-elasticsearch-ecs-logger` and this project has slightly changed for better\ncompatibility with Opensearch and for the purposes of simplification.\n\n## License\n\nDistributed under the terms of [Apache 2.0][apache-2.0] license, opensearch-logger is free and open source software.\n\n[opensearch]: https://opensearch.org/\n[opensearch-py]: https://pypi.org/project/opensearch-py/\n[logging]: https://docs.python.org/3/library/logging.html\n[ecs]: https://www.elastic.co/guide/en/ecs/current/index.html\n[logging-config]: https://docs.python.org/3/library/logging.config.html\n[pyenv]: https://github.com/pyenv/pyenv\n[poetry]: https://python-poetry.org/\n[ecs-mapping]: https://github.com/vduseev/opensearch-logger/blob/main/mappings/ecs1.4.0_compatible_minimal.json\n[apache-2.0]: https://github.com/vduseev/opensearch-logger/blob/main/LICENSE.md\n[python-elasticsearch-ecs-logger]: https://github.com/IMInterne/python-elasticsearch-ecs-logger\n[python-elasticsearch-logger]: https://github.com/cmanaha/python-elasticsearch-logger\n',
    'author': 'vduseev',
    'author_email': 'vagiz@duseev.com',
    'maintainer': 'vduseev',
    'maintainer_email': 'vagiz@duseev.com',
    'url': 'https://github.com/vduseev/python-opensearch-logger',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
