""""""
# ----------------------------------------------------------------------------
# Description    : SCPI interface
# Git repository : https://gitlab.com/qblox/packages/software/qblox_instruments.git
# Copyright (C) Qblox BV (2020)
# ----------------------------------------------------------------------------


# ----------------------------------------------------------------------------
# THIS FILE IS AUTOMATICALLY GENERATED. DO NOT MODIFY THIS FILE MANUALLY!
# ----------------------------------------------------------------------------


# -- include ------------------------------------------------------------------

import sys
import struct
import re
import json
from functools import wraps
from typing    import Any, Dict, List, Optional, Tuple

# Add IEEE488.2 support
from qblox_instruments.ieee488_2 import Ieee488_2, Transport


# -- decorator-----------------------------------------------------------------

def scpi_error_check(func):
    """
    Decorator that catches and checks for errors on an SCPI call.

    Parameters
    ----------
    func
        Class method that performs an SCPI call

    Returns
    ----------

    Raises
    ----------
    RuntimeError
        An error was found in system error.
    """

    @wraps(func)
    def decorator_wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except OSError:
            raise
        except Exception as err:
            args[0]._check_error_queue(err)
        finally:
            args[0]._check_error_queue()
    return decorator_wrapper


# -- class --------------------------------------------------------------------

class PulsarQcm(Ieee488_2):
    """
    This interface provides an API for the mandatory and required SCPI calls and adds
    Pulsar related functionality
    (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).
    """

    # -------------------------------------------------------------------------
    def __init__(self, transport: Transport, debug: int=0):
        """
        Creates SCPI interface object.

        Parameters
        ----------
        transport : Transport
            Transport class responsible for the lowest level of communication
            (e.g. Ethernet).
        debug : int
            Debug level (0 = normal, 1 = no version check, >1 = no version or
            error checking).

        Returns
        ----------

        Raises
        ----------
        ConnectionError
            Debug level is 0 and there is a device or version mismatch.
        """

        # Store parameters for later use.
        self._debug = debug

        # Initialize parent class.
        super(PulsarQcm, self).__init__(transport)

        if self._debug == 0:
            # Check if device and build are compatible.
            _, device, *_, build = self._read("*IDN?").split(',')
            if device.replace("Cluster", "Pulsar") != 'Pulsar QCM':
                raise ConnectionError("This class expects a Pulsar QCM device, but found a {}. Please use the right device class.".format(device))
            build_ref = "fwVersion=0.8.0 fwBuild=08/12/2022-11:21:38 fwHash=0x94624739 fwDirty=0 kmodVersion=0.8.0 kmodBuild=08/12/2022-11:21:38 kmodHash=0x94624739 kmodDirty=0 swVersion=0.8.0 swBuild=08/12/2022-11:21:38 swHash=0x94624739 swDirty=0 cfgManVersion=0.2.0 cfgManBuild=08/12/2022-11:21:38 cfgManHash=0x94624739 cfgManDirty=0"
            if build != build_ref:
                build     = re.split(' |=', build)
                build_ref = re.split(' |=', build_ref)
                raise ConnectionError("Qblox Instruments package version is not compatible with device version:\n" + \
                                      "\n" + \
                                      "                        {:<25} {}\n".format("Expected:", "Actual:")  + \
                                      "Firmware;      Version: {:<25} {}\n".format(build_ref[1],  build[1])  + \
                                      "               Date:    {:<25} {}\n".format(build_ref[3],  build[3])  + \
                                      "               Hash:    {:<25} {}\n".format(build_ref[5],  build[5])  + \
                                      "               Dirty:   {:<25} {}\n".format(build_ref[7],  build[7])  + \
                                      "Kernel module; Version: {:<25} {}\n".format(build_ref[9],  build[9])  + \
                                      "               Date:    {:<25} {}\n".format(build_ref[11], build[11]) + \
                                      "               Hash:    {:<25} {}\n".format(build_ref[13], build[13]) + \
                                      "               Dirty:   {:<25} {}\n".format(build_ref[15], build[15]) + \
                                      "Application;   Version: {:<25} {}\n".format(build_ref[17], build[17]) + \
                                      "               Date:    {:<25} {}\n".format(build_ref[19], build[19]) + \
                                      "               Hash:    {:<25} {}\n".format(build_ref[21], build[21]) + \
                                      "               Dirty:   {:<25} {}\n".format(build_ref[23], build[23]) + \
                                      "\n" + \
                                      "Please update your device's firmware or the Qblox Instruments package.")

            # Clear SCPI error queue.
            while int(self._read('SYSTem:ERRor:COUNt?')) != 0:
                self._read('SYSTem:ERRor:NEXT?')

    # -------------------------------------------------------------------------
    def get_system_error(self) -> str:
        """
        Get system error from queue (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        str
            System error description string.

        """

        # SCPI call
        var0 = self._read('SYSTem:ERRor:NEXT?')

        return var0

    # -------------------------------------------------------------------------
    def get_num_system_error(self) -> int:
        """
        Get number of system errors (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Current number of system errors.

        """

        # SCPI call
        var0 = self._read('SYSTem:ERRor:COUNt?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_idn(self) -> str:
        """
        Get device identity and build information.

        Parameters
        ----------
        None

        Returns
        -------
        str
            Concatenated list of strings separated by the semicolon character. The IDN consists of four strings respectively ordered as:
            
            - Manufacturer
            - Model
            - Serial number
            - Build information

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('*IDN?')

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_system_version(self) -> str:
        """
        Get SCPI system version (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        str
            SCPI system version.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('SYSTem:VERSion?')

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_scpi_commands(self) -> str:
        """
        Get SCPI commands.

        Parameters
        ----------
        None

        Returns
        -------
        str
            Concatenated list of strings separated by the semicolon character. Each command consists of nine sections respectively ordered as:
            
             - SCPI command pattern
             - SCPI input type
             - SCPI output type
             - Python function
             - Python input types (comma separated)
             - Python input variable names (comma separated)
             - Python output types (comma separated)
             - User access level (0 = public, >=1 = private)
             - Comment
            

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('*CMDS?')

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _temp_file_new(self) -> int:
        """
        Creates a temporary file for the purpose of sending a large batch of data to the instrument, such as an update file.

        Parameters
        ----------
        None

        Returns
        -------
        int
            Handle by which the file can be identified.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('TMPFILe:NEW?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _temp_file_delete(self, handle: int) -> None:
        """
        Deletes a temporary file.

        Parameters
        ----------
        handle : int
            Temporary file handle.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('TMPFILe:DELete {}'.format(handle))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _temp_file_size(self, handle: int) -> int:
        """
        Returns the size of a temporary file in bytes.

        Parameters
        ----------
        handle : int
            Temporary file handle.

        Returns
        -------
        int
            Size of the file in bytes.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('TMPFILe:SIZe? {}'.format(handle))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _temp_file_block_count(self, handle: int) -> int:
        """
        Returns the number of blocks that can be read from a temporary file.

        Parameters
        ----------
        handle : int
            Temporary file handle.

        Returns
        -------
        int
            Number of blocks that can be read.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('TMPFILe:BLOCK:COUNT? {}'.format(handle))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _temp_file_block_read(self, handle: int, block: int) -> bytes:
        """
        Reads the nth block of a temporary file.

        Parameters
        ----------
        handle : int
            Temporary file handle.
        block : int
            Block index.

        Returns
        -------
        bytes
            Contents of the block.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        var0 = self._read_bin('TMPFILe:BLOCK:READ? {},{}'.format(handle, block))

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _temp_file_append(self, handle: int, data: bytes) -> None:
        """
        Appends a block of data to a temporary file.

        Parameters
        ----------
        handle : int
            Temporary file handle.
        data : bytes
            Data to append.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'bytes'])

        # SCPI call
        self._write_bin('TMPFILe:APPend {},'.format(handle), data)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _task_poll(self, handle: int) -> float:
        """
        Polls for completion of the given task.

        Parameters
        ----------
        handle : int
            Task handle.

        Returns
        -------
        float
            Task progress from 0.0 to 1.0. 1.0 indicates completion.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('TASK:POLL? {}'.format(handle))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _task_get_result(self, handle: int) -> Any:
        """
        Retrieves the value returned by a task. The task must be complete. If the task yielded an error, the error will be returned by this call. The task handle is deleted once the value is returned.

        Parameters
        ----------
        handle : int
            Task handle.

        Returns
        -------
        any
            The result of the task.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read_bin('TASK:RESult? {}'.format(handle))

        return json.loads(var0.decode('utf-8'))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def set_name(self, name: str) -> None:
        """
        Sets the customer-specified name of the instrument. The name must not contain any newlines, backslashes, or double quotes.

        Parameters
        ----------
        name : str
            The new name for the device.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['str'])

        # SCPI call
        self._write('SYSTem:NAME "{}"'.format(name))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_name(self) -> str:
        """
        Returns the customer-specified name of the instrument.

        Parameters
        ----------
        None

        Returns
        -------
        str
            The name of the device.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('SYSTem:NAME?')

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _download_log(self, source: str, format: int) -> int:
        """
        Prepares a batch of log data for retrieval via temp_file_*(). This may take some time, so instead of returning immediately, this spawns a task. task_poll() may be used to poll the state of the task, once this returns True (task complete), use task_get_result() to get the temp_file handle.

        Parameters
        ----------
        source : str
            The log source. Must be "system" for the Linux system log,"cfg_man" for the configuration manager log, or "app" for the main application log.
        format : int
            The format. Specify a positive integer to get up to that number of lines from the log. Specify zero to get the complete current log file (older log entries may exist via logrotate, only the file currently being written is returned). Specify a negative number to get all available log data in .tgz format.

        Returns
        -------
        int
            Task handle.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['str', 'int'])

        # SCPI call
        var0 = self._read('SYSTem:LOG? "{}",{}'.format(source, format))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def set_ip_config(self, config: str) -> None:
        """
        Reconfigures the IP address of this device. The configuration will not go into effect until reboot() is called or the device is power-cycled.

        Parameters
        ----------
        config : str
            IP configuration. May be one of the following things:
             - an IPv4 address including prefix length, for example 192.168.0.2/24, - the string `dhcp` to enable IPv4 DHCP, - an IPv6 address including prefix length, for example 1:2::3:4/64, or - a semicolon-separated combination of an IPv4 configuration (IP address or `dhcp`) and an IPv6 address.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['str'])

        # SCPI call
        self._write('SYSTem:IPCONFig "{}"'.format(config))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_ip_config(self) -> str:
        """
        Returns the IP address configuration that will go into effect when the device reboots.

        Parameters
        ----------
        None

        Returns
        -------
        str
            IP configuration. Can be one of the following things:
             - an IPv4 address including prefix length, for example 192.168.0.2/24, - the string `dhcp` to enable IPv4 DHCP, - an IPv6 address including prefix length, for example 1:2::3:4/64, or - a semicolon-separated combination of an IPv4 configuration (IP address or `dhcp`) and an IPv6 address.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('SYSTem:IPCONFig?')

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_update_module_types(self) -> str:
        """
        Returns a comma-separated list of device types that may be updated in addition to our own device type in the form of cluster modules. Returns ``not_applicable`` if this is not applicable to this device type.

        Parameters
        ----------
        None

        Returns
        -------
        str
            Comma-separated list of device types.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('SYSTem:UPDATE:MODules?')

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _update(self, handle: int) -> int:
        """
        Updates the device with an update tarball that was previously uploaded using temp_file_*() commands. This command merely starts the update process, and returns a task handle that can polled for completion via task_poll(), at which point any errors that may have occurred must be retrieved using task_get_result(). The task will return the string "OK" if successful. If this happens, the device must be rebooted via the reboot() command or via a power cycle to fully apply the update. Do not power down the device while the update task is running.

        Parameters
        ----------
        handle : int
            Temporary file handle for the update tarball.

        Returns
        -------
        int
            Task handle.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SYSTem:UPDATE:PREPare? {}'.format(handle))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _rollback(self) -> int:
        """
        Rolls back the device to the previous version, if rollback information is available. This command merely starts the rollback process, and returns a task handle that can polled for completion via task_poll(), at which point any errors that may have occurred must be retrieved using task_get_result(). The task will return the string "OK" if successful. If this happens, the device must be rebooted via the reboot() command or via a power cycle to fully apply the rollback. Do not power down the device while the rollback task is running.

        Parameters
        ----------
        None

        Returns
        -------
        int
            Task handle.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('SYSTem:ROLLBACK:PREPare?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def reboot(self) -> None:
        """
        Reboots the instrument.

        Parameters
        ----------
        None

        Returns
        -------
        None

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        self._write('SYSTem:REBOOT')

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _reset(self) -> None:
        """
        Reset device and clear all status and event registers (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        None

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        self._write('*RST')

    # -------------------------------------------------------------------------
    @scpi_error_check
    def clear(self) -> None:
        """
        Clear all status and event registers (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        None

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        self._write('*CLS')

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_status_byte(self) -> int:
        """
        Get status byte register. Register is only cleared when feeding registers are cleared (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Status byte register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('*STB?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def set_service_request_enable(self, reg: int) -> None:
        """
        Set service request enable register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        reg : int
            Service request enable register.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('*SRE {}'.format(reg))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_service_request_enable(self) -> int:
        """
        Get service request enable register. The register is cleared after reading it (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Service request enable register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('*SRE?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def set_standard_event_status_enable(self, reg: int) -> None:
        """
        Set standard event status enable register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        reg : int
            Standard event status enable register.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('*ESE {}'.format(reg))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_standard_event_status_enable(self) -> int:
        """
        Get standard event status enable register. The register is cleared after reading it (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Standard event status enable register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('*ESE?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_standard_event_status(self) -> int:
        """
        Get standard event status register. The register is cleared after reading it (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Standard event status register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('*ESR?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def set_operation_complete(self) -> None:
        """
        Set device in operation complete query active state (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        None

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        self._write('*OPC')

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_operation_complete(self) -> bool:
        """
        Get operation complete state (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        bool
            Operation complete state (False = running, True = completed).

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('*OPC?')

        return bool(int(var0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def test(self) -> bool:
        """
        Run self-test. Currently not implemented (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        bool
            Test result (False = failed, True = success).

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('*TST?')

        return bool(int(var0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def wait(self) -> None:
        """
        Wait until operations completed before continuing (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        None

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        self._write('*WAI')

    # -------------------------------------------------------------------------
    @scpi_error_check
    def preset_system_status(self) -> None:
        """
        Preset system status registers. Connects general system status flags for PLL unlock and temperature out-of-range indications
        to event status enable, status questionable temperature and status questionable frequency registers respectively
        (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        None

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        self._write('STATus:PRESet')

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_questionable_condition(self) -> int:
        """
        Get status questionable condition register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Status questionable condition register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:CONDition?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_questionable_event(self) -> int:
        """
        Get status questionable event register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Status questionable event register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:EVENt?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def set_questionable_enable(self, reg: int) -> None:
        """
        Set status questionable enable register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        reg : int
            Status questionable enable register.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('STATus:QUEStionable:ENABle {}'.format(reg))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_questionable_enable(self) -> int:
        """
        Get status questionable enable register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Status questionable enable register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:ENABle?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_operation_condition(self) -> int:
        """
        Get status operation condition register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Status operation condition register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:OPERation:CONDition?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_operation_events(self) -> int:
        """
        Get status operation event register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Status operation event register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:OPERation:EVENt?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def set_operation_enable(self, reg: int) -> None:
        """
        Set status operation enable register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        reg : int
            Status operation enable register.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('STATus:OPERation:ENABle {}'.format(reg))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_operation_enable(self) -> int:
        """
        Get status operation enable register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Status operation enable register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:OPERation:ENABle?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def identify(self) -> None:
        """
        Toggle frontpanel LEDs to visually identify the instrument.

        Parameters
        ----------
        None

        Returns
        -------
        None

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        self._write('*IDENtify')

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_system_state(self) -> str:
        """
        Get general system state.

        Parameters
        ----------
        None

        Returns
        -------
        str
            Concatenated list of strings separated by the semicolon character. Status is indicated by one status string and an optional number of flags respectively ordered as:
            
            :Status:
            
             - ``OKAY``: System is okay.
             - ``CRITICAL``: An error indicated by the flags occurred, but has been resolved.
             - ``ERROR``: An error indicated by the flags is occurring.
            
            :Flags:
            
             - ``CARRIER PLL UNLOCKED``
             - ``FPGA PLL UNLOCKED``
             - ``LO PLL UNLOCKED`` (only for RF-modules)
             - ``FPGA TEMPERATURE OUT-OF-RANGE``
             - ``CARRIER TEMPERATURE OUT-OF-RANGE``
             - ``AFE TEMPERATURE OUT-OF-RANGE``
             - ``LO TEMPERATURE OUT-OF-RANGE``
             - ``BACKPLANE TEMPERATURE OUT-OF-RANGE`` (only for Cluster)
            

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:GENeral:STATE?')

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_reference_source(self, internal: bool) -> None:
        """
        Set reference (10MHz) clock source.

        Parameters
        ----------
        internal : bool
            Reference clock source (False = External, True = Internal).

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['bool'])

        # SCPI call
        self._write('STATus:QUEStionable:FREQuency:REFerence:SRC {}'.format(0 if internal == False else 1))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_reference_source(self) -> bool:
        """
        Get reference (10MHz) clock source.

        Parameters
        ----------
        None

        Returns
        -------
        bool
            Reference clock source (False = External, True = Internal).

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:FREQuency:REFerence:SRC?')

        return bool(int(var0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_lo_hw_present(self) -> bool:
        """
        Check if LO hardware is present.

        Parameters
        ----------
        None

        Returns
        -------
        bool
            LO hardware presence (False = not present, True = present).

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('LO:PRESent?')

        return bool(int(var0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_current_afe_temperature(self) -> float:
        """
        Get current analog frontend board temperature (inside device).

        Parameters
        ----------
        None

        Returns
        -------
        float
            Current analog frontend board temperature.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:TEMPerature:AFE:CURRent?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_maximum_afe_temperature(self) -> float:
        """
        Get maximum analog frontend board temperature since boot or clear (inside device).

        Parameters
        ----------
        None

        Returns
        -------
        float
            Maximum analog frontend board temperature.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:TEMPerature:AFE:MAXimum?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_current_lo_temperature(self) -> float:
        """
        Get current local oscillator board temperature (inside device).

        Parameters
        ----------
        None

        Returns
        -------
        float
            Current local oscillator board temperature.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:TEMPerature:LO:CURRent?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_maximum_lo_temperature(self) -> float:
        """
        Get maximum local oscillator board temperature since boot or clear (inside device).

        Parameters
        ----------
        None

        Returns
        -------
        float
            Maximum local oscillator board temperature.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:TEMPerature:LO:MAXimum?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_current_fpga_temperature(self) -> float:
        """
        Get current FPGA junction temperature (inside device).

        Parameters
        ----------
        None

        Returns
        -------
        float
            Current FPGA junction temperature.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:TEMPerature:FPGA:CURRent?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_maximum_fpga_temperature(self) -> float:
        """
        Get maximum FPGA junction temperature since boot or clear (inside device).

        Parameters
        ----------
        None

        Returns
        -------
        float
            Maximum FPGA junction temperature.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:TEMPerature:FPGA:MAXimum?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_current_carrier_temperature(self) -> float:
        """
        Get current carrier board temperature (inside device).

        Parameters
        ----------
        None

        Returns
        -------
        float
            Current carrier board temperature.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:TEMPerature:CARRier:CURRent?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_maximum_carrier_temperature(self) -> float:
        """
        Get maximum carrier board temperature since boot or clear (inside device).

        Parameters
        ----------
        None

        Returns
        -------
        float
            Maximum carrier board temperature.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:TEMPerature:CARRier:MAXimum?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_lo_freq_0(self, freq: int) -> None:
        """
        Set LO frequency for output 0.

        Parameters
        ----------
        freq : int
            Frequency in Hertz.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('STATus:QUEStionable:FREQuency:LO0 {}'.format(freq))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_lo_freq_0(self) -> int:
        """
        Get LO frequency for output 0.

        Parameters
        ----------
        None

        Returns
        -------
        int
            Current frequency in Hertz.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:FREQuency:LO0?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_lo_pwr_0(self, pwr: int) -> None:
        """
        Set LO power for output 0.

        Parameters
        ----------
        pwr : int
            Power in dBm.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('STATus:QUEStionable:POWer:LO0 {}'.format(pwr))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_lo_pwr_0(self) -> int:
        """
        Get LO power for output 0.

        Parameters
        ----------
        None

        Returns
        -------
        int
            Power in dBm.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:POWer:LO0?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_lo_enable_0(self, lo_enable_0: bool) -> None:
        """
        Set LO0status.

        Parameters
        ----------
        lo_enable_0 : bool
            LO0status

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['bool'])

        # SCPI call
        self._write('LO0:ENAble {}'.format(0 if lo_enable_0 == False else 1))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_lo_enable_0(self) -> bool:
        """
        Get LO 0status.

        Parameters
        ----------
        None

        Returns
        -------
        bool
            LO0status

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('LO0:ENAble?')

        return bool(int(var0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_lo_freq_1(self, freq: int) -> None:
        """
        Set LO frequency for output 1.

        Parameters
        ----------
        freq : int
            Frequency in Hertz.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('STATus:QUEStionable:FREQuency:LO1 {}'.format(freq))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_lo_freq_1(self) -> int:
        """
        Get LO frequency for output 1.

        Parameters
        ----------
        None

        Returns
        -------
        int
            Current frequency in Hertz.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:FREQuency:LO1?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_lo_pwr_1(self, pwr: int) -> None:
        """
        Set LO power for output 1.

        Parameters
        ----------
        pwr : int
            Power in dBm.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('STATus:QUEStionable:POWer:LO1 {}'.format(pwr))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_lo_pwr_1(self) -> int:
        """
        Get LO power for output 1.

        Parameters
        ----------
        None

        Returns
        -------
        int
            Power in dBm.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:POWer:LO1?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_lo_enable_1(self, lo_enable_1: bool) -> None:
        """
        Set LO1status.

        Parameters
        ----------
        lo_enable_1 : bool
            LO1status

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['bool'])

        # SCPI call
        self._write('LO1:ENAble {}'.format(0 if lo_enable_1 == False else 1))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_lo_enable_1(self) -> bool:
        """
        Get LO 1status.

        Parameters
        ----------
        None

        Returns
        -------
        bool
            LO1status

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('LO1:ENAble?')

        return bool(int(var0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_assembler_status(self) -> bool:
        """
        Get assembler status. Refer to the assembler log to get more information regarding the assembler result.

        Parameters
        ----------
        None

        Returns
        -------
        bool
            Assembler status (False = failed, True = success).

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:ASSEMbler:SUCCess?')

        return bool(int(var0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_assembler_log(self) -> str:
        """
        Get assembler log.

        Parameters
        ----------
        None

        Returns
        -------
        str
            Assembler log.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read_bin('STATus:ASSEMbler:LOG?')

        return var0.decode('utf-8', 'ignore')

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_out_amp_offset_0(self, offset: float) -> None:
        """
        Set output amplifier offset for output 0.

        Parameters
        ----------
        offset : float
            Offset in range of 0 to 56.5 millivolt.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['float'])

        # SCPI call
        self._write('AFE:OFFSet:OUTAMP0 {}'.format(offset))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_out_amp_offset_0(self) -> float:
        """
        Get output amplifier offset for output 0.

        Parameters
        ----------
        None

        Returns
        -------
        float
            Offset in range of 0 to 56.5 millivolt.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('AFE:OFFSet:OUTAMP0?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_dac_offset_0(self, offset: float) -> None:
        """
        Set dac offset for output 0.

        Parameters
        ----------
        offset : float
            Offset in volts.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['float'])

        # SCPI call
        self._write('AFE:OFFSet:DAC0 {}'.format(offset))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_dac_offset_0(self) -> float:
        """
        Get dac offset for output 0.

        Parameters
        ----------
        None

        Returns
        -------
        float
            Offset in volts.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('AFE:OFFSet:DAC0?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_out_amp_offset_1(self, offset: float) -> None:
        """
        Set output amplifier offset for output 1.

        Parameters
        ----------
        offset : float
            Offset in range of 0 to 56.5 millivolt.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['float'])

        # SCPI call
        self._write('AFE:OFFSet:OUTAMP1 {}'.format(offset))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_out_amp_offset_1(self) -> float:
        """
        Get output amplifier offset for output 1.

        Parameters
        ----------
        None

        Returns
        -------
        float
            Offset in range of 0 to 56.5 millivolt.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('AFE:OFFSet:OUTAMP1?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_dac_offset_1(self, offset: float) -> None:
        """
        Set dac offset for output 1.

        Parameters
        ----------
        offset : float
            Offset in volts.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['float'])

        # SCPI call
        self._write('AFE:OFFSet:DAC1 {}'.format(offset))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_dac_offset_1(self) -> float:
        """
        Get dac offset for output 1.

        Parameters
        ----------
        None

        Returns
        -------
        float
            Offset in volts.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('AFE:OFFSet:DAC1?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_out_amp_offset_2(self, offset: float) -> None:
        """
        Set output amplifier offset for output 2.

        Parameters
        ----------
        offset : float
            Offset in range of 0 to 56.5 millivolt.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['float'])

        # SCPI call
        self._write('AFE:OFFSet:OUTAMP2 {}'.format(offset))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_out_amp_offset_2(self) -> float:
        """
        Get output amplifier offset for output 2.

        Parameters
        ----------
        None

        Returns
        -------
        float
            Offset in range of 0 to 56.5 millivolt.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('AFE:OFFSet:OUTAMP2?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_dac_offset_2(self, offset: float) -> None:
        """
        Set dac offset for output 2.

        Parameters
        ----------
        offset : float
            Offset in volts.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['float'])

        # SCPI call
        self._write('AFE:OFFSet:DAC2 {}'.format(offset))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_dac_offset_2(self) -> float:
        """
        Get dac offset for output 2.

        Parameters
        ----------
        None

        Returns
        -------
        float
            Offset in volts.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('AFE:OFFSet:DAC2?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_out_amp_offset_3(self, offset: float) -> None:
        """
        Set output amplifier offset for output 3.

        Parameters
        ----------
        offset : float
            Offset in range of 0 to 56.5 millivolt.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['float'])

        # SCPI call
        self._write('AFE:OFFSet:OUTAMP3 {}'.format(offset))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_out_amp_offset_3(self) -> float:
        """
        Get output amplifier offset for output 3.

        Parameters
        ----------
        None

        Returns
        -------
        float
            Offset in range of 0 to 56.5 millivolt.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('AFE:OFFSet:OUTAMP3?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_dac_offset_3(self, offset: float) -> None:
        """
        Set dac offset for output 3.

        Parameters
        ----------
        offset : float
            Offset in volts.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['float'])

        # SCPI call
        self._write('AFE:OFFSet:DAC3 {}'.format(offset))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_dac_offset_3(self) -> float:
        """
        Get dac offset for output 3.

        Parameters
        ----------
        None

        Returns
        -------
        float
            Offset in volts.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('AFE:OFFSet:DAC3?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_out_att_0(self, attenuation: int) -> None:
        """
        Set output attenuator for output 0.

        Parameters
        ----------
        attenuation : int
            Attenuation in dB in a range of 0 to 60 in steps of 2dB.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('AFE:ATT:OUT0 {}'.format(attenuation))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_out_att_0(self) -> int:
        """
        Get output attenuator for output 0.

        Parameters
        ----------
        None

        Returns
        -------
        int
            Attenuation in dB in a range of 0 to 60 in steps of 2dB.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('AFE:ATT:OUT0?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_out_att_1(self, attenuation: int) -> None:
        """
        Set output attenuator for output 1.

        Parameters
        ----------
        attenuation : int
            Attenuation in dB in a range of 0 to 60 in steps of 2dB.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('AFE:ATT:OUT1 {}'.format(attenuation))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_out_att_1(self) -> int:
        """
        Get output attenuator for output 1.

        Parameters
        ----------
        None

        Returns
        -------
        int
            Attenuation in dB in a range of 0 to 60 in steps of 2dB.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('AFE:ATT:OUT1?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_sequencer_channel_map(self, sequencer: int, channel_map: bytes) -> None:
        """
        Set channel map of the indexed sequencer. The channel map consists of list of associated output channels for the given sequencer index. If an invalid sequencer index is given or the channel map is not valid, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        channel_map : bytes
            Channel map list.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'bytes'])

        # SCPI call
        self._write_bin('SEQuencer{}:CHANnelmap '.format(sequencer), channel_map)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_sequencer_channel_map(self, sequencer: int) -> bytes:
        """
        Get channel map of the indexed sequencer. The channel map consists of list of associated output channels for the given sequencer index. If an invalid sequencer index is given or the channel map is not valid, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.

        Returns
        -------
        bytes
            Channel map list.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read_bin('SEQuencer{}:CHANnelmap?'.format(sequencer))

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_sequencer_program(self, sequencer: int, program: str) -> None:
        """
        Assemble and set the resulting Q1ASM program for the indexed sequencer. If an invalid sequencer index is given or assembling fails, an error is set in system error. The assembler status and log can be retrieved using separate commands. To set the Q1ASM program, the sequencer will be stopped.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        program : str
            Q1ASM program.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'str'])

        # SCPI call
        self._write_bin('SEQuencer{}:PROGram '.format(sequencer), program.encode('ascii'))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_sequencer_config(self, sequencer: int, config: bytes) -> None:
        """
        Set configuration of the indexed sequencer. The configuration consists of multiple parameters in a C struct format. If an invalid sequencer index is given or the configuration struct does not have the correct format, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        config : bytes
            Configuration struct.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'bytes'])

        # SCPI call
        self._write_bin('SEQuencer{}:CONFiguration '.format(sequencer), config)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_sequencer_config(self, sequencer: int) -> bytes:
        """
        Get configuration of the indexed sequencer. The configuration consists of multiple parameters in a C struct format. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.

        Returns
        -------
        bytes
            Configuration struct.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read_bin('SEQuencer{}:CONFiguration?'.format(sequencer))

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _arm_sequencer(self, sequencer: int) -> None:
        """
        Prepare the indexed sequencer to start by putting it in the armed state. If no sequencer index is given, all sequencers are armed. Any sequencer that was already running is stopped and rearmed. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('SEQuencer{}:ARM'.format(sequencer))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _start_sequencer(self, sequencer: int) -> None:
        """
        Start the indexed sequencer, thereby putting it in the running state. If an invalid sequencer index is given or the indexed sequencer was not yet armed, an error is set in system error. If no sequencer index is given, all armed sequencers are started and any sequencer not in the armed state is ignored. However, if no sequencer index is given and no sequencers are armed, and error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('SEQuencer{}:START'.format(sequencer))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _stop_sequencer(self, sequencer: int) -> None:
        """
        Stop the indexed sequencer, thereby putting it in the stopped state. If an invalid sequencer index is given, an error is set in system error. If no sequencer index is given, all sequencers are stopped.

        Parameters
        ----------
        sequencer : int
            Sequencer index.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('SEQuencer{}:STOP'.format(sequencer))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_sequencer_state(self, sequencer: int) -> str:
        """
        Get the sequencer state. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.

        Returns
        -------
        str
            Concatenated list of strings separated by the semicolon character. Status is indicated by one status string and an optional number of flags respectively ordered as:
            
            :Status:
            
             - IDLE: Sequencer waiting to be armed and started.
             - ARMED: Sequencer is armed and ready to start.
             - RUNNING: Sequencer is running.
             - Q1 STOPPED: Classical part of the sequencer has stopped, waiting for real-time part to stop.
             - STOPPED: Sequencer has completely stopped.
            
            :Flags:
            
             - DISARMED: Sequencer was disarmed.
             - FORCED STOP: Sequencer was stopped while still running.
             - SEQUENCE PROCESSOR Q1 ILLEGAL INSTRUCTION: Classical sequencer part executed an unknown instruction.
             - SEQUENCE PROCESSOR RT EXEC ILLEGAL INSTRUCTION: Real-time sequencer part executed an unknown instruction.
             - AWG WAVE PLAYBACK INDEX INVALID PATH 0: AWG path 0 tried to play an unknown waveform.
             - AWG WAVE PLAYBACK INDEX INVALID PATH 1: AWG path 1 tried to play an unknown waveform.
             - ACQ WEIGHT PLAYBACK INDEX INVALID PATH 0: Acquisition path 0 tried to play an unknown weight.
             - ACQ WEIGHT PLAYBACK INDEX INVALID PATH 1: Acquisition path 1 tried to play an unknown weight.
             - ACQ SCOPE DONE PATH 0: Scope acquisition for path 0 has finished.
             - ACQ SCOPE OUT-OF-RANGE PATH 0: Scope acquisition data for path 0 was out-of-range.
             - ACQ SCOPE OVERWRITTEN PATH 0: Scope acquisition data for path 0 was overwritten.
             - ACQ SCOPE DONE PATH 1: Scope acquisition for path 1 has finished.
             - ACQ SCOPE OUT-OF-RANGE PATH 1: Scope acquisition data for path 1 was out-of-range.
             - ACQ SCOPE OVERWRITTEN PATH 1: Scope acquisition data for path 1 was overwritten.
             - ACQ BINNING DONE: Acquisition binning completed.
             - ACQ BINNING FIFO ERROR: Acquisition binning encountered internal FIFO error.
             - ACQ BINNING COMM ERROR: Acquisition binning encountered internal communication error.
             - ACQ BINNING OUT-OF-RANGE: Acquisition binning data out-of-range.
             - ACQ INDEX INVALID: Acquisition tried to process an invalid acquisition.
             - ACQ BIN INDEX INVALID: Acquisition tried to process an invalid bin.
             - CLOCK INSTABILITY: Clock source instability occurred.
            

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SEQuencer{}:STATE?'.format(sequencer))

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _add_awg_waveform(self, sequencer: int, name: str, size: int, is_integer: bool) -> None:
        """
        Add new waveform to waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given or if the waveform causes the waveform memory limit to be exceeded, an error is set in system error. The waveform names 'all' and 'ALL' are reserved and adding those will also result in an error being set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.
        size : int
            Waveform length in number of samples.
        is_integer : bool
            Waveform is provided as integers (False = floats, True = integers).

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'str', 'int', 'bool'])

        # SCPI call
        self._write('SEQuencer{}:AWG:WLISt:WAVeform:NEW "{}",{},{}'.format(sequencer, name, size, 0 if is_integer == False else 1))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _delete_awg_waveform(self, sequencer: int, name: str) -> None:
        """
        Delete waveform from waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given or if a non-existing waveform name is given, an error is set in system error. The waveform names 'all' and 'ALL' are reserved and deleting those will cause a purge of the entire waveform list.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'str'])

        # SCPI call
        self._write('SEQuencer{}:AWG:WLISt:WAVeform:DELete "{}"'.format(sequencer, name))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_awg_waveform_data(self, sequencer: int, name: str, waveform: List[float]) -> None:
        """
        Set waveform data of waveform in waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given or if a non-existing waveform name is given, an error is set in system error. If the waveform size does not match the size specified while adding the waveform or if the waveform samples are out-of-range, an error is set in the system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.
        waveform : list[float]
            List of floats in the range of 1.0 to -1.0 representing the waveform samples.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'str', 'float'])

        # SCPI call
        self._write_bin('SEQuencer{}:AWG:WLISt:WAVeform:DATA "{}",'.format(sequencer, name), struct.pack('f'*len(waveform), *waveform))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_awg_waveform_data(self, sequencer: int, name: str, start: int, size: int) -> List[float]:
        """
        Get waveform data of waveform in waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given or if a non-existing waveform name is given, an error is set in system error. The start and size arguments can be used to return a section of the waveform. If the start argument is not given, zero is used as the default start sample. If the size argument is not given, all samples starting from the start sample are returned.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.
        start : int
            First sample within the waveform to be returned.
        size : int
            Number of samples starting from the start sample to be returned.

        Returns
        -------
        list[float]
            List of floats in the range of 1.0 to -1.0 representing the waveform samples.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'str', 'int', 'int'])

        # SCPI call
        var0 = self._read_bin('SEQuencer{}:AWG:WLISt:WAVeform:DATA? "{}",{},{}'.format(sequencer, name, start, size))

        return struct.unpack('f'*int(len(var0)/4), var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_awg_waveform_index(self, sequencer: int, name: str, index: int) -> None:
        """
        Set waveform index of waveform in waveform list of indexed sequencer's AWG path. The index is used by the sequencer Q1ASM program to refer to the waveform. If an invalid sequencer or waveform index is given or if a non-existing waveform name is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.
        index : int
            Waveform index of the waveform in the waveform list.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'str', 'int'])

        # SCPI call
        self._write('SEQuencer{}:AWG:WLISt:WAVeform:INDex "{}",{}'.format(sequencer, name, index))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_awg_waveform_index(self, sequencer: int, name: str) -> int:
        """
        Get waveform index of waveform in waveform list of indexed sequencer's AWG path. The index is used by the sequencer Q1ASM program to refer to the waveform. If an invalid sequencer index is given or if a non-existing waveform name is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.

        Returns
        -------
        int
            Waveform index of the waveform in the waveform list.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'str'])

        # SCPI call
        var0 = self._read('SEQuencer{}:AWG:WLISt:WAVeform:INDex? "{}"'.format(sequencer, name))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_awg_waveform_length(self, sequencer: int, name: str) -> int:
        """
        Get length of waveform in waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given or if a non-existing waveform name is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.

        Returns
        -------
        int
            Waveform length in number of samples.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'str'])

        # SCPI call
        var0 = self._read('SEQuencer{}:AWG:WLISt:WAVeform:LENGth? "{}"'.format(sequencer, name))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_awg_waveform_name(self, sequencer: int, index: int) -> str:
        """
        Get name of waveform in waveform list of indexed sequencer's AWG path based on its waveform index. If an invalid sequencer index is given or if a non-existing waveform index is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        index : int
            Waveform index.

        Returns
        -------
        str
            Waveform name.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        var0 = self._read('SEQuencer{}:AWG:WLISt:WAVeform:NAME? {}'.format(sequencer, index))

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_awg_num_waveforms(self, sequencer: int) -> int:
        """
        Get number of waveforms in waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.

        Returns
        -------
        int
            Number of waveforms.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SEQuencer{}:AWG:WLISt:SIZE?'.format(sequencer))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_awg_waveforms(self, sequencer: int) -> List[bytes]:
        """
        Get all waveforms in waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.

        Returns
        -------
        list[bytes]
            Dictionary with waveforms.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SEQuencer{}:AWG:WLISt?'.format(sequencer))

        return var0

    # ---------------------------------------------------------------------
    def _check_in_type(self, in_arg_dict: Dict, in_type_list: List) -> None:
        """
        Checks input argument types against reference types.

        Parameters
        ----------
        in_arg_dict : dict
            Dictionary with input arguments created by locals().
        in_type_list : list
            List of reference input argument types.

        Returns
        ----------

        Raises
        ----------
        TypeError
            Input argument type mismatch.
        """

        if self._debug <= 1:
            del in_arg_dict['self']
            in_val_list = [in_arg_dict[name] for name in in_arg_dict]
            for i, (in_val, in_type) in enumerate(zip(in_val_list, in_type_list)):
                if (type(in_val).__name__ == "list" or
                   type(in_val).__name__ == "ndarray"):
                    if len(in_val) > 0:
                        in_val = in_val[0]
                    else:
                        raise TypeError(
                            "Unexpected type for input argument {}, expected {} but got empty {}.".format(i, in_type, str(type(in_val).__name__))
                        )
                if type(in_val).__name__[:len(in_type)] != in_type:
                    raise TypeError(
                        "Unexpected type for input argument {}, expected {} but got {}.".format(i, in_type, str(type(in_val).__name__))
                    )

    # -------------------------------------------------------------------------
    def _check_error_queue(self, err: Optional[Exception]=None) -> None:
        """
        Check system error for errors. Empties and prints the complete error
        queue.

        Parameters
        ----------
        err : Optional[Exception]
            Exception to reraise.

        Returns
        ----------

        Raises
        ----------
        Exception
            An exception was passed as input argument.
        RuntimeError:
            An error was found in system error.
        """

        if self._debug <= 1:
            errors = [str(err)] if err is not None else []
            while int(self._read('SYSTem:ERRor:COUNt?')) != 0:
                errors.append(','.join(self._read('SYSTem:ERRor:NEXT?').split(',')[1:]))

            if len(errors) > 0:
                if err is not None:
                    err_type = type(err)
                else:
                    err_type = RuntimeError
                raise err_type('\n'.join(errors)).with_traceback(sys.exc_info()[2]) from None
