# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['portabletext_html']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'portabletext-html',
    'version': '1.0.0',
    'description': "HTML renderer for Sanity's Portable Text format",
    'long_description': '[![pypi](https://img.shields.io/pypi/v/portabletext-html.svg)](https://pypi.org/project/portabletext-html/)\n[![test](https://github.com/otovo/python-portabletext-html/actions/workflows/test.yml/badge.svg)](https://github.com/otovo/python-portabletext-html/actions/workflows/test.yml)\n[![code coverage](https://codecov.io/gh/otovo/python-portabletext-html/branch/main/graph/badge.svg)](https://codecov.io/gh/otovo/python-portabletext-html)\n[![supported python versions](https://img.shields.io/badge/python-3.7%2B-blue)](https://pypi.org/project/python-portabletext-html/)\n\n# Portable Text HTML Renderer for Python\n\nThis package generates HTML from [Portable Text](https://github.com/portabletext/portabletext).\n\nFor the most part, it mirrors [Sanity\'s](https://www.sanity.io/) own [block-content-to-html](https://www.npmjs.com/package/%40sanity/block-content-to-html) NPM library.\n\n## Installation\n\n```\npip install portabletext-html\n```\n\n## Usage\n\nInstantiate the `PortableTextRenderer` class with your content and call the `render` method.\n\nThe following content\n\n```python\nfrom portabletext_html import PortableTextRenderer\n\nrenderer = PortableTextRenderer({\n    "_key": "R5FvMrjo",\n    "_type": "block",\n    "children": [\n        {"_key": "cZUQGmh4", "_type": "span", "marks": ["strong"], "text": "A word of"},\n        {"_key": "toaiCqIK", "_type": "span", "marks": ["strong"], "text": " warning;"},\n        {"_key": "gaZingsA", "_type": "span", "marks": [], "text": " Sanity is addictive."}\n    ],\n    "markDefs": [],\n    "style": "normal"\n})\nrenderer.render()\n```\n\nGenerates this HTML\n```html\n<p><strong>A word of warning;</strong> Sanity is addictive.</p>\n```\n\n### Supported types\n\nThe `block` and `span` types are supported out of the box.\n\n### Custom types\n\nBeyond the built-in types, you have the freedom to provide\nyour own serializers to render any custom `_type` the way you\nwould like to.\n\nTo illustrate, if you passed this data to the renderer class:\n\n```python\nfrom portabletext_html import PortableTextRenderer\n\nrenderer = PortableTextRenderer({\n    "_type": "block",\n    "_key": "foo",\n    "style": "normal",\n    "children": [\n        {\n            "_type": "span",\n            "text": "Press, "\n        },\n        {\n            "_type": "button",\n            "text": "here"\n        },\n        {\n            "_type": "span",\n            "text": ", now!"\n        }\n    ]\n})\nrenderer.render()\n```\n\nThe renderer would actually throw an error here, since `button`\ndoes not have a corresponding built-in type serializer by default.\n\nTo render this text you must provide your own serializer, like this:\n\n```python\nfrom portabletext_html import PortableTextRenderer\n\n\ndef button_serializer(node: dict, context: Optional[Block], list_item: bool):\n    return f\'<button>{node["text"]}</button>\'\n\n\nrenderer = PortableTextRenderer(\n    ...,\n    custom_serializers={\'button\': button_serializer}\n)\noutput = renderer.render()\n```\n\nWith the custom serializer provided, the renderer would now successfully\noutput the following HTML:\n\n```html\n<p>Press <button>here</button>, now!</p>\n```\n\n### Supported mark definitions\n\nThe package provides several built-in marker definitions and styles:\n\n**decorator marker definitions**\n\n- `em`\n- `strong`\n- `code`\n- `underline`\n- `strike-through`\n\n**annotation marker definitions**\n\n- `link`\n- `comment`\n\n### Custom mark definitions\n\nLike with custom type serializers, additional serializers for\nmarker definitions and styles can be passed in like this:\n\n```python\nfrom portabletext_html import PortableTextRenderer\n\nrenderer = PortableTextRenderer(\n    ...,\n    custom_marker_definitions={\'em\': ComicSansEmphasis}\n)\nrenderer.render()\n```\n\nThe primary difference between a type serializer and a mark definition serializer\nis that the latter uses a class structure, and has three required methods.\n\nHere\'s an example of a custom style, adding an extra font\nto the built-in equivalent serializer:\n\n```python\nfrom portabletext_html.marker_definitions import MarkerDefinition\n\n\nclass ComicSansEmphasis(MarkerDefinition):\n    tag = \'em\'\n\n    @classmethod\n    def render_prefix(cls, span: Span, marker: str, context: Block) -> str:\n        return f\'<{cls.tag} style="font-family: "Comic Sans MS", "Comic Sans", cursive;">\'\n\n    @classmethod\n    def render_suffix(cls, span: Span, marker: str, context: Block) -> str:\n        return f\'</{cls.tag}>\'\n\n    @classmethod\n    def render(cls, span: Span, marker: str, context: Block) -> str:\n        result = cls.render_prefix(span, marker, context)\n        result += str(span.text)\n        result += cls.render_suffix(span, marker, context)\n        return result\n```\n\nSince the `render_suffix` and `render` methods here are actually identical to the base class,\nthey do not need to be specified, and the whole example can be reduced to:\n\n```python\nfrom portabletext_html.marker_definitions import MarkerDefinition  # base\nfrom portabletext_html import PortableTextRenderer\n\n\nclass ComicSansEmphasis(MarkerDefinition):\n    tag = \'em\'\n\n    @classmethod\n    def render_prefix(cls, span: Span, marker: str, context: Block) -> str:\n        return f\'<{cls.tag} style="font-family: "Comic Sans MS", "Comic Sans", cursive;">\'\n\n\nrenderer = PortableTextRenderer(\n    ...,\n    custom_marker_definitions={\'em\': ComicSansEmphasis}\n)\nrenderer.render()\n```\n\n\n### Supported styles\n\nBlocks can optionally define a `style` tag. These styles are supported:\n\n- `h1`\n- `h2`\n- `h3`\n- `h4`\n- `h5`\n- `h6`\n- `blockquote`\n- `normal`\n\n## Missing features\n\nFor anyone interested, we would be happy to see a\ndefault built-in serializer for the `image` type added.\nIn the meantime, users should be able to serialize image types by passing a custom serializer.\n\n## Contributing\n\nContributions are always appreciated 👏\n\nFor details, see the [CONTRIBUTING.md](https://github.com/otovo/python-portabletext-html/blob/main/CONTRIBUTING.md).\n',
    'author': 'Kristian Klette',
    'author_email': 'klette@otovo.com',
    'maintainer': 'Sondre Lillebø Gundersen',
    'maintainer_email': 'sondrelg@live.no',
    'url': 'https://github.com/otovo/python-sanity-html',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
