"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisEventSource = void 0;
const cdk = require("@aws-cdk/core");
const stream_1 = require("./stream");
/**
 * Use an Amazon Kinesis stream as an event source for AWS Lambda.
 */
class KinesisEventSource extends stream_1.StreamEventSource {
    constructor(stream, props) {
        super(props);
        this.stream = stream;
        this._eventSourceMappingId = undefined;
        this.props.batchSize !== undefined && cdk.withResolved(this.props.batchSize, batchSize => {
            if (batchSize < 1 || batchSize > 10000) {
                throw new Error(`Maximum batch size must be between 1 and 10000 inclusive (given ${this.props.batchSize})`);
            }
        });
    }
    bind(target) {
        const eventSourceMapping = target.addEventSourceMapping(`KinesisEventSource:${this.stream.node.uniqueId}`, this.enrichMappingOptions({ eventSourceArn: this.stream.streamArn }));
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.stream.grantRead(target);
        // The `grantRead` API provides all the permissions recommended by the Kinesis team for reading a stream.
        // `DescribeStream` permissions are not required to read a stream as it's covered by the `DescribeStreamSummary`
        // and `SubscribeToShard` APIs.
        // The Lambda::EventSourceMapping resource validates against the `DescribeStream` permission. So we add it explicitly.
        // FIXME This permission can be removed when the event source mapping resource drops it from validation.
        this.stream.grant(target, 'kinesis:DescribeStream');
    }
    /**
     * The identifier for this EventSourceMapping
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('KinesisEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.KinesisEventSource = KinesisEventSource;
//# sourceMappingURL=data:application/json;base64,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