"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DynamoEventSource = void 0;
const stream_1 = require("./stream");
/**
 * Use an Amazon DynamoDB stream as an event source for AWS Lambda.
 */
class DynamoEventSource extends stream_1.StreamEventSource {
    constructor(table, props) {
        super(props);
        this.table = table;
        this._eventSourceMappingId = undefined;
        if (this.props.batchSize !== undefined && (this.props.batchSize < 1 || this.props.batchSize > 1000)) {
            throw new Error(`Maximum batch size must be between 1 and 1000 inclusive (given ${this.props.batchSize})`);
        }
    }
    bind(target) {
        if (!this.table.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.table.node.path}`);
        }
        const eventSourceMapping = target.addEventSourceMapping(`DynamoDBEventSource:${this.table.node.uniqueId}`, this.enrichMappingOptions({ eventSourceArn: this.table.tableStreamArn }));
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.table.grantStreamRead(target);
    }
    /**
     * The identifier for this EventSourceMapping
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('DynamoEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.DynamoEventSource = DynamoEventSource;
//# sourceMappingURL=data:application/json;base64,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