# vim: ft=python fileencoding=utf-8 sts=4 sw=4 et:

# Copyright 2014-2021 Florian Bruhin (The Compiler) <mail@qutebrowser.org>
#
# This file is part of qutebrowser.
#
# qutebrowser is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# qutebrowser is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with qutebrowser.  If not, see <https://www.gnu.org/licenses/>.

"""Get arguments to pass to Qt."""

import os
import sys
import argparse
from typing import Any, Dict, Iterator, List, Optional, Sequence, Tuple

from qutebrowser.config import config
from qutebrowser.misc import objects
from qutebrowser.utils import usertypes, qtutils, utils


_ENABLE_FEATURES = '--enable-features='
_DISABLE_FEATURES = '--disable-features='


def qt_args(namespace: argparse.Namespace) -> List[str]:
    """Get the Qt QApplication arguments based on an argparse namespace.

    Args:
        namespace: The argparse namespace.

    Return:
        The argv list to be passed to Qt.
    """
    argv = [sys.argv[0]]

    if namespace.qt_flag is not None:
        argv += ['--' + flag[0] for flag in namespace.qt_flag]

    if namespace.qt_arg is not None:
        for name, value in namespace.qt_arg:
            argv += ['--' + name, value]

    argv += ['--' + arg for arg in config.val.qt.args]

    if objects.backend != usertypes.Backend.QtWebEngine:
        assert objects.backend == usertypes.Backend.QtWebKit, objects.backend
        return argv

    feature_prefixes = (_ENABLE_FEATURES, _DISABLE_FEATURES)
    feature_flags = [flag for flag in argv if flag.startswith(feature_prefixes)]
    argv = [flag for flag in argv if not flag.startswith(feature_prefixes)]
    argv += list(_qtwebengine_args(namespace, feature_flags))

    return argv


def _qtwebengine_features(
        feature_flags: Sequence[str],
) -> Tuple[Sequence[str], Sequence[str]]:
    """Get a tuple of --enable-features/--disable-features flags for QtWebEngine.

    Args:
        feature_flags: Existing flags passed via the commandline.
    """
    enabled_features = []
    disabled_features = []

    for flag in feature_flags:
        if flag.startswith(_ENABLE_FEATURES):
            flag = flag[len(_ENABLE_FEATURES):]
            enabled_features += flag.split(',')
        elif flag.startswith(_DISABLE_FEATURES):
            flag = flag[len(_DISABLE_FEATURES):]
            disabled_features += flag.split(',')
        else:
            raise utils.Unreachable(flag)

    if qtutils.version_check('5.15', compiled=False) and utils.is_linux:
        # Enable WebRTC PipeWire for screen capturing on Wayland.
        #
        # This is disabled in Chromium by default because of the "dialog hell":
        # https://bugs.chromium.org/p/chromium/issues/detail?id=682122#c50
        # https://github.com/flatpak/xdg-desktop-portal-gtk/issues/204
        #
        # However, we don't have Chromium's confirmation dialog in qutebrowser,
        # so we should only get qutebrowser's permission dialog.
        #
        # In theory this would be supported with Qt 5.13 already, but
        # QtWebEngine only started picking up PipeWire correctly with Qt
        # 5.15.1. Checking for 5.15 here to pick up Archlinux' patched package
        # as well.
        #
        # This only should be enabled on Wayland, but it's too early to check
        # that, as we don't have a QApplication available at this point. Thus,
        # just turn it on unconditionally on Linux, which shouldn't hurt.
        enabled_features.append('WebRTCPipeWireCapturer')

    if not utils.is_mac:
        # Enable overlay scrollbars.
        #
        # There are two additional flags in Chromium:
        #
        # - OverlayScrollbarFlashAfterAnyScrollUpdate
        # - OverlayScrollbarFlashWhenMouseEnter
        #
        # We don't expose/activate those, but the changes they introduce are
        # quite subtle: The former seems to show the scrollbar handle even if
        # there was a 0px scroll (though no idea how that can happen...). The
        # latter flashes *all* scrollbars when a scrollable area was entered,
        # which doesn't seem to make much sense.
        if config.val.scrolling.bar == 'overlay':
            enabled_features.append('OverlayScrollbar')

    if (qtutils.version_check('5.14', compiled=False) and
            config.val.content.headers.referer == 'same-domain'):
        # Handling of reduced-referrer-granularity in Chromium 76+
        # https://chromium-review.googlesource.com/c/chromium/src/+/1572699
        #
        # Note that this is removed entirely (and apparently the default) starting with
        # Chromium 89 (Qt 5.15.x or 6.x):
        # https://chromium-review.googlesource.com/c/chromium/src/+/2545444
        enabled_features.append('ReducedReferrerGranularity')

    if qtutils.version_check('5.15.2', compiled=False, exact=True):
        # WORKAROUND for https://bugreports.qt.io/browse/QTBUG-89740
        # FIXME Not needed anymore with QtWebEngne 5.15.3 (or Qt 6), but we'll probably
        # have no way to detect that...
        disabled_features.append('InstalledApp')

    return (enabled_features, disabled_features)


def _qtwebengine_args(
        namespace: argparse.Namespace,
        feature_flags: Sequence[str],
) -> Iterator[str]:
    """Get the QtWebEngine arguments to use based on the config."""
    is_qt_514 = (qtutils.version_check('5.14', compiled=False) and
                 not qtutils.version_check('5.15', compiled=False))

    if is_qt_514:
        # WORKAROUND for https://bugreports.qt.io/browse/QTBUG-82105
        yield '--disable-shared-workers'

    # WORKAROUND equivalent to
    # https://codereview.qt-project.org/c/qt/qtwebengine/+/256786
    # also see:
    # https://codereview.qt-project.org/c/qt/qtwebengine-chromium/+/265753
    if qtutils.version_check('5.12.3', compiled=False):
        if 'stack' in namespace.debug_flags:
            # Only actually available in Qt 5.12.5, but let's save another
            # check, as passing the option won't hurt.
            yield '--enable-in-process-stack-traces'
    else:
        if 'stack' not in namespace.debug_flags:
            yield '--disable-in-process-stack-traces'

    if 'chromium' in namespace.debug_flags:
        yield '--enable-logging'
        yield '--v=1'

    if 'wait-renderer-process' in namespace.debug_flags:
        yield '--renderer-startup-dialog'

    from qutebrowser.browser.webengine import darkmode
    blink_settings = list(darkmode.settings())
    if blink_settings:
        yield '--blink-settings=' + ','.join(f'{k}={v}' for k, v in blink_settings)

    enabled_features, disabled_features = _qtwebengine_features(feature_flags)
    if enabled_features:
        yield _ENABLE_FEATURES + ','.join(enabled_features)
    if disabled_features:
        yield _DISABLE_FEATURES + ','.join(disabled_features)

    yield from _qtwebengine_settings_args()


def _qtwebengine_settings_args() -> Iterator[str]:
    settings: Dict[str, Dict[Any, Optional[str]]] = {
        'qt.force_software_rendering': {
            'software-opengl': None,
            'qt-quick': None,
            'chromium': '--disable-gpu',
            'none': None,
        },
        'content.canvas_reading': {
            True: None,
            False: '--disable-reading-from-canvas',
        },
        'content.webrtc_ip_handling_policy': {
            'all-interfaces': None,
            'default-public-and-private-interfaces':
                '--force-webrtc-ip-handling-policy='
                'default_public_and_private_interfaces',
            'default-public-interface-only':
                '--force-webrtc-ip-handling-policy='
                'default_public_interface_only',
            'disable-non-proxied-udp':
                '--force-webrtc-ip-handling-policy='
                'disable_non_proxied_udp',
        },
        'qt.process_model': {
            'process-per-site-instance': None,
            'process-per-site': '--process-per-site',
            'single-process': '--single-process',
        },
        'qt.low_end_device_mode': {
            'auto': None,
            'always': '--enable-low-end-device-mode',
            'never': '--disable-low-end-device-mode',
        },
        'content.headers.referer': {
            'always': None,
        }
    }

    if (qtutils.version_check('5.14', compiled=False) and
            not qtutils.version_check('5.15.2', compiled=False)):
        # In Qt 5.14 to 5.15.1, `--force-dark-mode` is used to set the
        # preferred colorscheme. In Qt 5.15.2, this is handled by a
        # blink-setting in browser/webengine/darkmode.py instead.
        settings['colors.webpage.prefers_color_scheme_dark'] = {
            True: '--force-dark-mode',
            False: None,
        }

    referrer_setting = settings['content.headers.referer']
    if qtutils.version_check('5.14', compiled=False):
        # Starting with Qt 5.14, this is handled via --enable-features
        referrer_setting['same-domain'] = None
    else:
        referrer_setting['same-domain'] = '--reduced-referrer-granularity'

    can_override_referer = (
        qtutils.version_check('5.12.4', compiled=False) and
        not qtutils.version_check('5.13.0', compiled=False, exact=True)
    )
    # WORKAROUND for https://bugreports.qt.io/browse/QTBUG-60203
    referrer_setting['never'] = None if can_override_referer else '--no-referrers'

    for setting, args in sorted(settings.items()):
        arg = args[config.instance.get(setting)]
        if arg is not None:
            yield arg


def init_envvars() -> None:
    """Initialize environment variables which need to be set early."""
    if objects.backend == usertypes.Backend.QtWebEngine:
        software_rendering = config.val.qt.force_software_rendering
        if software_rendering == 'software-opengl':
            os.environ['QT_XCB_FORCE_SOFTWARE_OPENGL'] = '1'
        elif software_rendering == 'qt-quick':
            os.environ['QT_QUICK_BACKEND'] = 'software'
        elif software_rendering == 'chromium':
            os.environ['QT_WEBENGINE_DISABLE_NOUVEAU_WORKAROUND'] = '1'
    else:
        assert objects.backend == usertypes.Backend.QtWebKit, objects.backend

    if config.val.qt.force_platform is not None:
        os.environ['QT_QPA_PLATFORM'] = config.val.qt.force_platform
    if config.val.qt.force_platformtheme is not None:
        os.environ['QT_QPA_PLATFORMTHEME'] = config.val.qt.force_platformtheme

    if config.val.window.hide_decoration:
        os.environ['QT_WAYLAND_DISABLE_WINDOWDECORATION'] = '1'

    if config.val.qt.highdpi:
        env_var = ('QT_ENABLE_HIGHDPI_SCALING'
                   if qtutils.version_check('5.14', compiled=False)
                   else 'QT_AUTO_SCREEN_SCALE_FACTOR')
        os.environ[env_var] = '1'

    for var, val in config.val.qt.environ.items():
        if val is None and var in os.environ:
            del os.environ[var]
        elif val is not None:
            os.environ[var] = val
