"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const subs = require("../lib");
// tslint:disable:object-literal-key-quotes
let stack;
let topic;
beforeEach(() => {
    stack = new core_1.Stack();
    topic = new sns.Topic(stack, 'MyTopic', {
        topicName: 'topicName',
        displayName: 'displayName'
    });
});
test('url subscription', () => {
    topic.addSubscription(new subs.UrlSubscription('https://foobar.com/'));
    expect(stack).toMatchTemplate({
        'Resources': {
            'MyTopic86869434': {
                'Type': 'AWS::SNS::Topic',
                'Properties': {
                    'DisplayName': 'displayName',
                    'TopicName': 'topicName'
                }
            },
            'MyTopichttpsfoobarcomDEA92AB5': {
                'Type': 'AWS::SNS::Subscription',
                'Properties': {
                    'Endpoint': 'https://foobar.com/',
                    'Protocol': 'https',
                    'TopicArn': {
                        'Ref': 'MyTopic86869434'
                    }
                }
            }
        }
    });
});
test('url subscription (with raw delivery)', () => {
    topic.addSubscription(new subs.UrlSubscription('https://foobar.com/', {
        rawMessageDelivery: true
    }));
    expect(stack).toMatchTemplate({
        'Resources': {
            'MyTopic86869434': {
                'Type': 'AWS::SNS::Topic',
                'Properties': {
                    'DisplayName': 'displayName',
                    'TopicName': 'topicName'
                }
            },
            'MyTopichttpsfoobarcomDEA92AB5': {
                'Type': 'AWS::SNS::Subscription',
                'Properties': {
                    'Endpoint': 'https://foobar.com/',
                    'Protocol': 'https',
                    'TopicArn': { 'Ref': 'MyTopic86869434' },
                    'RawMessageDelivery': true
                }
            }
        }
    });
});
test('url subscription (unresolved url with protocol)', () => {
    const urlToken = core_1.Token.asString({ Ref: 'my-url-1' });
    topic.addSubscription(new subs.UrlSubscription(urlToken, { protocol: sns.SubscriptionProtocol.HTTPS }));
    expect(stack).toMatchTemplate({
        'Resources': {
            'MyTopic86869434': {
                'Type': 'AWS::SNS::Topic',
                'Properties': {
                    'DisplayName': 'displayName',
                    'TopicName': 'topicName'
                }
            },
            'MyTopicTokenSubscription141DD1BE2': {
                'Type': 'AWS::SNS::Subscription',
                'Properties': {
                    'Endpoint': {
                        'Ref': 'my-url-1'
                    },
                    'Protocol': 'https',
                    'TopicArn': { 'Ref': 'MyTopic86869434' },
                }
            }
        }
    });
});
test('url subscription (double unresolved url with protocol)', () => {
    const urlToken1 = core_1.Token.asString({ Ref: 'my-url-1' });
    const urlToken2 = core_1.Token.asString({ Ref: 'my-url-2' });
    topic.addSubscription(new subs.UrlSubscription(urlToken1, { protocol: sns.SubscriptionProtocol.HTTPS }));
    topic.addSubscription(new subs.UrlSubscription(urlToken2, { protocol: sns.SubscriptionProtocol.HTTPS }));
    expect(stack).toMatchTemplate({
        'Resources': {
            'MyTopic86869434': {
                'Type': 'AWS::SNS::Topic',
                'Properties': {
                    'DisplayName': 'displayName',
                    'TopicName': 'topicName'
                }
            },
            'MyTopicTokenSubscription141DD1BE2': {
                'Type': 'AWS::SNS::Subscription',
                'Properties': {
                    'Endpoint': {
                        'Ref': 'my-url-1'
                    },
                    'Protocol': 'https',
                    'TopicArn': { 'Ref': 'MyTopic86869434' },
                }
            },
            'MyTopicTokenSubscription293BFE3F9': {
                'Type': 'AWS::SNS::Subscription',
                'Properties': {
                    'Endpoint': {
                        'Ref': 'my-url-2'
                    },
                    'Protocol': 'https',
                    'TopicArn': { 'Ref': 'MyTopic86869434' },
                }
            }
        }
    });
});
test('url subscription (unknown protocol)', () => {
    expect(() => topic.addSubscription(new subs.UrlSubscription('some-protocol://foobar.com/')))
        .toThrowError(/URL must start with either http:\/\/ or https:\/\//);
});
test('url subscription (unresolved url without protocol)', () => {
    const urlToken = core_1.Token.asString({ Ref: 'my-url-1' });
    expect(() => topic.addSubscription(new subs.UrlSubscription(urlToken)))
        .toThrowError(/Must provide protocol if url is unresolved/);
});
test('queue subscription', () => {
    const queue = new sqs.Queue(stack, 'MyQueue');
    topic.addSubscription(new subs.SqsSubscription(queue));
    expect(stack).toMatchTemplate({
        'Resources': {
            'MyTopic86869434': {
                'Type': 'AWS::SNS::Topic',
                'Properties': {
                    'DisplayName': 'displayName',
                    'TopicName': 'topicName'
                }
            },
            'MyQueueE6CA6235': {
                'Type': 'AWS::SQS::Queue'
            },
            'MyQueuePolicy6BBEDDAC': {
                'Type': 'AWS::SQS::QueuePolicy',
                'Properties': {
                    'PolicyDocument': {
                        'Statement': [
                            {
                                'Action': 'sqs:SendMessage',
                                'Condition': {
                                    'ArnEquals': {
                                        'aws:SourceArn': {
                                            'Ref': 'MyTopic86869434'
                                        }
                                    }
                                },
                                'Effect': 'Allow',
                                'Principal': {
                                    'Service': 'sns.amazonaws.com'
                                },
                                'Resource': {
                                    'Fn::GetAtt': [
                                        'MyQueueE6CA6235',
                                        'Arn'
                                    ]
                                }
                            }
                        ],
                        'Version': '2012-10-17'
                    },
                    'Queues': [
                        {
                            'Ref': 'MyQueueE6CA6235'
                        }
                    ]
                }
            },
            'MyQueueMyTopic9B00631B': {
                'Type': 'AWS::SNS::Subscription',
                'Properties': {
                    'Protocol': 'sqs',
                    'TopicArn': {
                        'Ref': 'MyTopic86869434'
                    },
                    'Endpoint': {
                        'Fn::GetAtt': [
                            'MyQueueE6CA6235',
                            'Arn'
                        ]
                    }
                }
            }
        }
    });
});
test('queue subscription (with raw delivery)', () => {
    const queue = new sqs.Queue(stack, 'MyQueue');
    topic.addSubscription(new subs.SqsSubscription(queue, { rawMessageDelivery: true }));
    expect(stack).toHaveResource('AWS::SNS::Subscription', {
        'Endpoint': {
            'Fn::GetAtt': [
                'MyQueueE6CA6235',
                'Arn'
            ]
        },
        'Protocol': 'sqs',
        'TopicArn': {
            'Ref': 'MyTopic86869434'
        },
        'RawMessageDelivery': true
    });
});
test('lambda subscription', () => {
    const fction = new lambda.Function(stack, 'MyFunc', {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler',
        code: lambda.Code.fromInline('exports.handler = function(e, c, cb) { return cb() }')
    });
    topic.addSubscription(new subs.LambdaSubscription(fction));
    expect(stack).toMatchTemplate({
        'Resources': {
            'MyTopic86869434': {
                'Type': 'AWS::SNS::Topic',
                'Properties': {
                    'DisplayName': 'displayName',
                    'TopicName': 'topicName'
                }
            },
            'MyFuncServiceRole54065130': {
                'Type': 'AWS::IAM::Role',
                'Properties': {
                    'AssumeRolePolicyDocument': {
                        'Statement': [
                            {
                                'Action': 'sts:AssumeRole',
                                'Effect': 'Allow',
                                'Principal': {
                                    'Service': 'lambda.amazonaws.com'
                                }
                            }
                        ],
                        'Version': '2012-10-17'
                    },
                    'ManagedPolicyArns': [
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        'Ref': 'AWS::Partition'
                                    },
                                    ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole'
                                ]
                            ]
                        }
                    ]
                }
            },
            'MyFunc8A243A2C': {
                'Type': 'AWS::Lambda::Function',
                'Properties': {
                    'Code': {
                        'ZipFile': 'exports.handler = function(e, c, cb) { return cb() }'
                    },
                    'Handler': 'index.handler',
                    'Role': {
                        'Fn::GetAtt': [
                            'MyFuncServiceRole54065130',
                            'Arn'
                        ]
                    },
                    'Runtime': 'nodejs10.x'
                },
                'DependsOn': [
                    'MyFuncServiceRole54065130'
                ]
            },
            'MyFuncAllowInvokeMyTopicDD0A15B8': {
                'Type': 'AWS::Lambda::Permission',
                'Properties': {
                    'Action': 'lambda:InvokeFunction',
                    'FunctionName': {
                        'Fn::GetAtt': [
                            'MyFunc8A243A2C',
                            'Arn'
                        ]
                    },
                    'Principal': 'sns.amazonaws.com',
                    'SourceArn': {
                        'Ref': 'MyTopic86869434'
                    }
                }
            },
            'MyFuncMyTopic93B6FB00': {
                'Type': 'AWS::SNS::Subscription',
                'Properties': {
                    'Protocol': 'lambda',
                    'TopicArn': {
                        'Ref': 'MyTopic86869434'
                    },
                    'Endpoint': {
                        'Fn::GetAtt': [
                            'MyFunc8A243A2C',
                            'Arn'
                        ]
                    }
                }
            }
        }
    });
});
test('email subscription', () => {
    topic.addSubscription(new subs.EmailSubscription('foo@bar.com'));
    expect(stack).toMatchTemplate({
        'Resources': {
            'MyTopic86869434': {
                'Type': 'AWS::SNS::Topic',
                'Properties': {
                    'DisplayName': 'displayName',
                    'TopicName': 'topicName'
                }
            },
            'MyTopicfoobarcomA344CADA': {
                'Type': 'AWS::SNS::Subscription',
                'Properties': {
                    'Endpoint': 'foo@bar.com',
                    'Protocol': 'email',
                    'TopicArn': {
                        'Ref': 'MyTopic86869434'
                    }
                }
            }
        }
    });
});
test('email subscription with unresolved', () => {
    const emailToken = core_1.Token.asString({ Ref: 'my-email-1' });
    topic.addSubscription(new subs.EmailSubscription(emailToken));
    expect(stack).toMatchTemplate({
        'Resources': {
            'MyTopic86869434': {
                'Type': 'AWS::SNS::Topic',
                'Properties': {
                    'DisplayName': 'displayName',
                    'TopicName': 'topicName'
                }
            },
            'MyTopicTokenSubscription141DD1BE2': {
                'Type': 'AWS::SNS::Subscription',
                'Properties': {
                    'Endpoint': {
                        'Ref': 'my-email-1'
                    },
                    'Protocol': 'email',
                    'TopicArn': {
                        'Ref': 'MyTopic86869434'
                    }
                }
            }
        }
    });
});
test('email and url subscriptions with unresolved', () => {
    const emailToken = core_1.Token.asString({ Ref: 'my-email-1' });
    const urlToken = core_1.Token.asString({ Ref: 'my-url-1' });
    topic.addSubscription(new subs.EmailSubscription(emailToken));
    topic.addSubscription(new subs.UrlSubscription(urlToken, { protocol: sns.SubscriptionProtocol.HTTPS }));
    expect(stack).toMatchTemplate({
        'Resources': {
            'MyTopic86869434': {
                'Type': 'AWS::SNS::Topic',
                'Properties': {
                    'DisplayName': 'displayName',
                    'TopicName': 'topicName'
                }
            },
            'MyTopicTokenSubscription141DD1BE2': {
                'Type': 'AWS::SNS::Subscription',
                'Properties': {
                    'Endpoint': {
                        'Ref': 'my-email-1'
                    },
                    'Protocol': 'email',
                    'TopicArn': {
                        'Ref': 'MyTopic86869434'
                    }
                }
            },
            'MyTopicTokenSubscription293BFE3F9': {
                'Type': 'AWS::SNS::Subscription',
                'Properties': {
                    'Endpoint': {
                        'Ref': 'my-url-1'
                    },
                    'Protocol': 'https',
                    'TopicArn': {
                        'Ref': 'MyTopic86869434'
                    }
                }
            }
        }
    });
});
test('email and url subscriptions with unresolved - four subscriptions', () => {
    const emailToken1 = core_1.Token.asString({ Ref: 'my-email-1' });
    const emailToken2 = core_1.Token.asString({ Ref: 'my-email-2' });
    const emailToken3 = core_1.Token.asString({ Ref: 'my-email-3' });
    const emailToken4 = core_1.Token.asString({ Ref: 'my-email-4' });
    topic.addSubscription(new subs.EmailSubscription(emailToken1));
    topic.addSubscription(new subs.EmailSubscription(emailToken2));
    topic.addSubscription(new subs.EmailSubscription(emailToken3));
    topic.addSubscription(new subs.EmailSubscription(emailToken4));
    expect(stack).toMatchTemplate({
        'Resources': {
            'MyTopic86869434': {
                'Type': 'AWS::SNS::Topic',
                'Properties': {
                    'DisplayName': 'displayName',
                    'TopicName': 'topicName'
                }
            },
            'MyTopicTokenSubscription141DD1BE2': {
                'Type': 'AWS::SNS::Subscription',
                'Properties': {
                    'Endpoint': {
                        'Ref': 'my-email-1'
                    },
                    'Protocol': 'email',
                    'TopicArn': {
                        'Ref': 'MyTopic86869434'
                    }
                }
            },
            'MyTopicTokenSubscription293BFE3F9': {
                'Type': 'AWS::SNS::Subscription',
                'Properties': {
                    'Endpoint': {
                        'Ref': 'my-email-2'
                    },
                    'Protocol': 'email',
                    'TopicArn': {
                        'Ref': 'MyTopic86869434'
                    }
                }
            },
            'MyTopicTokenSubscription335C2B4CA': {
                'Type': 'AWS::SNS::Subscription',
                'Properties': {
                    'Endpoint': {
                        'Ref': 'my-email-3'
                    },
                    'Protocol': 'email',
                    'TopicArn': {
                        'Ref': 'MyTopic86869434'
                    }
                }
            },
            'MyTopicTokenSubscription4DBE52A3F': {
                'Type': 'AWS::SNS::Subscription',
                'Properties': {
                    'Endpoint': {
                        'Ref': 'my-email-4'
                    },
                    'Protocol': 'email',
                    'TopicArn': {
                        'Ref': 'MyTopic86869434'
                    }
                }
            }
        }
    });
});
test('multiple subscriptions', () => {
    const queue = new sqs.Queue(stack, 'MyQueue');
    const func = new lambda.Function(stack, 'MyFunc', {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler',
        code: lambda.Code.fromInline('exports.handler = function(e, c, cb) { return cb() }')
    });
    topic.addSubscription(new subs.SqsSubscription(queue));
    topic.addSubscription(new subs.LambdaSubscription(func));
    expect(stack).toMatchTemplate({
        'Resources': {
            'MyTopic86869434': {
                'Type': 'AWS::SNS::Topic',
                'Properties': {
                    'DisplayName': 'displayName',
                    'TopicName': 'topicName'
                }
            },
            'MyQueueE6CA6235': {
                'Type': 'AWS::SQS::Queue'
            },
            'MyQueuePolicy6BBEDDAC': {
                'Type': 'AWS::SQS::QueuePolicy',
                'Properties': {
                    'PolicyDocument': {
                        'Statement': [
                            {
                                'Action': 'sqs:SendMessage',
                                'Condition': {
                                    'ArnEquals': {
                                        'aws:SourceArn': {
                                            'Ref': 'MyTopic86869434'
                                        }
                                    }
                                },
                                'Effect': 'Allow',
                                'Principal': {
                                    'Service': 'sns.amazonaws.com'
                                },
                                'Resource': {
                                    'Fn::GetAtt': [
                                        'MyQueueE6CA6235',
                                        'Arn'
                                    ]
                                }
                            }
                        ],
                        'Version': '2012-10-17'
                    },
                    'Queues': [
                        {
                            'Ref': 'MyQueueE6CA6235'
                        }
                    ]
                }
            },
            'MyQueueMyTopic9B00631B': {
                'Type': 'AWS::SNS::Subscription',
                'Properties': {
                    'Protocol': 'sqs',
                    'TopicArn': {
                        'Ref': 'MyTopic86869434'
                    },
                    'Endpoint': {
                        'Fn::GetAtt': [
                            'MyQueueE6CA6235',
                            'Arn'
                        ]
                    }
                }
            },
            'MyFuncServiceRole54065130': {
                'Type': 'AWS::IAM::Role',
                'Properties': {
                    'AssumeRolePolicyDocument': {
                        'Statement': [
                            {
                                'Action': 'sts:AssumeRole',
                                'Effect': 'Allow',
                                'Principal': {
                                    'Service': 'lambda.amazonaws.com'
                                }
                            }
                        ],
                        'Version': '2012-10-17'
                    },
                    'ManagedPolicyArns': [
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        'Ref': 'AWS::Partition'
                                    },
                                    ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole'
                                ]
                            ]
                        }
                    ]
                }
            },
            'MyFunc8A243A2C': {
                'Type': 'AWS::Lambda::Function',
                'Properties': {
                    'Code': {
                        'ZipFile': 'exports.handler = function(e, c, cb) { return cb() }'
                    },
                    'Handler': 'index.handler',
                    'Role': {
                        'Fn::GetAtt': [
                            'MyFuncServiceRole54065130',
                            'Arn'
                        ]
                    },
                    'Runtime': 'nodejs10.x'
                },
                'DependsOn': [
                    'MyFuncServiceRole54065130'
                ]
            },
            'MyFuncAllowInvokeMyTopicDD0A15B8': {
                'Type': 'AWS::Lambda::Permission',
                'Properties': {
                    'Action': 'lambda:InvokeFunction',
                    'FunctionName': {
                        'Fn::GetAtt': [
                            'MyFunc8A243A2C',
                            'Arn'
                        ]
                    },
                    'Principal': 'sns.amazonaws.com',
                    'SourceArn': {
                        'Ref': 'MyTopic86869434'
                    }
                }
            },
            'MyFuncMyTopic93B6FB00': {
                'Type': 'AWS::SNS::Subscription',
                'Properties': {
                    'Protocol': 'lambda',
                    'TopicArn': {
                        'Ref': 'MyTopic86869434'
                    },
                    'Endpoint': {
                        'Fn::GetAtt': [
                            'MyFunc8A243A2C',
                            'Arn'
                        ]
                    }
                }
            }
        }
    });
});
test('throws with mutliple subscriptions of the same subscriber', () => {
    const queue = new sqs.Queue(stack, 'MyQueue');
    topic.addSubscription(new subs.SqsSubscription(queue));
    expect(() => topic.addSubscription(new subs.SqsSubscription(queue)))
        .toThrowError(/A subscription with id \"MyTopic\" already exists under the scope MyQueue/);
});
test('with filter policy', () => {
    const fction = new lambda.Function(stack, 'MyFunc', {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler',
        code: lambda.Code.fromInline('exports.handler = function(e, c, cb) { return cb() }')
    });
    topic.addSubscription(new subs.LambdaSubscription(fction, {
        filterPolicy: {
            color: sns.SubscriptionFilter.stringFilter({
                whitelist: ['red'],
                matchPrefixes: ['bl', 'ye'],
            }),
            size: sns.SubscriptionFilter.stringFilter({
                blacklist: ['small', 'medium'],
            }),
            price: sns.SubscriptionFilter.numericFilter({
                between: { start: 100, stop: 200 }
            })
        }
    }));
    expect(stack).toHaveResource('AWS::SNS::Subscription', {
        'FilterPolicy': {
            'color': [
                'red',
                {
                    'prefix': 'bl'
                },
                {
                    'prefix': 'ye'
                }
            ],
            'size': [
                {
                    'anything-but': [
                        'small',
                        'medium'
                    ]
                }
            ],
            'price': [
                {
                    'numeric': [
                        '>=',
                        100,
                        '<=',
                        200
                    ]
                }
            ]
        }
    });
});
test('region property is present on an imported topic', () => {
    const imported = sns.Topic.fromTopicArn(stack, 'mytopic', 'arn:aws:sns:us-east-1:1234567890:mytopic');
    const queue = new sqs.Queue(stack, 'myqueue');
    imported.addSubscription(new subs.SqsSubscription(queue));
    expect(stack).toHaveResource('AWS::SNS::Subscription', {
        Region: 'us-east-1'
    });
});
test('region property on an imported topic as a parameter', () => {
    const topicArn = new core_1.CfnParameter(stack, 'topicArn');
    const imported = sns.Topic.fromTopicArn(stack, 'mytopic', topicArn.valueAsString);
    const queue = new sqs.Queue(stack, 'myqueue');
    imported.addSubscription(new subs.SqsSubscription(queue));
    expect(stack).toHaveResource('AWS::SNS::Subscription', {
        Region: {
            'Fn::Select': [3, { 'Fn::Split': [':', { 'Ref': 'topicArn' }] }]
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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