from __future__ import annotations

import pathlib
import sys
from typing import TYPE_CHECKING

from installer.exceptions import InvalidWheelSource
from installer.sources import WheelFile as _WheelFile
from pip._vendor.pkg_resources import EggInfoDistribution

from pdm import termui
from pdm.models import pip_shims
from pdm.models.requirements import parse_requirement
from pdm.utils import cached_property

if TYPE_CHECKING:
    from pip._vendor.pkg_resources import Distribution

    from pdm.models.candidates import Candidate
    from pdm.models.environment import Environment


def is_dist_editable(dist: Distribution) -> bool:
    return isinstance(dist, EggInfoDistribution) or getattr(dist, "editable", False)


def format_dist(dist: Distribution) -> str:
    formatter = "{version}{path}"
    path = ""
    if is_dist_editable(dist):
        path = f" (-e {dist.location})"
    return formatter.format(version=termui.yellow(dist.version), path=path)


class WheelFile(_WheelFile):
    @cached_property
    def dist_info_dir(self) -> str:
        namelist = self._zipfile.namelist()
        try:
            return next(
                name.split("/")[0]
                for name in namelist
                if name.split("/")[0].endswith(".dist-info")
            )
        except StopIteration:  # pragma: no cover
            canonical_name = super().dist_info_dir
            raise InvalidWheelSource(
                f"The wheel doesn't contain metadata {canonical_name!r}"
            )


class Installer:  # pragma: no cover
    """The installer that performs the installation and uninstallation actions."""

    def __init__(self, environment: Environment, auto_confirm: bool = True) -> None:
        self.environment = environment
        self.auto_confirm = auto_confirm
        # XXX: Patch pip to make it work under multi-thread mode
        pip_shims.pip_logging._log_state.indentation = 0

    def install(self, candidate: Candidate) -> None:
        if candidate.req.editable:
            self.install_editable(candidate.ireq)
        else:
            built = candidate.build()
            self.install_wheel(built)

    def install_wheel(self, wheel: str) -> None:
        from installer import __version__, install
        from installer.destinations import SchemeDictionaryDestination

        destination = SchemeDictionaryDestination(
            self.environment.get_paths(),
            interpreter=self.environment.interpreter.executable,
            script_kind=self._get_kind(),
        )

        with WheelFile.open(wheel) as source:
            install(
                source=source,
                destination=destination,
                # Additional metadata that is generated by the installation tool.
                additional_metadata={
                    "INSTALLER": f"installer {__version__}".encode(),
                },
            )

    def install_editable(self, ireq: pip_shims.InstallRequirement) -> None:
        from pdm.builders import EditableBuilder

        builder = EditableBuilder(ireq.unpacked_source_directory, self.environment)
        setup_path = builder.ensure_setup_py()
        paths = self.environment.get_paths()
        install_script = pathlib.Path(__file__).with_name("_editable_install.py")
        install_args = [
            self.environment.interpreter.executable,
            "-u",
            str(install_script),
            setup_path,
            paths["prefix"],
            paths["purelib"],
            paths["scripts"],
        ]
        builder.install(["setuptools"])
        builder.subprocess_runner(install_args, ireq.unpacked_source_directory)

    def uninstall(self, dist: Distribution) -> None:
        req = parse_requirement(dist.project_name)
        ireq = pip_shims.install_req_from_line(dist.project_name)
        ireq.req = req  # type: ignore

        pathset = ireq.uninstall(auto_confirm=self.auto_confirm)
        if pathset:
            pathset.commit()

    def _get_kind(self) -> str:
        if sys.platform != "win32":
            return "posix"
        is_32bit = self.environment.interpreter.is_32bit
        # TODO: support win arm64
        if is_32bit:
            return "win-ia32"
        else:
            return "win-amd64"
