"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_ec2_1 = require("monocdk/aws-ec2");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * VPCs have Flow Logs enabled - (Control IDs: 2.2, 10.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6)
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_ec2_1.CfnVPC) {
        const vpcLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_ec2_1.CfnFlowLog) {
                if (isMatchingCompliantFlowLog(child, vpcLogicalId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether a given Flow Log is compliant and associated with the given VPC
 * @param node the CfnFlowLog to check
 * returns whether the CfnFlowLog is compliant
 */
function isMatchingCompliantFlowLog(node, vpcLogicalId) {
    const resourceLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.resourceId);
    if (node.resourceType === 'VPC' && resourceLogicalId === vpcLogicalId) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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