"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_elasticloadbalancingv2_1 = require("monocdk/aws-elasticloadbalancingv2");
const aws_wafv2_1 = require("monocdk/aws-wafv2");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * ALBs are associated with AWS WAFv2 web ACLs - (Control ID: 6.6)
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_elasticloadbalancingv2_1.CfnLoadBalancer) {
        const type = nag_pack_1.resolveIfPrimitive(node, node.type);
        if (type === undefined || type === 'application') {
            const loadBalancerLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
            let found = false;
            for (const child of monocdk_1.Stack.of(node).node.findAll()) {
                if (child instanceof aws_wafv2_1.CfnWebACLAssociation) {
                    if (isMatchingWebACLAssociation(child, loadBalancerLogicalId)) {
                        found = true;
                        break;
                    }
                }
            }
            if (!found) {
                return false;
            }
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether a given Web ACL Association is associated with the given Load Balancer
 * @param node the CfnWebACLAssociation to check
 * @param loadBalancerLogicalId the Cfn Logical ID of the Load Balancer
 * returns whether the CfnWebACLAssociation is associates with the given Load Balancer
 */
function isMatchingWebACLAssociation(node, loadBalancerLogicalId) {
    const resourceLogicalId = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.resourceArn));
    if (new RegExp(`${loadBalancerLogicalId}(?![\\w])`, 'gm').test(resourceLogicalId)) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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